package ru.yandex.direct.core.entity.banner.type.internal;

import java.util.Map;

import javax.annotation.Nullable;

import ru.yandex.direct.core.entity.banner.model.BannerWithInternalInfo;
import ru.yandex.direct.core.entity.banner.model.TemplateVariable;
import ru.yandex.direct.core.entity.banner.type.href.BannerUrlCheckService;
import ru.yandex.direct.core.entity.internalads.model.InternalTemplateInfo;
import ru.yandex.direct.core.entity.internalads.model.ResourceInfo;
import ru.yandex.direct.core.entity.internalads.model.ResourceType;
import ru.yandex.direct.core.entity.internalads.service.validation.defects.InternalAdDefects;
import ru.yandex.direct.core.service.urlchecker.UrlCheckResult;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.result.Defect;

import static ru.yandex.direct.utils.FunctionalUtils.listToMap;

/**
 * Абстрактный валидатор, включайщий в себя общую логику проверки {@link TemplateVariable}.
 */
public abstract class AbstractTemplateVariablesValidator<T> implements Validator<T, Defect> {

    protected final BannerWithInternalInfo banner;
    protected final InternalTemplateInfo internalTemplateInfo;

    // каждому ресурсу шаблона должен соответствовать тип
    protected final Map<Long, ResourceType> resourceIdToType;

    protected final BannerUrlCheckService bannerUrlCheckService;

    AbstractTemplateVariablesValidator(BannerWithInternalInfo banner,
                                       InternalTemplateInfo internalTemplateInfo,
                                       BannerUrlCheckService bannerUrlCheckService) {
        this.banner = banner;
        this.internalTemplateInfo = internalTemplateInfo;
        this.bannerUrlCheckService = bannerUrlCheckService;

        this.resourceIdToType = getResourceTypes(internalTemplateInfo);
    }

    private Map<Long, ResourceType> getResourceTypes(InternalTemplateInfo templateInfo) {
        return listToMap(templateInfo.getResources(), ResourceInfo::getId, ResourceInfo::getType);
    }

    /**
     * Проверяет, что ссылка по указанному урлу доступна
     *
     * @param templateVariable значение ресурса баннера, с типом ссылки
     */
    @Nullable
    protected Defect variableValueUrlIsReachable(TemplateVariable templateVariable) {
        var resourceType = resourceIdToType.get(templateVariable.getTemplateResourceId());
        if (resourceType != ResourceType.URL) {
            return null;
        }
        var url = templateVariable.getInternalValue();
        if (url == null) {
            return null;
        }
        var checkResult = bannerUrlCheckService.isUrlReachable(url);
        if (!checkResult.getResult()) {
            return urlCheckResultErrorToDefect(checkResult);
        }
        return null;
    }

    private static Defect urlCheckResultErrorToDefect(UrlCheckResult result) {
        return InternalAdDefects.urlUnreachable(result.getError().name());
    }
}
