package ru.yandex.direct.core.entity.banner.type.internal;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.type.TypeFactory;
import org.jooq.DSLContext;
import org.jooq.util.mysql.MySQLDSL;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.banner.model.BannerWithInternalInfo;
import ru.yandex.direct.core.entity.banner.model.InternalModerationInfo;
import ru.yandex.direct.core.entity.banner.model.TemplateVariable;
import ru.yandex.direct.core.entity.banner.repository.type.AbstractFlatRelatedEntityUpsertRepositoryTypeSupport;
import ru.yandex.direct.dbschema.ppc.tables.records.BannersInternalRecord;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapper;
import ru.yandex.direct.jooqmapper.JooqMapperBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.utils.JsonUtils;

import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.booleanProperty;
import static ru.yandex.direct.dbschema.ppc.tables.BannersInternal.BANNERS_INTERNAL;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.jooqmapper.write.WriterBuilders.fromProperty;

@Component
@ParametersAreNonnullByDefault
public class BannerWithInternalInfoRepositoryTypeSupport
        extends AbstractFlatRelatedEntityUpsertRepositoryTypeSupport
        <BannerWithInternalInfo, BannersInternalRecord> {

    private static final JavaType TEMPLATE_VARIABLES_TYPE = TypeFactory.defaultInstance()
            .constructCollectionType(List.class, TemplateVariable.class);

    private static final Set<ModelProperty<? super BannerWithInternalInfo, ?>> MUTABLE_PROPERTIES = Set.of(
            BannerWithInternalInfo.DESCRIPTION,
            BannerWithInternalInfo.TEMPLATE_VARIABLES,
            BannerWithInternalInfo.MODERATION_INFO,
            BannerWithInternalInfo.IS_STOPPED_BY_URL_MONITORING
    );

    private static final Logger logger =
            LoggerFactory.getLogger(BannerWithInternalInfoRepositoryTypeSupport.class);

    @Autowired
    public BannerWithInternalInfoRepositoryTypeSupport(DslContextProvider dslContextProvider) {
        super(dslContextProvider, BANNERS_INTERNAL.BID, createMapper());
    }

    @Override
    protected boolean isAddEntity(BannerWithInternalInfo banner) {
        return banner.getTemplateId() != null;
    }

    @Override
    protected boolean isDeleteEntity(AppliedChanges<BannerWithInternalInfo> appliedChanges) {
        if (appliedChanges.deleted(BannerWithInternalInfo.TEMPLATE_ID)) {
            logger.error("Try to delete internal data for bid: {}. Delete entity is not supported for " +
                    "BannerWithInternalInfo", appliedChanges.getModel().getId());
        }
        return false;
    }

    @Override
    protected boolean isUpsertEntity(AppliedChanges<BannerWithInternalInfo> appliedChanges) {
        return !isDeleteEntity(appliedChanges) && MUTABLE_PROPERTIES.stream().anyMatch(appliedChanges::changed);
    }

    @Override
    protected void upsertEntity(
            DSLContext context,
            Collection<BannerWithInternalInfo> banners) {
        var insertHelper = new InsertHelper<>(context, BANNERS_INTERNAL);
        insertHelper.addAll(getJooqMapper(), banners);
        insertHelper.onDuplicateKeyUpdate()
                .set(BANNERS_INTERNAL.DESCRIPTION, MySQLDSL.values(BANNERS_INTERNAL.DESCRIPTION))
                .set(BANNERS_INTERNAL.TEMPLATE_VARIABLES, MySQLDSL.values(BANNERS_INTERNAL.TEMPLATE_VARIABLES))
                .set(BANNERS_INTERNAL.MODERATION_INFO, MySQLDSL.values(BANNERS_INTERNAL.MODERATION_INFO))
                .set(BANNERS_INTERNAL.IS_STOPPED_BY_URL_MONITORING,
                        MySQLDSL.values(BANNERS_INTERNAL.IS_STOPPED_BY_URL_MONITORING));
        insertHelper.executeIfRecordsAdded();
    }

    @Override
    public Class<BannerWithInternalInfo> getTypeClass() {
        return BannerWithInternalInfo.class;
    }

    private static JooqMapper<BannerWithInternalInfo> createMapper() {
        return JooqMapperBuilder.<BannerWithInternalInfo>builder()
                .map(property(BannerWithInternalInfo.DESCRIPTION, BANNERS_INTERNAL.DESCRIPTION))
                .map(property(BannerWithInternalInfo.TEMPLATE_ID, BANNERS_INTERNAL.TEMPLATE_ID))
                .map(convertibleProperty(BannerWithInternalInfo.TEMPLATE_VARIABLES,
                        BANNERS_INTERNAL.TEMPLATE_VARIABLES,
                        BannerWithInternalInfoRepositoryTypeSupport::templateVariablesFromDb,
                        BannerWithInternalInfoRepositoryTypeSupport::templateVariablesToDb))
                .map(convertibleProperty(BannerWithInternalInfo.MODERATION_INFO,
                        BANNERS_INTERNAL.MODERATION_INFO,
                        BannerWithInternalInfoRepositoryTypeSupport::moderationInfoFromDb,
                        BannerWithInternalInfoRepositoryTypeSupport::moderationInfoToDb))
                .map(booleanProperty(BannerWithInternalInfo.IS_STOPPED_BY_URL_MONITORING,
                        BANNERS_INTERNAL.IS_STOPPED_BY_URL_MONITORING))
                .writeField(BANNERS_INTERNAL.BID, fromProperty(BannerWithInternalInfo.ID))
                .build();
    }

    public static List<TemplateVariable> templateVariablesFromDb(@Nullable String templateVariables) {
        if (templateVariables == null) {
            return Collections.emptyList();
        }
        return JsonUtils.fromJson(templateVariables, TEMPLATE_VARIABLES_TYPE);
    }

    private static String templateVariablesToDb(
            List<TemplateVariable> templateVariables) {
        return JsonUtils.toJson(templateVariables);
    }

    @Nullable
    public static InternalModerationInfo moderationInfoFromDb(@Nullable String moderationInfo) {
        return moderationInfo == null ? null : JsonUtils.fromJson(moderationInfo, InternalModerationInfo.class);
    }

    @Nullable
    public static String moderationInfoToDb(@Nullable InternalModerationInfo moderationInfo) {
        return JsonUtils.toJsonNullable(moderationInfo);
    }
}
