package ru.yandex.direct.core.entity.banner.type.internal;

import java.util.List;
import java.util.Objects;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.banner.container.BannersModerationContainer;
import ru.yandex.direct.core.entity.banner.container.BannersUpdateOperationContainer;
import ru.yandex.direct.core.entity.banner.model.BannerWithInternalAdModerationInfo;
import ru.yandex.direct.core.entity.banner.model.BannerWithInternalInfo;
import ru.yandex.direct.core.entity.banner.model.InternalModerationInfo;
import ru.yandex.direct.core.entity.banner.service.type.update.AbstractBannerUpdateOperationTypeSupport;
import ru.yandex.direct.core.entity.internalads.model.InternalTemplateInfo;
import ru.yandex.direct.core.entity.internalads.service.InternalAdUrlMacrosService;
import ru.yandex.direct.core.entity.internalads.service.TemplateInfoService;
import ru.yandex.direct.core.entity.moderation.service.sending.internalad.InternalBannerModerationHelper;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelProperty;

import static java.util.Collections.emptyMap;
import static ru.yandex.direct.core.entity.internalads.Constants.isModeratedTemplate;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapAndFilterToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
public class BannerWithInternalInfoUpdateOperationTypeSupport
        extends AbstractBannerUpdateOperationTypeSupport<BannerWithInternalInfo> {

    private final TemplateInfoService templateInfoService;

    @Autowired
    public BannerWithInternalInfoUpdateOperationTypeSupport(TemplateInfoService templateInfoService) {
        this.templateInfoService = templateInfoService;
    }

    @Override
    public Class<BannerWithInternalInfo> getTypeClass() {
        return BannerWithInternalInfo.class;
    }

    @Override
    public void beforeExecution(BannersUpdateOperationContainer updateContainer,
                                List<AppliedChanges<BannerWithInternalInfo>> changesList) {

        var templateIds = mapAndFilterToSet(mapList(changesList, AppliedChanges::getModel),
                BannerWithInternalInfo::getTemplateId, Objects::nonNull);

        var internalTemplateInfoList = templateInfoService.getByTemplateIds(templateIds);
        var templateIdToTemplateInfo = listToMap(internalTemplateInfoList, InternalTemplateInfo::getTemplateId);

        StreamEx.of(changesList)
                .forEach(changes ->
                        changes.modify(BannerWithInternalInfo.TEMPLATE_VARIABLES,
                                InternalAdUrlMacrosService.enrichDefaultUrlParametrsForBanner(changes.getModel(),
                                        templateIdToTemplateInfo)));

        dropStoppedByUrlMonitoringForResumedBanners(changesList);
    }

    /**
     * Для включенных баннеров сбрасываем isStoppedByUrlMonitoring
     */
    private void dropStoppedByUrlMonitoringForResumedBanners(List<AppliedChanges<BannerWithInternalInfo>> appliedChanges) {
        appliedChanges
                .stream()
                .filter(c -> c.changed(BannerWithInternalInfo.STATUS_SHOW))
                .filter(c -> c.getNewValue(BannerWithInternalInfo.STATUS_SHOW))
                .forEach(bannerChanges -> {
                    bannerChanges.modify(BannerWithInternalInfo.IS_STOPPED_BY_URL_MONITORING, false);
                });
    }

    @Override
    public boolean needModeration(BannersModerationContainer container,
                                  AppliedChanges<BannerWithInternalInfo> appliedChanges) {
        if (!isModeratedTemplate(appliedChanges.getModel().getTemplateId())) {
            return false;
        }

        return hasChangedFieldsWhichSendingToModeration(appliedChanges)
                // это когда не изменились поля, которые отправляем на модерацию, но выбрали сохранить в черновик
                // или отправить на модерацию
                || hasChangedSendToModerationValue(appliedChanges);
    }

    @Override
    public boolean needBsResync(AppliedChanges<BannerWithInternalInfo> appliedChanges) {
        return appliedChanges.changed(BannerWithInternalInfo.TEMPLATE_VARIABLES)
                || appliedChanges.changed(BannerWithInternalInfo.STATUS_SHOW);
    }

    @Override
    public boolean needLastChangeReset(AppliedChanges<BannerWithInternalInfo> appliedChanges) {
        return appliedChanges.changed(BannerWithInternalInfo.DESCRIPTION)
                || appliedChanges.changed(BannerWithInternalInfo.TEMPLATE_VARIABLES)
                || appliedChanges.changed(BannerWithInternalInfo.MODERATION_INFO);
    }


    /**
     * Метод проверяет изменилось ли хоть одно поле, которое мы отправляем на модерацию.
     * Для этого создаем модель запроса для отправки на Модерацию по новым параметрам баннера и по старым и сравниваем
     * запросы между собой. Без создания запроса не обойтись, т.к. при отправке есть логика по мапингу ресурсов на
     * отправляемые поля...
     * Не все поля баннера мы отправляем на модерацию, например moderationInfo->statusShowAfterModeration или некоторые
     * ресурсы шаблонов (старые или экспериментально новые)
     */
    private static boolean hasChangedFieldsWhichSendingToModeration(AppliedChanges<BannerWithInternalInfo> appliedChanges) {
        var newBannerWithInternalAdModerationInfo = toBannerWithInternalAdModerationInfo(appliedChanges, true);
        var oldBannerWithInternalAdModerationInfo = toBannerWithInternalAdModerationInfo(appliedChanges, false);

        var newRequestData = InternalBannerModerationHelper.INSTANCE
                .createInternalBannerRequestData(newBannerWithInternalAdModerationInfo, false, emptyMap(), "");
        var oldRequestData = InternalBannerModerationHelper.INSTANCE
                .createInternalBannerRequestData(oldBannerWithInternalAdModerationInfo, false, emptyMap(), "");
        return !newRequestData.equals(oldRequestData);
    }

    private static BannerWithInternalAdModerationInfo toBannerWithInternalAdModerationInfo(
            AppliedChanges<BannerWithInternalInfo> appliedChanges,
            boolean getNewValue
    ) {
        return new BannerWithInternalAdModerationInfo()
                .withDescription(getValue(appliedChanges, BannerWithInternalInfo.DESCRIPTION, getNewValue))
                .withTemplateId(getValue(appliedChanges, BannerWithInternalInfo.TEMPLATE_ID, getNewValue))
                .withTemplateVariables(getValue(appliedChanges, BannerWithInternalInfo.TEMPLATE_VARIABLES, getNewValue))
                .withModerationInfo(getValue(appliedChanges, BannerWithInternalInfo.MODERATION_INFO, getNewValue));
    }

    private static <T> T getValue(AppliedChanges<BannerWithInternalInfo> appliedChanges,
                                  ModelProperty<? super BannerWithInternalInfo, T> modelProperty,
                                  boolean getNewValue) {
        return getNewValue ? appliedChanges.getNewValue(modelProperty) : appliedChanges.getOldValue(modelProperty);
    }

    private static boolean hasChangedSendToModerationValue(AppliedChanges<BannerWithInternalInfo> appliedChanges) {
        if (!appliedChanges.changed(BannerWithInternalInfo.MODERATION_INFO)) {
            return false;
        }

        InternalModerationInfo newModerationValue = appliedChanges.getNewValue(BannerWithInternalInfo.MODERATION_INFO);
        InternalModerationInfo oldModerationValue = appliedChanges.getOldValue(BannerWithInternalInfo.MODERATION_INFO);

        //noinspection ConstantConditions
        return !newModerationValue.getSendToModeration().equals(oldModerationValue.getSendToModeration());
    }

}
