package ru.yandex.direct.core.entity.banner.type.internal;

import java.util.Collection;
import java.util.Collections;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupForBannerOperation;
import ru.yandex.direct.core.entity.adgroup.model.InternalAdGroup;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.banner.container.BannersOperationContainer;
import ru.yandex.direct.core.entity.banner.model.BannerWithInternalInfo;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.model.InternalModerationInfo;
import ru.yandex.direct.core.entity.banner.model.TemplateVariable;
import ru.yandex.direct.core.entity.banner.type.href.BannerUrlCheckService;
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign;
import ru.yandex.direct.core.entity.campaign.model.InternalCampaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository;
import ru.yandex.direct.core.entity.image.model.BannerImageFormat;
import ru.yandex.direct.core.entity.image.repository.BannerImageFormatRepository;
import ru.yandex.direct.core.entity.internalads.model.InternalAdPlaceInfo;
import ru.yandex.direct.core.entity.internalads.model.InternalAdsProduct;
import ru.yandex.direct.core.entity.internalads.model.InternalTemplateInfo;
import ru.yandex.direct.core.entity.internalads.model.TemplatePlace;
import ru.yandex.direct.core.entity.internalads.service.InternalAdsProductService;
import ru.yandex.direct.core.entity.internalads.service.PlaceService;
import ru.yandex.direct.core.entity.internalads.service.TemplateInfoService;
import ru.yandex.direct.core.entity.internalads.service.TemplatePlaceService;
import ru.yandex.direct.core.entity.internalads.service.TemplateResourceService;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.regions.GeoTreeFactory;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static java.util.Collections.emptySet;
import static java.util.function.Function.identity;
import static ru.yandex.direct.core.entity.banner.model.BannerWithInternalInfo.DESCRIPTION;
import static ru.yandex.direct.core.entity.banner.model.BannerWithInternalInfo.MODERATION_INFO;
import static ru.yandex.direct.core.entity.banner.model.BannerWithInternalInfo.TEMPLATE_ID;
import static ru.yandex.direct.core.entity.banner.model.BannerWithInternalInfo.TEMPLATE_VARIABLES;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapAndFilterToSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.eachNotNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.isNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.defect.CommonDefects.invalidValue;

@Component
public class BannerWithInternalInfoValidatorProvider {

    private final TemplateInfoService templateInfoService;
    private final TemplatePlaceService templatePlaceService;
    private final TemplateResourceService templateResourceService;
    private final InternalAdsProductService internalAdsProductService;
    private final CampaignTypedRepository campaignTypedRepository;
    private final AdGroupRepository adGroupRepository;
    private final BannerImageFormatRepository bannerImageFormatRepository;
    private final GeoTreeFactory geoTreeFactory;
    private final BannerUrlCheckService bannerUrlCheckService;
    private final PlaceService placeService;

    @Autowired
    public BannerWithInternalInfoValidatorProvider(
            TemplateInfoService templateInfoService,
            TemplatePlaceService templatePlaceService,
            TemplateResourceService templateResourceService,
            GeoTreeFactory geoTreeFactory,
            InternalAdsProductService internalAdsProductService,
            CampaignTypedRepository campaignTypedRepository,
            AdGroupRepository adGroupRepository,
            BannerImageFormatRepository bannerImageFormatRepository,
            BannerUrlCheckService bannerUrlCheckService,
            PlaceService placeService) {
        this.templateInfoService = templateInfoService;
        this.templatePlaceService = templatePlaceService;
        this.templateResourceService = templateResourceService;
        this.internalAdsProductService = internalAdsProductService;
        this.campaignTypedRepository = campaignTypedRepository;
        this.adGroupRepository = adGroupRepository;
        this.bannerImageFormatRepository = bannerImageFormatRepository;
        this.geoTreeFactory = geoTreeFactory;
        this.bannerUrlCheckService = bannerUrlCheckService;
        this.placeService = placeService;
    }

    Validator<List<BannerWithInternalInfo>, Defect> bannerWithInternalExtraInfoListValidator(
            BannersOperationContainer container) {
        return banners -> {
            var internalAdsProduct = internalAdsProductService.getProduct(container.getClientId());
            var bannerToCampaign = getBannerToCampaign(container, banners);
            var bannerToAdGroup = getBannerToAdGroup(container, banners);

            var bannerToAllowedTemplates = collectAllowedTemplates(banners, bannerToCampaign);

            var templateIdToTemplateInfo = getTemplatesInfo(banners);
            var imageHashToImageInfo = collectImageInfoMap(container, banners);

            var templateIdToPlaceInfo = getTemplateIdToPlaceInfo(banners, bannerToCampaign);

            ListValidationBuilder<BannerWithInternalInfo, Defect> lvb = ListValidationBuilder.of(banners);
            lvb.checkEachBy(bannerWithInternalExtraInfoValidator(bannerToAllowedTemplates, templateIdToTemplateInfo,
                    imageHashToImageInfo, internalAdsProduct, bannerToCampaign, bannerToAdGroup,
                    templateIdToPlaceInfo, container));
            return lvb.getResult();
        };
    }

    public Validator<List<ModelChanges<BannerWithSystemFields>>, Defect> resumeBannerWithInternalExtraInfoListValidator(
            Map<Long, BannerWithInternalInfo> banners) {
        return modelChangesList -> {
            var templateIdToTemplateInfo = getTemplatesInfo(banners.values());

            ListValidationBuilder<ModelChanges<BannerWithSystemFields>, Defect> lvb =
                    ListValidationBuilder.of(modelChangesList);

            lvb.checkEachBy(resumeBannerWithInternalExtraInfoValidator(banners, templateIdToTemplateInfo),
                    When.valueIs(isInternalBanner(banners)));

            return lvb.getResult();
        };
    }

    private Map<Long, InternalTemplateInfo> getTemplatesInfo(Collection<BannerWithInternalInfo> banners) {
        var templateIds = mapAndFilterToSet(
                banners, BannerWithInternalInfo::getTemplateId, Objects::nonNull);

        var internalTemplateInfoList = templateInfoService.getByTemplateIds(templateIds);

        return listToMap(internalTemplateInfoList, InternalTemplateInfo::getTemplateId);
    }

    private static Predicate<ModelChanges<BannerWithSystemFields>> isInternalBanner(
            Map<Long, BannerWithInternalInfo> banners) {
        return mc -> banners.containsKey(mc.getId());
    }

    /**
     * Получить модель кампании внутренней рекламы по моделям баннеров
     * В возращаемой мапе не будет кампании - для баннера у которого в базе не нашли группу
     */
    private Map<BannerWithInternalInfo, InternalCampaign> getBannerToCampaign(BannersOperationContainer container,
                                                                              List<BannerWithInternalInfo> banners) {
        var campaignIds = listToSet(container.getCampaigns(), CommonCampaign::getId);
        var typedCampaignsMap = campaignTypedRepository.getTypedCampaignsMap(container.getShard(), campaignIds);

        return StreamEx.of(banners)
                .mapToEntry(container::getCampaign)
                .nonNullValues()
                .mapValues(CommonCampaign::getId)
                .mapValues(typedCampaignsMap::get)
                .selectValues(InternalCampaign.class)
                .toCustomMap(IdentityHashMap::new);
    }

    private Map<BannerWithInternalInfo, InternalAdGroup> getBannerToAdGroup(BannersOperationContainer container,
                                                                            List<BannerWithInternalInfo> banners) {
        var adGroupIds = mapAndFilterToSet(container.getUniqueAdGroups(),
                AdGroupForBannerOperation::getId, Objects::nonNull);
        var adGroups = adGroupRepository.getAdGroups(container.getShard(), adGroupIds);
        var adGroupById = listToMap(adGroups, AdGroup::getId);

        return StreamEx.of(banners)
                .mapToEntry(container::getAdGroup)
                .nonNullValues()
                .mapValues(adGroupForBannerOperation -> adGroupForBannerOperation.getId() == null
                        // когда создание новой группы с новым баннером, будет InternalAdGroup но без id
                        ? adGroupForBannerOperation
                        // иначе надо брать группу из базы, т.к. в AdGroupForBannerOperation нет всех необходимых полей
                        : adGroupById.get(adGroupForBannerOperation.getId()))
                .selectValues(InternalAdGroup.class)
                .toCustomMap(IdentityHashMap::new);
    }

    private Validator<BannerWithInternalInfo, Defect> bannerWithInternalExtraInfoValidator(
            IdentityHashMap<BannerWithInternalInfo, Set<Long>> bannerToAllowedTemplates,
            Map<Long, InternalTemplateInfo> templateIdToTemplateInfo,
            Map<String, BannerImageFormat> imageHashToImageInfo,
            InternalAdsProduct internalAdsProduct,
            Map<BannerWithInternalInfo, InternalCampaign> bannerToCampaign,
            Map<BannerWithInternalInfo, InternalAdGroup> bannerToAdGroup,
            Map<Long, InternalAdPlaceInfo> templateIdToPlaceInfo,
            BannersOperationContainer container) {
        return banner -> {
            ModelItemValidationBuilder<BannerWithInternalInfo> ivb = ModelItemValidationBuilder.of(banner);
            var allowedTemplates = bannerToAllowedTemplates.getOrDefault(banner, Collections.emptySet());
            ivb.item(DESCRIPTION)
                    .checkBy(descriptionValidator());

            InternalAdPlaceInfo placeInfo = templateIdToPlaceInfo.get(banner.getTemplateId());
            boolean isModeratedPlace = placeInfo != null && placeInfo.isModerated();
            ivb.item(MODERATION_INFO)
                    .check(notNull(), When.isTrue(isModeratedPlace))
                    .check(isNull(), When.isFalse(isModeratedPlace))
                    .checkBy(moderationInfoValidator(), When.notNullAnd(When.isTrue(isModeratedPlace)));

            var vbTemplateId = ivb.item(TEMPLATE_ID)
                    .checkBy(templateIdValidator(allowedTemplates, isModeratedPlace));

            ivb.item(TEMPLATE_VARIABLES)
                    .check(notNull())
                    .check(eachNotNull())
                    .checkBy(templateVariablesValidator(banner, templateIdToTemplateInfo,
                                    imageHashToImageInfo, internalAdsProduct, container,
                                    bannerToCampaign.get(banner), bannerToAdGroup.get(banner)),
                            When.isValidBoth(vbTemplateId));

            boolean enableMaxStopsCountValidator = container.getClientEnabledFeatures()
                    .contains(FeatureName.MAX_STOPS_COUNT_VALIDATOR_FOR_INTERNAL_ADS_ENABLED.getName());
            ivb.checkBy(maxStopsCountValidator(bannerToCampaign.get(banner), bannerToAdGroup.get(banner)),
                    When.isTrue(enableMaxStopsCountValidator));

            return ivb.getResult();
        };
    }

    private Validator<String, Defect> descriptionValidator() {
        return DescriptionValidator.descriptionValidator();
    }

    private Validator<Long, Defect> templateIdValidator(Set<Long> allowedTemplates, boolean isModeratedPlace) {
        return TemplateIdValidator.templateIdValidator(allowedTemplates, isModeratedPlace);
    }

    private Validator<List<TemplateVariable>, Defect> templateVariablesValidator(
            BannerWithInternalInfo banner,
            Map<Long, InternalTemplateInfo> templateIdToTemplateInfo,
            Map<String, BannerImageFormat> imageHashToImageInfo,
            InternalAdsProduct internalAdsProduct,
            BannersOperationContainer container,
            @Nullable InternalCampaign campaign,
            @Nullable InternalAdGroup adGroup) {
        if (!templateIdToTemplateInfo.containsKey(banner.getTemplateId())) {
            return any -> ValidationResult.failed(any, invalidValue());
        }
        return TemplateVariablesValidator.templateVariablesValidator(banner,
                templateIdToTemplateInfo.get(banner.getTemplateId()), internalAdsProduct, getGeoTree(),
                bannerUrlCheckService, container, campaign, adGroup, imageHashToImageInfo);
    }

    private Validator<InternalModerationInfo, Defect> moderationInfoValidator() {
        return new InternalModerationInfoValidator();
    }

    private Validator<BannerWithInternalInfo, Defect> maxStopsCountValidator(@Nullable InternalCampaign campaign,
                                                                             @Nullable InternalAdGroup adGroup) {
        return new InternalMaxStopsCountValidator(campaign, adGroup);
    }

    private Validator<ModelChanges<BannerWithSystemFields>, Defect> resumeBannerWithInternalExtraInfoValidator(
            Map<Long, BannerWithInternalInfo> banners,
            Map<Long, InternalTemplateInfo> templateIdToTemplateInfo) {
        return modelChanges -> {
            var banner = banners.get(modelChanges.getId());
            var internalTemplateInfo = templateIdToTemplateInfo.get(banner.getTemplateId());

            ResumeTemplateVariablesValidator resumeTemplateVariablesValidator = new ResumeTemplateVariablesValidator(
                    banner, internalTemplateInfo, bannerUrlCheckService);

            return resumeTemplateVariablesValidator.apply(modelChanges);
        };
    }

    /**
     * Возвращает доступные шаблоны для баннера
     * В возвращаемой мапе не будет записей для баннера, у которого в базе не нашли группу
     */
    private IdentityHashMap<BannerWithInternalInfo, Set<Long>> collectAllowedTemplates(
            List<BannerWithInternalInfo> banners,
            Map<BannerWithInternalInfo, InternalCampaign> bannerToCampaign) {
        var placeIds = listToSet(bannerToCampaign.values(), InternalCampaign::getPlaceId);
        var allowedPlaceIdToTemplateIds = StreamEx.of(templatePlaceService.getByPlaceIds(placeIds))
                .mapToEntry(TemplatePlace::getPlaceId, TemplatePlace::getTemplateId)
                .grouping(Collectors.toSet());

        return StreamEx.of(banners)
                .mapToEntry(bannerToCampaign::get)
                .nonNullValues()
                .mapValues(InternalCampaign::getPlaceId)
                .mapValues(placeId -> allowedPlaceIdToTemplateIds.getOrDefault(placeId, emptySet()))
                .toCustomMap(IdentityHashMap::new);
    }


    /**
     * Собрать отображение хэша картинки в информацию о картинке из БД
     * <p>
     * Если в баннере есть картиночная переменная, хэшу которой не соотвествует никакая картинка,
     * то в результирующей мапе не будет этого хэша.
     *
     * @return отображение хэша картинки в информацию {@link BannerImageFormat}
     */
    private Map<String, BannerImageFormat> collectImageInfoMap(
            BannersOperationContainer container,
            List<BannerWithInternalInfo> banners) {
        var allTemplateVariables = StreamEx.of(banners)
                .map(BannerWithInternalInfo::getTemplateVariables)
                .nonNull()
                .flatCollection(identity())
                .toList();

        var allMentionedTemplateResourceIds = StreamEx.of(allTemplateVariables)
                .map(TemplateVariable::getTemplateResourceId)
                .nonNull()
                .toSet();

        var imageVariableIds = templateResourceService
                .getImageResourceIdsByResourceIds(allMentionedTemplateResourceIds);

        var imageHashes = StreamEx.of(allTemplateVariables)
                .filter(v -> imageVariableIds.contains(v.getTemplateResourceId()))
                .map(TemplateVariable::getInternalValue)
                .nonNull()
                .toSet();

        return bannerImageFormatRepository.getBannerImageFormats(container.getShard(), container.getClientId(),
                imageHashes);
    }

    /**
     * Возвращает валидные плейсы для шаблонов баннеров
     */
    private Map<Long, InternalAdPlaceInfo> getTemplateIdToPlaceInfo(
            List<BannerWithInternalInfo> banners,
            Map<BannerWithInternalInfo, InternalCampaign> bannerToCampaign) {
        var templateIdToPlaceId = StreamEx.of(banners)
                .mapToEntry(BannerWithInternalInfo::getTemplateId, bannerToCampaign::get)
                .nonNullValues()
                .distinctKeys()
                .mapValues(InternalCampaign::getPlaceId)
                .toMap();

        var placeIdToPlaceInfo = listToMap(
                placeService.getPlaceInfoForValidPlaceByIds(templateIdToPlaceId.values()), InternalAdPlaceInfo::getId);

        return EntryStream.of(templateIdToPlaceId)
                .mapValues(placeIdToPlaceInfo::get)
                .nonNullValues()
                .toMap();
    }

    private GeoTree getGeoTree() {
        // Для внутренней рекламы используем российское гео дерево
        return geoTreeFactory.getRussianGeoTree();
    }
}
