package ru.yandex.direct.core.entity.banner.type.internal;

import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.regex.Pattern;

import javax.annotation.Nullable;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.banner.model.TemplateVariable;
import ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects;
import ru.yandex.direct.core.entity.internalads.Constants;
import ru.yandex.direct.utils.StringUtils;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Comparator.comparing;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.invalidSpecSymbols;
import static ru.yandex.direct.core.entity.internalads.Constants.ALLOWED_SPEC_SYMBOLS_FOR_TEXT;
import static ru.yandex.direct.core.entity.internalads.Constants.ALLOWED_TAGS_FOR_TEXT;
import static ru.yandex.direct.core.entity.internalads.Constants.ILLEGAL_SPEC_SYMBOLS_WITHOUT_BRACKETS;

/**
 * Валидатор для определения правильности спецсимволов для переменных текстовых ресурсов.
 * <p>
 * Алгоритм работы:
 * <p>
 * Текст проверяется по паттернам {@link SpecSymbolsTextVariableValidator#SPEC_SYMBOLS_PATTERN} и
 * {@link SpecSymbolsTextVariableValidator#TAGS_PATTERN} для определения всех встречающихся спецсимволов с
 * последующей проверкой, что они могут находиться в тексте
 * (т.е. содержатся в {@link Constants#ALLOWED_SPEC_SYMBOLS_FOR_TEXT} или {@link Constants#ALLOWED_TAGS_FOR_TEXT})
 * <p>
 * Далее текст по каждому спецсимволу из
 * {@link SpecSymbolsTextVariableValidator#SORTED_ALLOWED_SPEC_SYMBOLS}
 * (отсортированный список по длине (от большей длины к меньшей) спецсимволов из
 * {@link Constants#ALLOWED_SPEC_SYMBOLS_FOR_TEXT} и {@link Constants#ALLOWED_TAGS_FOR_TEXT})
 * проверяется на корректность последовательно: находятся все вхождения данного спецсимвола в тексте, для каждого из
 * которых проверяется, что верно указаны скобки
 * (см. метод {@link SpecSymbolsTextVariableValidator#isSpecSymbolCorrect}),
 * в последствии удаляются все вхождения спецсимвола из текста для проверки на последующие спецсимволы.
 * Удаление всех вхождений нужно для того, что бы не возникали конфликты между спецсимволами при проверке
 */
public class SpecSymbolsTextVariableValidator implements Validator<TemplateVariable, Defect> {

    private static final Pattern SPEC_SYMBOLS_PATTERN = Pattern.compile("&+[^;\\s]*+;+");
    private static final Pattern TAGS_PATTERN = Pattern.compile("<+[^>\\s]*+>+");
    private static final String SPEC_SYMBOL_BRACKETS = "&;";
    private static final String TAG_BRACKETS = "<>";
    private static final List<String> SORTED_ALLOWED_SPEC_SYMBOLS = StreamEx.of(ALLOWED_SPEC_SYMBOLS_FOR_TEXT)
            .append(ALLOWED_TAGS_FOR_TEXT.keySet())
            .reverseSorted(comparing(String::length))
            .toImmutableList();

    @Override
    public ValidationResult<TemplateVariable, Defect> apply(TemplateVariable templateVariable) {
        ItemValidationBuilder<TemplateVariable, Defect> vb = ItemValidationBuilder.of(templateVariable);

        vb.check(SpecSymbolsTextVariableValidator::specSymbolsIsCorrectForTextResourceIfNeeded);

        return vb.getResult();
    }

    /**
     * Валидация спецсимволов для текстовых ресурсов.
     * Возвращается ошибка, если внутри & и ; (< и >) спецсимвол введен неверно
     * или в спецсимволе отсутствуют & или ; (< или >)
     */
    @Nullable
    private static Defect specSymbolsIsCorrectForTextResourceIfNeeded(TemplateVariable templateVariable) {
        String text = templateVariable.getInternalValue();

        if (text == null || text.isBlank()) {
            return null;
        }

        // Получение из текста не поддерживаемых спецсимволов и тегов
        Optional<String> notAllowedSpecSymbol =
                getInvalidSpecSymbolByPattern(text, TAGS_PATTERN, ALLOWED_TAGS_FOR_TEXT.keySet())
                        .or(() -> getInvalidSpecSymbolByPattern(text, SPEC_SYMBOLS_PATTERN,
                                ALLOWED_SPEC_SYMBOLS_FOR_TEXT));

        if (notAllowedSpecSymbol.isPresent()) {
            return BannerDefects.invalidSpecSymbols(notAllowedSpecSymbol.get());
        }

        String preprocessedText = text.replace('\n', ' ');

        // Получение из текста таких спецсимволов, у которых неверно поставлены скобки (&..; или <..>)
        for (String specSymbol : SORTED_ALLOWED_SPEC_SYMBOLS) {
            boolean isTag = ALLOWED_TAGS_FOR_TEXT.containsKey(specSymbol);
            String brackets = isTag ? TAG_BRACKETS : SPEC_SYMBOL_BRACKETS;
            String specSymbolWithBrackets = brackets.charAt(0) + specSymbol + brackets.charAt(1);

            List<Integer> occurrences = StringUtils.getIndexesOfAllOccurrences(preprocessedText, specSymbol);
            for (int index : occurrences) {
                if (!isSpecSymbolCorrect(preprocessedText, specSymbol, index, brackets)) {
                    return invalidSpecSymbols(specSymbolWithBrackets);
                }
            }

            // Все вхождения спецсимвола заменяются на " " для того, что бы данный спецсивол
            // не влиял на вхождения других. Например для текста "</strong>" проверка на <strong> не даст ошибку
            preprocessedText = preprocessedText.replace(specSymbolWithBrackets, " ");
        }

        return null;
    }

    private static Optional<String> getInvalidSpecSymbolByPattern(String text, Pattern pattern,
                                                                  Set<String> allowedSpecSymbols) {
        return StreamEx.of(pattern.matcher(text).results())
                .mapToEntry(mr -> text.substring(mr.start() + 1, mr.end() - 1))
                .removeValues(allowedSpecSymbols::contains)
                .mapKeyValue((mr, s) -> mr.group())
                .findFirst();
    }

    /**
     * Спецсимвол считается некорректным, когда
     * - Присутствуют только открывающий или закрывающий символ
     * - Отсутствует и открывающий, и закрывающий символы, при этом спецсимвол должен быть указан в
     * {@link Constants#ILLEGAL_SPEC_SYMBOLS_WITHOUT_BRACKETS}
     */
    private static boolean isSpecSymbolCorrect(String text, String specSymbol, int startIndex, String brackets) {
        Character leftChar = startIndex != 0 ? text.charAt(startIndex - 1) : null;
        boolean isLeftBlank = leftChar == null || Character.isSpaceChar(leftChar);
        boolean isLeftCorrect = !isLeftBlank && brackets.indexOf(leftChar) == 0;

        int lastIndex = startIndex + specSymbol.length();
        Character rightChar = lastIndex != text.length() ? text.charAt(lastIndex) : null;
        boolean isRightBlank = rightChar == null || Character.isSpaceChar(rightChar);
        boolean isRightCorrect = !isRightBlank && brackets.indexOf(rightChar) == 1;

        if (ILLEGAL_SPEC_SYMBOLS_WITHOUT_BRACKETS.contains(specSymbol)) {
            // проверка, что спецсимвол содержит открывающие &(<) и закрывающие ;(>) символы
            return isLeftCorrect && isRightCorrect;
        } else {
            // проверка, что спецсимвол содержит либо и открывающие &(<) и закрывающие ;(>) символы,
            // либо не содержит ни те, ни другие
            return !isLeftCorrect ^ isRightCorrect;
        }
    }
}
