package ru.yandex.direct.core.entity.banner.type.internal;

import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.core.entity.banner.model.TemplateVariable;
import ru.yandex.direct.core.entity.internalads.model.InternalAdsProduct;
import ru.yandex.direct.core.entity.internalads.model.InternalAdsProductOption;
import ru.yandex.direct.core.entity.internalads.model.InternalTemplateInfo;
import ru.yandex.direct.core.entity.internalads.model.ResourceInfo;
import ru.yandex.direct.core.entity.internalads.model.ResourceType;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.regions.Region;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.Collections.singleton;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;

/**
 * Проверяет наличие возрастной метки
 */
@ParametersAreNonnullByDefault
public class TemplateVariablesAgeValidator implements Validator<List<TemplateVariable>, Defect> {
    private final InternalAdsProduct internalAdsProduct;
    private final GeoTree geoTree;
    private final boolean campaignIsMobile;
    private final List<Long> adGroupGeo;
    private final Map<Long, ResourceType> resourceIdToType;

    public static TemplateVariablesAgeValidator templateVariablesAgeValidator(
            InternalTemplateInfo internalTemplateInfo,
            InternalAdsProduct internalAdsProduct,
            GeoTree geoTree,
            boolean campaignIsMobile,
            List<Long> adGroupGeo) {
        return new TemplateVariablesAgeValidator(internalTemplateInfo, internalAdsProduct, geoTree,
                campaignIsMobile, adGroupGeo);
    }

    private TemplateVariablesAgeValidator(InternalTemplateInfo internalTemplateInfo,
                                          InternalAdsProduct internalAdsProduct,
                                          GeoTree geoTree,
                                          boolean campaignIsMobile,
                                          List<Long> adGroupGeo) {

        this.internalAdsProduct = checkNotNull(internalAdsProduct);
        this.geoTree = checkNotNull(geoTree);

        this.campaignIsMobile = campaignIsMobile;
        this.adGroupGeo = adGroupGeo;

        this.resourceIdToType = getResourceTypes(internalTemplateInfo);
    }

    @Override
    public ValidationResult<List<TemplateVariable>, Defect> apply(List<TemplateVariable> variables) {
        ListValidationBuilder<TemplateVariable, Defect> vb = ListValidationBuilder.of(variables);
        vb.checkEachBy(this::variableValueCompliesWithCustomValidation);
        return vb.getResult();
    }

    private ValidationResult<TemplateVariable, Defect> variableValueCompliesWithCustomValidation(TemplateVariable templateVariable) {
        ItemValidationBuilder<TemplateVariable, Defect> vb = ItemValidationBuilder.of(templateVariable);

        vb.check(this::ageVariablePresentIfNeeded, When.valueIs(this::variableTypeIsAge));

        return vb.getResult();
    }

    @Nullable
    private Defect ageVariablePresentIfNeeded(TemplateVariable templateVariable) {
        boolean hasAnyRegionInRussia = geoTree.isAnyRegionIncludedIn(adGroupGeo, singleton(Region.RUSSIA_REGION_ID));
        boolean isAgeValueRequired = hasAnyRegionInRussia
                && (campaignIsMobile || internalAdsProduct.getOptions().contains(InternalAdsProductOption.SOFTWARE));

        return isAgeValueRequired && valueIsAbsent(templateVariable) ? CommonDefects.requiredButEmpty() : null;
    }

    private boolean variableTypeIsAge(TemplateVariable templateVariable) {
        return checkNotNull(resourceIdToType.get(templateVariable.getTemplateResourceId())) == ResourceType.AGE;
    }

    private boolean valueIsAbsent(TemplateVariable var) {
        return var.getInternalValue() == null;
    }

    private Map<Long, ResourceType> getResourceTypes(InternalTemplateInfo templateInfo) {
        return listToMap(templateInfo.getResources(), ResourceInfo::getId, ResourceInfo::getType);
    }
}
