package ru.yandex.direct.core.entity.banner.type.internal;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.adgroup.model.InternalAdGroup;
import ru.yandex.direct.core.entity.banner.container.BannersAddOperationContainer;
import ru.yandex.direct.core.entity.banner.container.BannersOperationContainer;
import ru.yandex.direct.core.entity.banner.model.BannerWithInternalInfo;
import ru.yandex.direct.core.entity.banner.model.InternalModerationInfo;
import ru.yandex.direct.core.entity.banner.model.TemplateVariable;
import ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects;
import ru.yandex.direct.core.entity.banner.type.href.BannerUrlCheckService;
import ru.yandex.direct.core.entity.campaign.model.InternalCampaign;
import ru.yandex.direct.core.entity.image.model.BannerImageFormat;
import ru.yandex.direct.core.entity.internalads.Constants;
import ru.yandex.direct.core.entity.internalads.model.AbstractResourceInfo;
import ru.yandex.direct.core.entity.internalads.model.InternalAdsProduct;
import ru.yandex.direct.core.entity.internalads.model.InternalTemplateInfo;
import ru.yandex.direct.core.entity.internalads.model.ResourceChoice;
import ru.yandex.direct.core.entity.internalads.model.ResourceInfo;
import ru.yandex.direct.core.entity.internalads.model.ResourceRestriction;
import ru.yandex.direct.core.entity.internalads.model.ResourceType;
import ru.yandex.direct.core.entity.internalads.restriction.Restriction;
import ru.yandex.direct.core.entity.internalads.restriction.RestrictionImage;
import ru.yandex.direct.core.entity.internalads.restriction.RestrictionString;
import ru.yandex.direct.core.entity.internalads.service.validation.defects.InternalAdDefects;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptyList;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.templateVariablesMismatch;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

/**
 * Валидирует список заполненных переменных шаблона.
 * <p>
 * Для переменных одного баннера.
 */
public class TemplateVariablesValidator extends AbstractTemplateVariablesValidator<List<TemplateVariable>> {
    private static final SpecSymbolsTextVariableValidator SPEC_SYMBOLS_TEXT_VARIABLE_VALIDATOR =
            new SpecSymbolsTextVariableValidator();

    private final Map<String, BannerImageFormat> imageHashToImageInfo;
    private final Map<Long, List<Restriction>> resourceIdToRestrictions;
    private final InternalAdsProduct internalAdsProduct;
    private final GeoTree geoTree;
    private final boolean campaignIsMobile;
    private final boolean campaignHasImpressionRate;
    private final boolean adGroupsHasImpressionRate;
    private final boolean isUnreachableUrlAnErrorInValidationResult;
    private final List<Long> adGroupGeo;

    // для ресурсов, у которых нет choices -- записей нет
    private final Map<Long, Set<String>> resourceIdToChoiceValues;

    static TemplateVariablesValidator templateVariablesValidator(
            BannerWithInternalInfo banner,
            InternalTemplateInfo internalTemplateInfo,
            InternalAdsProduct internalAdsProduct,
            GeoTree geoTree,
            BannerUrlCheckService bannerUrlCheckService,
            BannersOperationContainer container,
            @Nullable InternalCampaign campaign,
            @Nullable InternalAdGroup adGroup,
            Map<String, BannerImageFormat> imageHashToImageInfo) {
        return new TemplateVariablesValidator(banner, internalTemplateInfo, internalAdsProduct, geoTree,
                bannerUrlCheckService, container, campaign, adGroup, imageHashToImageInfo);
    }

    private TemplateVariablesValidator(BannerWithInternalInfo banner,
                                       InternalTemplateInfo internalTemplateInfo,
                                       InternalAdsProduct internalAdsProduct,
                                       GeoTree geoTree,
                                       BannerUrlCheckService bannerUrlCheckService,
                                       BannersOperationContainer container,
                                       @Nullable InternalCampaign campaign,
                                       @Nullable InternalAdGroup adGroup,
                                       Map<String, BannerImageFormat> imageHashToImageInfo) {
        super(banner, checkNotNull(internalTemplateInfo), bannerUrlCheckService);

        this.internalAdsProduct = checkNotNull(internalAdsProduct);
        this.geoTree = checkNotNull(geoTree);
        this.imageHashToImageInfo = checkNotNull(imageHashToImageInfo);
        // При копировании недоступные урлы идут в warnings (чтобы баннеры копировались остановленными), иначе в errors
        this.isUnreachableUrlAnErrorInValidationResult =
                !((container instanceof BannersAddOperationContainer) && ((BannersAddOperationContainer) container).isCopy());

        Map<Long, ResourceInfo> resourceInfoMap = StreamEx.of(internalTemplateInfo.getResources())
                .mapToEntry(AbstractResourceInfo::getId, Function.identity())
                .toMap();

        this.campaignIsMobile = nvl(ifNotNull(campaign, InternalCampaign::getIsMobile), false);
        this.campaignHasImpressionRate = ifNotNull(campaign, InternalCampaign::getImpressionRateCount) != null;

        this.adGroupGeo = nvl(ifNotNull(adGroup, InternalAdGroup::getGeo), Collections.emptyList());
        this.adGroupsHasImpressionRate = ifNotNull(adGroup, InternalAdGroup::getRf) != null;

        this.resourceIdToRestrictions =
                EntryStream.of(resourceInfoMap).mapValues(ResourceInfo::getValueRestrictions).toImmutableMap();

        this.resourceIdToChoiceValues = EntryStream.of(resourceInfoMap)
                .mapValues(ResourceInfo::getChoices)
                .nonNullValues()
                .removeValues(List::isEmpty)
                .mapValues(choices -> listToSet(choices, ResourceChoice::getValue))
                .toImmutableMap();
    }

    @Override
    public ValidationResult<List<TemplateVariable>, Defect> apply(List<TemplateVariable> variables) {
        ListValidationBuilder<TemplateVariable, Defect> vb = ListValidationBuilder.of(variables);
        vb.check(this::variablesFitToTemplateInfo);

        // ошибки несоотвествия набора переменных шаблону (в variablesFitToTemplateInfo) вешаются на весь список и не
        // видны
        // для When.isValid на отдельных элементах, но проверять отдельные переменные не имеет смысла, если набор
        // неверен, поэтому
        // дальнейшие проверки под if-ом
        if (!vb.getResult().hasAnyErrors()) {
            vb.checkBy(this::requiredVariablesHaveValue);
        }

        // дальнейшие проверки проверяют значения и расчитывают, что набор переменных валиден, все нужные переменные
        // заполнены
        if (!vb.getResult().hasAnyErrors()) {
            vb.checkEach(this::variableValueCompliesWithChoices, When.isValid());
            vb.checkEachBy(this::variableValueCompliesWithRestrictions, When.isValid());
            vb.checkEachBy(this::variableValueCompliesWithCustomValidation, When.isValid());
            vb.checkBy(TemplateVariablesAgeValidator.templateVariablesAgeValidator(internalTemplateInfo,
                    internalAdsProduct, geoTree, campaignIsMobile, adGroupGeo), When.isValid());

            Boolean statusShowAfterModeration =
                    ifNotNull(banner.getModerationInfo(), InternalModerationInfo::getStatusShowAfterModeration);
            // Проверка доступности ссылок делается только для включенных баннеров или если планируем включить новую
            // версию модерируемого баннера - statusShowAfterModeration
            // statusShow=null бывает у новых баннеров, считаем их включенными по умолчанию (см. схему БД)
            if (nvl(banner.getStatusShow(), true) || nvl(statusShowAfterModeration, false)) {
                if (isUnreachableUrlAnErrorInValidationResult) {
                    vb.checkEach(this::variableValueUrlIsReachable, When.isValid());
                } else {
                    vb.weakCheckEach(this::variableValueUrlIsReachable, When.isValid());
                }
            }
        }
        return vb.getResult();
    }

    private ValidationResult<List<TemplateVariable>, Defect> requiredVariablesHaveValue(List<TemplateVariable> templateVariables) {
        List<ResourceRestriction> resourceRestrictions = internalTemplateInfo.getResourceRestrictions();
        if (resourceRestrictions == null || resourceRestrictions.isEmpty()) {
            return ValidationResult.success(templateVariables);
        }
        if (StreamEx.of(resourceRestrictions).anyMatch(rr -> varsPresentOrAbsentAsNeeded(rr, templateVariables))) {
            return ValidationResult.success(templateVariables);
        }

        // взять первый и расставить ошибки
        ResourceRestriction mismatchedResourceRestriction = resourceRestrictions.get(0);
        ListValidationBuilder<TemplateVariable, Defect> vb = ListValidationBuilder.of(templateVariables);
        vb.check(addResourceRestrictionsErrorMessage(internalTemplateInfo.getResourceRestrictionsErrorMessage()));
        vb.checkEach(requiredValueIsPresent(mismatchedResourceRestriction));
        vb.checkEach(valueIsAbsentAsNeeded(mismatchedResourceRestriction));
        return vb.getResult();
    }

    private Constraint<List<TemplateVariable>, Defect> addResourceRestrictionsErrorMessage(String errorMessage) {
        return Constraint.fromPredicate(templateVariables -> false,
                InternalAdDefects.resourceRestrictionsNotFollowed(errorMessage));
    }

    @Nonnull
    private Constraint<TemplateVariable, Defect> requiredValueIsPresent(ResourceRestriction resourceRestriction) {
        Set<Long> required = resourceRestriction.getRequired();
        return variable -> {
            if (variable == null || !required.contains(variable.getTemplateResourceId())) {
                return null;
            }
            return valueIsPresent(variable) ? null : CommonDefects.requiredButEmpty();
        };
    }

    @Nonnull
    private Constraint<TemplateVariable, Defect> valueIsAbsentAsNeeded(ResourceRestriction resourceRestriction) {
        Set<Long> mustBeAbsent = resourceRestriction.getAbsent();
        return variable -> {
            if (variable == null || !mustBeAbsent.contains(variable.getTemplateResourceId())) {
                return null;
            }
            return valueIsAbsent(variable) ? null : CommonDefects.mustBeEmpty();
        };
    }

    @Nullable
    private Defect variablesFitToTemplateInfo(List<TemplateVariable> variables) {
        if (variables.size() != internalTemplateInfo.getResources().size()) {
            return templateVariablesMismatch();
        }
        Set<Long> idsFromVariables = listToSet(variables, TemplateVariable::getTemplateResourceId);
        Set<Long> idsFromResources = listToSet(internalTemplateInfo.getResources(), ResourceInfo::getId);
        if (idsFromResources.equals(idsFromVariables)) {
            return null;
        } else {
            return templateVariablesMismatch();
        }
    }

    @Nonnull
    private ValidationResult<TemplateVariable, Defect> variableValueCompliesWithRestrictions(TemplateVariable templateVariable) {
        if (templateVariable.getInternalValue() == null) {
            return ValidationResult.success(templateVariable);
        }

        ItemValidationBuilder<TemplateVariable, Defect> vb = ItemValidationBuilder.of(templateVariable);

        vb.check(this::imageVariableHasInfo, When.valueIs(this::variableTypeIsImage));

        if (!vb.getResult().hasAnyErrors()) {
            List<Restriction> restrictions =
                    resourceIdToRestrictions.getOrDefault(templateVariable.getTemplateResourceId(), emptyList());
            for (Restriction restriction : restrictions) {
                if (restriction instanceof RestrictionString) {
                    RestrictionString stringRestriction = (RestrictionString) restriction;
                    if (restriction.isStrict()) {
                        vb.check(var -> stringRestriction.check(var.getInternalValue()));
                    } else {
                        vb.weakCheck(var -> stringRestriction.check(var.getInternalValue()));
                    }
                } else if (restriction instanceof RestrictionImage) {
                    RestrictionImage imageRestriction = (RestrictionImage) restriction;

                    checkState(variableTypeIsImage(templateVariable));
                    BannerImageFormat imageInfo =
                            checkNotNull(imageHashToImageInfo.get(templateVariable.getInternalValue()));

                    if (restriction.isStrict()) {
                        vb.check(var -> imageRestriction.check(imageInfo));
                    } else {
                        vb.weakCheck(var -> imageRestriction.check(imageInfo));
                    }
                } else {
                    throw new IllegalStateException(String.format("unknown restriction type %s", restriction));
                }
                if (vb.getResult().hasAnyErrors()) {
                    break;
                }
            }
        }
        return vb.getResult();
    }

    @Nullable
    private Defect variableValueCompliesWithChoices(TemplateVariable templateVariable) {
        if (templateVariable.getInternalValue() == null) {
            return null;
        }

        Set<String> choices = resourceIdToChoiceValues.get(templateVariable.getTemplateResourceId());
        if (choices == null) {
            // у ресурса не заданы варианты значений, нечего проверять
            return null;
        }

        if (choices.contains(templateVariable.getInternalValue())) {
            return null;
        }

        return CommonDefects.invalidValue();
    }

    @Nonnull
    private ValidationResult<TemplateVariable, Defect> variableValueCompliesWithCustomValidation(TemplateVariable templateVariable) {
        ItemValidationBuilder<TemplateVariable, Defect> vb = ItemValidationBuilder.of(templateVariable);

        vb.check(this::campaignOrAdGroupImpressionRatePresentIfNeeded, When.valueIs(this::variableTypeIsCloseCounter));
        vb.checkBy(SPEC_SYMBOLS_TEXT_VARIABLE_VALIDATOR, When.valueIs(this::variableTypeIsText));

        return vb.getResult();
    }

    @Nullable
    private Defect imageVariableHasInfo(TemplateVariable templateVariable) {
        return imageHashToImageInfo.containsKey(templateVariable.getInternalValue())
                ? null
                : CommonDefects.objectNotFound();
    }

    /**
     * Валидация на RF (Ограничение частоты)
     * - Если Счетчик закрытия выбрали по кампании, то в кампании должен быть заполнен RF
     * - Если Счетчик закрытия выбрали по группе, то в группе должен быть заполнен RF
     */
    @Nullable
    private Defect campaignOrAdGroupImpressionRatePresentIfNeeded(TemplateVariable templateVariable) {
        if (!campaignHasImpressionRate
                && Constants.CLOSE_BY_CAMPAIGN_COUNTER_VALUE.equals(templateVariable.getInternalValue())) {
            return BannerDefects.requiredCampaignsImpressionRateButEmpty();
        }

        if (!adGroupsHasImpressionRate
                && Constants.CLOSE_BY_AD_GROUP_COUNTER_VALUE.equals(templateVariable.getInternalValue())) {
            return BannerDefects.requiredAdGroupsImpressionRateButEmpty();
        }

        return null;
    }

    private boolean variableTypeIsImage(TemplateVariable templateVariable) {
        return variableTypeIs(ResourceType.IMAGE, templateVariable);
    }

    private boolean variableTypeIsCloseCounter(TemplateVariable templateVariable) {
        return variableTypeIs(ResourceType.CLOSE_COUNTER, templateVariable);
    }

    private boolean variableTypeIsText(TemplateVariable templateVariable) {
        return variableTypeIs(ResourceType.TEXT, templateVariable);
    }

    private boolean variableTypeIs(ResourceType resourceType, TemplateVariable templateVariable) {
        return checkNotNull(resourceIdToType.get(templateVariable.getTemplateResourceId())) == resourceType;
    }

    private boolean varsPresentOrAbsentAsNeeded(ResourceRestriction resourceRestriction,
                                                List<TemplateVariable> templateVariables) {
        Set<Long> required = resourceRestriction.getRequired();
        Set<Long> mustBeAbsent = resourceRestriction.getAbsent();
        for (TemplateVariable var : templateVariables) {
            Long templateResourceId = var.getTemplateResourceId();
            if ((valueIsAbsent(var) && required.contains(templateResourceId)) || (valueIsPresent(var) && mustBeAbsent.contains(templateResourceId))) {
                return false;
            }
        }
        return true;
    }

    private boolean valueIsAbsent(TemplateVariable var) {
        return var.getInternalValue() == null;
    }

    private boolean valueIsPresent(TemplateVariable var) {
        return var.getInternalValue() != null;
    }
}
