package ru.yandex.direct.core.entity.banner.type.language;

import java.util.Collection;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;

import ru.yandex.direct.core.entity.region.AllowedRegionsForLanguageContainer;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.queryrec.QueryrecService;
import ru.yandex.direct.queryrec.model.Language;
import ru.yandex.direct.regions.GeoTreeFactory;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.result.Defect;

import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.inconsistentLanguageWithGeo;
import static ru.yandex.direct.core.entity.region.RegionConstants.LANGUAGE_TO_ALLOWED_REGIONS_CONTAINER;

public class BannerLanguageConstraint implements Constraint<String, Defect> {

    private final QueryrecService queryrecService;
    private final GeoTreeFactory geoTreeFactory;

    private final List<Long> adGroupGeoIds;
    private final Language campaignLanguage;
    private final Language bannerLanguage;
    private final ClientId clientId;
    private final Long clientRegionId;

    private BannerLanguageConstraint(QueryrecService queryrecService,
                                        GeoTreeFactory geoTreeFactory,
                                        @Nullable Language campaignLanguage,
                                        List<Long> adGroupGeoIds,
                                        Language bannerLanguage,
                                        ClientId clientId,
                                        @Nullable Long clientRegionId) {
        this.queryrecService = queryrecService;
        this.geoTreeFactory = geoTreeFactory;

        this.campaignLanguage = campaignLanguage;
        this.adGroupGeoIds = adGroupGeoIds;
        this.bannerLanguage = bannerLanguage;
        this.clientId = clientId;
        this.clientRegionId = clientRegionId;
    }

    public static BannerLanguageConstraint newBannerLanguageConstraint(QueryrecService queryrecService,
                                                                          GeoTreeFactory geoTreeFactory,
                                                                          @Nullable Language campaignLanguage,
                                                                          List<Long> adGroupGeoIds,
                                                                          Language bannerLanguage,
                                                                          ClientId clientId,
                                                                          @Nullable Long clientRegionId) {
        return new BannerLanguageConstraint(
                queryrecService,
                geoTreeFactory,
                campaignLanguage,
                adGroupGeoIds,
                bannerLanguage,
                clientId,
                clientRegionId);
    }

    @Override
    public Defect<Language> apply(String bannerText) {
        if (bannerText == null) {
            return null;
        }

        Language language = bannerLanguage == Language.UNKNOWN ?
                queryrecService.recognize(bannerText, clientId, clientRegionId) : bannerLanguage;

        return campaignLanguage == Language.UNKNOWN ?
                validateWithAdGroup(language) : validateWithCampaign(language);
    }

    private Defect<Language> validateWithCampaign(Language bannerLanguage) {
        if (isValidPair(bannerLanguage, campaignLanguage)) {
            return null;
        }
        return isConsistentWithAdGroupGeo(campaignLanguage) ? null : inconsistentLanguageWithGeo(campaignLanguage);
    }

    private Defect<Language> validateWithAdGroup(Language bannerLanguage) {
        return isConsistentWithAdGroupGeo(bannerLanguage) ? null : inconsistentLanguageWithGeo(bannerLanguage);
    }

    private boolean isValidPair(Language bannerLang, Language campaignLang) {
        return bannerLang == campaignLang
                || (bannerLang == Language.BELARUSIAN && campaignLang == Language.RUSSIAN);
    }

    private boolean isConsistentWithAdGroupGeo(Language language) {
        Set<Long> regionsByLanguage = getAvailableRegionIds(language);
        return regionsByLanguage == null || areRegionsIncluded(adGroupGeoIds, regionsByLanguage);
    }

    /**
     * Проверка вхождения регионов в заданное множество регионов верхнего уровня по российскому геодереву.
     *
     * @param checkedRegions id регионов для проверки
     * @param topRegions     множество регионов верхнего уровня (определённые по языку)
     * @return {@code true}, если проверяемые регионы входят в состав {@code topRegions}, иначе {@code false}.
     */
    private boolean areRegionsIncluded(Collection<Long> checkedRegions, Set<Long> topRegions) {
        return geoTreeFactory.getRussianGeoTree().isRegionsIncludedIn(checkedRegions, topRegions);
    }

    /**
     * @param language язык объявления/кампании
     * @return Множество id регионов, в которых разрешены показы объявлений на данном языке,
     * или {@code null}, если гео-ограничений по показам для этого языка нет.
     */
    @Nullable
    private Set<Long> getAvailableRegionIds(Language language) {
        return LANGUAGE_TO_ALLOWED_REGIONS_CONTAINER
                .getOrDefault(language, AllowedRegionsForLanguageContainer.EMPTY).getAllowedRegionIds();
    }
}
