package ru.yandex.direct.core.entity.banner.type.language;

import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.banner.container.BannersOperationContainer;
import ru.yandex.direct.core.entity.banner.model.BannerWithLanguage;
import ru.yandex.direct.core.entity.banner.repository.type.AbstractBannerDefaultRepositoryTypeSupport;
import ru.yandex.direct.core.entity.banner.repository.type.ModifiedPaths;
import ru.yandex.direct.core.entity.banner.type.system.BannerWithSystemFieldsMappings;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.mysql2grut.enummappers.BannerEnumMappers;
import ru.yandex.grut.objects.proto.client.Schema;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.Collections.emptySet;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS;

@Component
public class BannerWithLanguageRepositoryTypeSupport extends AbstractBannerDefaultRepositoryTypeSupport<BannerWithLanguage> {

    private static final Set<ModelProperty<? super BannerWithLanguage, ?>> GRUT_PROPERTIES =
            Set.of(BannerWithLanguage.LANGUAGE);

    @Autowired
    public BannerWithLanguageRepositoryTypeSupport(DslContextProvider dslContextProvider) {
        super(dslContextProvider, BannerWithLanguage.LANGUAGE, BANNERS.LANGUAGE,
                BannerWithSystemFieldsMappings::languageFromDb,
                BannerWithSystemFieldsMappings::languageToDb);
    }

    @Override
    public void checkBeforeInsert(BannerWithLanguage model) {
        checkLanguageIsNotNull(model);
    }

    @Override
    public void checkBeforeUpdate(Collection<AppliedChanges<BannerWithLanguage>> appliedChanges) {
        appliedChanges.forEach(x -> checkLanguageIsNotNull(x.getModel()));
    }

    @Override
    public Class<BannerWithLanguage> getTypeClass() {
        return BannerWithLanguage.class;
    }

    private void checkLanguageIsNotNull(BannerWithLanguage model) {
        checkNotNull(model.getLanguage(),
                "Language is null. Use language support to fill language. BannerId: " + model.getId());
    }

    @Override
    public Set<ModelProperty<? super BannerWithLanguage, ?>> getGrutSupportedProperties() {
        return GRUT_PROPERTIES;
    }

    @Override
    public Map<Long, ModifiedPaths> applyToGrutObjects(
            @NotNull Map<Long, Schema.TBannerV2.Builder> bannerBuilders,
            @NotNull Collection<AppliedChanges<BannerWithLanguage>> appliedChangesList,
            @NotNull BannersOperationContainer operationContainer) {
        Map<Long, ModifiedPaths> modifiedPathsMap = new HashMap<>();
        for (var appliedChanges : appliedChangesList) {
            if (appliedChanges.getPropertiesForUpdate().contains(BannerWithLanguage.LANGUAGE)) {
                Long id = appliedChanges.getModel().getId();
                var newValue = appliedChanges.getNewValue(BannerWithLanguage.LANGUAGE);
                ModifiedPaths modifiedPaths;
                if (newValue != null) {
                    Schema.TBannerV2.Builder bannerBuilder = bannerBuilders.get(id);
                    bannerBuilder.getSpecBuilder().setLanguage(
                            BannerEnumMappers.Companion.toGrutLanguage(newValue).getNumber()
                    );
                    modifiedPaths = new ModifiedPaths(Set.of("/spec/language"), emptySet());
                } else {
                    modifiedPaths = new ModifiedPaths(emptySet(), Set.of("/spec/language"));
                }
                modifiedPathsMap.put(id, modifiedPaths);
            }
        }
        return modifiedPathsMap;
    }
}
