package ru.yandex.direct.core.entity.banner.type.language;

import java.util.IdentityHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupForBannerOperation;
import ru.yandex.direct.core.entity.banner.container.BannersOperationContainer;
import ru.yandex.direct.core.entity.banner.model.BannerWithLanguage;
import ru.yandex.direct.core.entity.banner.service.text.BannerTextExtractor;
import ru.yandex.direct.core.entity.campaign.model.ContentLanguage;
import ru.yandex.direct.queryrec.QueryrecService;
import ru.yandex.direct.queryrec.model.Language;
import ru.yandex.direct.regions.GeoTreeFactory;
import ru.yandex.direct.utils.CommonUtils;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.result.Defect;

import static java.util.Collections.emptyList;
import static ru.yandex.direct.core.entity.banner.type.language.BannerLanguageConstraint.newBannerLanguageConstraint;
import static ru.yandex.direct.core.entity.banner.type.language.BannerLanguageConverter.convertLanguage;
import static ru.yandex.direct.utils.CommonUtils.nvl;

@Component
public class BannerWithLanguageValidatorProvider {

    private final QueryrecService queryrecService;
    private final GeoTreeFactory geoTreeFactory;
    private final BannerTextExtractor bannerTextExtractor;

    @Autowired
    public BannerWithLanguageValidatorProvider(
            QueryrecService queryrecService, GeoTreeFactory geoTreeFactory,
            BannerTextExtractor bannerTextExtractor) {
        this.queryrecService = queryrecService;
        this.geoTreeFactory = geoTreeFactory;
        this.bannerTextExtractor = bannerTextExtractor;
    }

    public <T extends BannerWithLanguage> Validator<List<T>, Defect> bannerWithLanguageValidator(
            BannersOperationContainer container) {
        return (banners) -> {
            ListValidationBuilder<T, Defect> lvb =
                    ListValidationBuilder.of(banners);
            //если операция комплексная, то пропускаем валидацию. Проверяется отдельно в комплексной валидации
            if (container.isPartOfComplexOperation()) {
                return lvb.getResult();
            }
            Set<Long> adGroupIds = StreamEx.of(banners)
                    .map(BannerWithLanguage::getAdGroupId)
                    .filter(CommonUtils::isValidId)
                    .toSet();

            Map<Long, AdGroupForBannerOperation> adGroupSimpleById = StreamEx.of(container.getUniqueAdGroups())
                    .filter(adGroup -> adGroupIds.contains(adGroup.getId()))
                    .toMap(AdGroupForBannerOperation::getId, Function.identity());
            Set<Long> existingAdGroupIds = adGroupSimpleById.keySet();

            Map<Long, Language> adGroupIdToCampaignLanguage = StreamEx.of(existingAdGroupIds)
                    .mapToEntry(adGroupId -> container.getCampaignIdToCampaignWithContentLanguageMap().get(
                            adGroupSimpleById.get(adGroupId).getCampaignId()))
                    .nonNullValues()
                    .mapValues(campaign -> Optional.ofNullable(campaign.getContentLanguage())
                            .map(ContentLanguage::getTypedValue)
                            .orElse(null))
                    .mapValues(Language::getByName)
                    .toMap();

            IdentityHashMap<T, String> bannerToTextMap = bannerTextExtractor.extractTexts(banners);

            lvb.checkEach(bannerLanguageConstraint(container, adGroupIdToCampaignLanguage,
                    adGroupSimpleById, bannerToTextMap));

            return lvb.getResult();
        };
    }

    private <T extends BannerWithLanguage> Constraint<T, Defect> bannerLanguageConstraint(
            BannersOperationContainer container,
            Map<Long, Language> campaignLanguageByAdGroupId,
            Map<Long, AdGroupForBannerOperation> adGroupSimpleById,
            IdentityHashMap<T, String> bannerToTextMap) {
        return (banner) -> {
            Language campaignLanguage = campaignLanguageByAdGroupId.get(banner.getAdGroupId());
            AdGroupForBannerOperation adGroup = adGroupSimpleById.get(banner.getAdGroupId());
            List<Long> adGroupGeo = adGroup == null ? emptyList() : nvl(adGroup.getGeo(), emptyList());
            Language bannerLanguage = convertLanguage(banner.getLanguage());

            String bannerText = bannerToTextMap.get(banner);
            return newBannerLanguageConstraint(queryrecService, geoTreeFactory,
                    campaignLanguage, adGroupGeo, bannerLanguage, container.getClientId(),
                    container.getClientRegionId())
                    .apply(bannerText);
        };
    }
}
