package ru.yandex.direct.core.entity.banner.type.leadformattributes;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.banner.model.BannerWithLeadformAttributes;
import ru.yandex.direct.core.entity.banner.type.href.BannerHrefValidator;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.banner.service.validation.BannerTextConstraints.charsAreAllowed;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.inconsistentLeadformHrefAndButtonText;
import static ru.yandex.direct.validation.constraint.StringConstraints.maxStringLength;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;

@Component
@ParametersAreNonnullByDefault
public class BannerWithLeadformAttributesValidatorProvider {

    public static final int MAX_TEXT_LENGTH = 25;

    private BannerWithLeadformAttributesValidatorProvider() {

    }

    public static Validator<BannerWithLeadformAttributes, Defect> bannerWithLeadformAttributesValidator() {
        return (banner) -> {
            ModelItemValidationBuilder<BannerWithLeadformAttributes> vb = ModelItemValidationBuilder.of(banner);

            vb.item(BannerWithLeadformAttributes.LEADFORM_HREF)
                    .checkBy(new BannerHrefValidator(), When.notNull())
                    .check(validateLeadformAttributesConsistency(banner));

            vb.item(BannerWithLeadformAttributes.LEADFORM_BUTTON_TEXT)
                    .check(notBlank())
                    .check(maxStringLength(MAX_TEXT_LENGTH))
                    .check(charsAreAllowed())
                    .check(validateLeadformAttributesConsistency(banner));

            return vb.getResult();
        };
    }

    private static Constraint<String, Defect> validateLeadformAttributesConsistency(BannerWithLeadformAttributes banner) {
        return Constraint.fromPredicate(
                (s) -> {
                    boolean hrefIsNull = banner.getLeadformHref() == null;
                    boolean textIsNull = banner.getLeadformButtonText() == null;

                    return hrefIsNull == textIsNull;
                }, inconsistentLeadformHrefAndButtonText());
    }
}
