package ru.yandex.direct.core.entity.banner.type.logos;

import java.util.List;
import java.util.Map;
import java.util.Set;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.banner.container.BannersOperationContainer;
import ru.yandex.direct.core.entity.banner.model.BannerWithLogo;
import ru.yandex.direct.core.entity.banner.model.BannerWithLogoExtended;
import ru.yandex.direct.core.entity.banner.model.ImageSize;
import ru.yandex.direct.core.entity.banner.model.ImageType;
import ru.yandex.direct.core.entity.banner.service.validation.BannerImageHashValidator;
import ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects;
import ru.yandex.direct.core.entity.image.repository.BannerImageFormatRepository;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackage;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ListItemValidator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.image.service.ImageConstants.LOGO_EXTENDED_IMAGE_FORMAT_TYPES;
import static ru.yandex.direct.core.entity.image.service.ImageUtils.isSizeAvailableForLogo;
import static ru.yandex.direct.core.entity.image.service.ImageUtils.isSizeAvailableForLogoExtended;
import static ru.yandex.direct.utils.FunctionalUtils.mapSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;

@Component
public class BannerWithLogoValidatorProvider {
    private static final Set<ImageType> LOGO_EXTENDED_IMAGE_TYPES = mapSet(LOGO_EXTENDED_IMAGE_FORMAT_TYPES,
            ImageType::fromSource);

    private final BannerImageFormatRepository bannerImageFormatRepository;

    public BannerWithLogoValidatorProvider(BannerImageFormatRepository bannerImageFormatRepository) {
        this.bannerImageFormatRepository = bannerImageFormatRepository;
    }

    public ListItemValidator<BannerWithLogo, Defect> bannerWithLogoValidator(BannersOperationContainer container,
                                                                             List<BannerWithLogo> banners) {
        Set<String> imageHashes = StreamEx.of(banners)
                .map(BannerWithLogo::getLogoImageHash)
                .nonNull()
                .toSet();

        Map<String, Pair<ImageType, ImageSize>> typeAndSizeByImageHash = bannerImageFormatRepository
                .getExistingImageSizes(container.getShard(), container.getClientId(), imageHashes);

        Map<String, ImageType> typedHashes = EntryStream.of(typeAndSizeByImageHash)
            .mapValues(Pair::getLeft)
            .toMap();

        BannerImageHashValidator logoHashValidator = new BannerImageHashValidator(typedHashes,
                Set.of(ImageType.LOGO));
        BannerImageHashValidator logoExtendedHashValidator = new BannerImageHashValidator(typedHashes,
                LOGO_EXTENDED_IMAGE_TYPES);

        return (index, banner) -> {
            boolean isLogoExtended = banner instanceof BannerWithLogoExtended;
            PricePackage pricePackage = container.getPricePackage(banner);
            boolean isControlByPackage = pricePackage != null && pricePackage.isFrontpageVideoPackage();
            ModelItemValidationBuilder<BannerWithLogo> vb = ModelItemValidationBuilder.of(banner);
            vb.item(BannerWithLogo.LOGO_IMAGE_HASH)
                    .checkBy(isLogoExtended ? logoExtendedHashValidator : logoHashValidator)
                    .check(Constraint.fromPredicate(imageHash -> {
                        ImageSize imageSize = typeAndSizeByImageHash.get(imageHash).getRight();
                        if (isLogoExtended) {
                            return isSizeAvailableForLogoExtended(imageSize.getWidth(), imageSize.getHeight());
                        }
                        return isSizeAvailableForLogo(imageSize.getWidth(), imageSize.getHeight());
                    }, BannerDefects.imageSizeInvalid()), When.isValid())
                    .check(notNull(), When.isTrue(isControlByPackage));
            return vb.getResult();
        };
    }
}
