package ru.yandex.direct.core.entity.banner.type.measurers;

import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.function.Function;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import com.google.common.base.Strings;

import ru.yandex.direct.core.entity.banner.model.BannerMeasurer;
import ru.yandex.direct.core.entity.banner.type.measurers.model.BannerMeasurerParamsAdmetrica;
import ru.yandex.direct.core.entity.banner.type.measurers.model.BannerMeasurerParamsWeborama;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.defect.CollectionDefects;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;

import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.JsonUtils.fromJson;

public class BannerMeasurersConstraints {
    private static final ObjectMapper MAPPER = new ObjectMapper()
            .registerModule(new JavaTimeModule())
            .enable(DeserializationFeature.FAIL_ON_TRAILING_TOKENS);
    private static final TypeReference<Map<String, Object>> GENERIC_MAP_TYPE = new TypeReference<>() {
    };

    private BannerMeasurersConstraints() {
    }

    public static Constraint<List<BannerMeasurer>, Defect> validateDuplicateMeasurerSystems() {
        return bannerMeasurers -> {
            if (bannerMeasurers == null || bannerMeasurers.isEmpty()) {
                return null;
            }

            var uniqueMeasurers = listToSet(bannerMeasurers, BannerMeasurer::getBannerMeasurerSystem);
            return uniqueMeasurers.size() < bannerMeasurers.size() ? CollectionDefects.duplicatedElement() : null;
        };
    }

    public static Constraint<BannerMeasurer, Defect> bannerMeasurerNotNull() {
        return measurer -> {
            boolean somethingIsNull = measurer == null
                    || measurer.getBannerMeasurerSystem() == null;
            return somethingIsNull ? CommonDefects.notNull() : null;
        };
    }

    public static Constraint<BannerMeasurer, Defect> validBannerMeasurerParams() {
        return measurer -> {
            try {
                switch (measurer.getBannerMeasurerSystem()) {
                    case ADMETRICA:
                        return validateCustomJsonParams(
                                measurer,
                                BannerMeasurerParamsAdmetrica.class,
                                BannerMeasurersConstraints::validateAdmetricaParams);

                    case WEBORAMA:
                        return validateCustomJsonParams(
                                measurer,
                                BannerMeasurerParamsWeborama.class,
                                BannerMeasurersConstraints::validateWeboramaParams);

                    case NO:
                        return CommonDefects.invalidValue();

                    default:
                        if (Strings.isNullOrEmpty(measurer.getParams())) {
                            return null;
                        }
                        MAPPER.readValue(measurer.getParams(), GENERIC_MAP_TYPE);
                        return null;
                }
            } catch (IllegalArgumentException | IOException exc) {
                return CommonDefects.invalidValue();
            }
        };
    }

    private static <T> Defect validateCustomJsonParams(
            BannerMeasurer measurer,
            Class<T> type,
            Function<T, Defect> validator) {
        var params = fromJson(measurer.getParams(), type);
        return validator.apply(params);
    }

    private static Defect validateAdmetricaParams(BannerMeasurerParamsAdmetrica params) {
        boolean somethingIsNull = params.getCampaignId() == null
                || params.getPlacementId() == null
                || params.getCreativeId() == null
                || params.getCriteria() == null
                || params.getType() == null;

        if (somethingIsNull) {
            return CommonDefects.invalidValue();
        }

        if (params.getCampaignId() <= 0 || params.getPlacementId() <= 0 || params.getCreativeId() <= 0) {
            return CommonDefects.invalidValue();
        }

        return null;
    }

    private static Defect validateWeboramaParams(BannerMeasurerParamsWeborama params) {
        boolean somethingIsNull = params.getClientId() == null
                || params.getInsertionId() == null
                || params.getCreativeId() == null;

        if (somethingIsNull) {
            return CommonDefects.invalidValue();
        }

        if (params.getClientId() <= 0 || params.getInsertionId() <= 0 || params.getCreativeId() <= 0) {
            return CommonDefects.invalidValue();
        }

        return null;
    }
}
