package ru.yandex.direct.core.entity.banner.type.measurers;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.banner.model.BannerMeasurer;
import ru.yandex.direct.core.entity.banner.model.BannerMeasurerSystem;
import ru.yandex.direct.dbschema.ppc.enums.BannerMeasurersMeasurerSystem;
import ru.yandex.direct.dbschema.ppc.enums.CampMeasurersMeasurerSystem;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;

import static java.util.Collections.emptyMap;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS_PERFORMANCE;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_MEASURERS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_PRICES;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMP_MEASURERS;
import static ru.yandex.direct.dbschema.ppc.Tables.CLIENTS;
import static ru.yandex.direct.dbschema.ppc.Tables.PERF_CREATIVES;
import static ru.yandex.direct.dbschema.ppc.Tables.PHRASES;
import static ru.yandex.direct.dbschema.ppc.Tables.USERS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.utils.CollectionUtils.isEmpty;

@Repository
@ParametersAreNonnullByDefault
public class BannerMeasurersRepository {

    private final DslContextProvider dslContextProvider;
    private final JooqMapperWithSupplier<BannerMeasurer> bannerMeasurerMapper = createBannerMeasurerMapper();

    @Autowired
    public BannerMeasurersRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
    }

    public void deleteBannersMeasurers(DSLContext context, List<Long> bannersIds) {
        if (bannersIds.isEmpty()) {
            return;
        }

        context.deleteFrom(BANNER_MEASURERS)
                .where(BANNER_MEASURERS.BID.in(bannersIds))
                .execute();
    }

    public String getBannersWithMoatCsv(int shard, LocalDateTime dateTimeFrom, boolean withHeader) {
        var bids =
                dslContextProvider.ppc(shard)
                        .select(BANNER_MEASURERS.BID)
                        .from(BANNER_MEASURERS)
                        .where(BANNER_MEASURERS.MEASURER_SYSTEM.eq(BannerMeasurersMeasurerSystem.moat)).fetch();

        var cids =
                dslContextProvider.ppc(shard)
                        .select(CAMP_MEASURERS.CID)
                        .from(CAMP_MEASURERS)
                        .where(CAMP_MEASURERS.MEASURER_SYSTEM.eq(CampMeasurersMeasurerSystem.moat)).fetch();

        return dslContextProvider.ppc(shard)
                .select(CLIENTS.CLIENT_ID.as("advertiser_id"),
                        USERS.LOGIN.as("advertiser_label"),
                        CAMPAIGNS.CID.as("campaign_id"),
                        CAMPAIGNS.NAME.as("campaign_label"),
                        PHRASES.PID.as("line_item_id"),
                        PHRASES.GROUP_NAME.as("line_item_label"),
                        DSL.iif(PERF_CREATIVES.CREATIVE_ID.isNotNull(), PERF_CREATIVES.CREATIVE_ID,
                                BANNERS.BID).as("creative_id"),
                        DSL.iif(PERF_CREATIVES.CREATIVE_ID.isNotNull(), PERF_CREATIVES.NAME, BANNERS.TITLE).as(
                                "creative_label"))
                .from(BANNERS)
                .leftJoin(BANNERS_PERFORMANCE).on(BANNERS.BID.eq(BANNERS_PERFORMANCE.BID))
                .leftJoin(PERF_CREATIVES).on(BANNERS_PERFORMANCE.CREATIVE_ID.eq(PERF_CREATIVES.CREATIVE_ID))
                .join(PHRASES).on(BANNERS.PID.eq(PHRASES.PID))
                .join(CAMPAIGNS).on(BANNERS.CID.eq(CAMPAIGNS.CID))
                .join(CLIENTS).on(CLIENTS.CLIENT_ID.eq(CAMPAIGNS.CLIENT_ID))
                .join(USERS).on(CLIENTS.CHIEF_UID.eq(USERS.UID))
                .where(
                        DSL.or(
                                BANNERS.BID.in(bids),
                                BANNERS.CID.in(cids)
                        )
                )
                .and(CAMPAIGNS.LAST_SHOW_TIME.ge(dateTimeFrom))
                .fetch()
                .formatCSV(withHeader, ',');
    }

    public Map<Long, List<BannerMeasurer>> getMeasurersByBannerIds(int shard, @Nullable Collection<Long> bannerIds) {
        if (isEmpty(bannerIds)) {
            return emptyMap();
        }

        return dslContextProvider.ppc(shard)
                .select(BANNER_MEASURERS.BID,
                        BANNER_MEASURERS.MEASURER_SYSTEM,
                        BANNER_MEASURERS.PARAMS,
                        BANNER_MEASURERS.HAS_INTEGRATION)
                .from(BANNER_MEASURERS)
                .where(BANNER_MEASURERS.BID.in(bannerIds))
                .fetchGroups(BANNER_PRICES.BID, bannerMeasurerMapper::fromDb);
    }


    private static JooqMapperWithSupplier<BannerMeasurer> createBannerMeasurerMapper() {
        return JooqMapperWithSupplierBuilder.builder(BannerMeasurer::new)
                .map(convertibleProperty(
                        BannerMeasurer.BANNER_MEASURER_SYSTEM,
                        BANNER_MEASURERS.MEASURER_SYSTEM,
                        BannerMeasurerSystem::fromSource,
                        BannerMeasurerSystem::toSource))
                .map(convertibleProperty(
                        BannerMeasurer.PARAMS,
                        BANNER_MEASURERS.PARAMS,
                        dbValue -> dbValue == null ? "" : dbValue,
                        javaValue -> "".equals(javaValue) ? null : javaValue))
                .map(convertibleProperty(
                        BannerMeasurer.HAS_INTEGRATION,
                        BANNER_MEASURERS.HAS_INTEGRATION,
                        dbValue -> dbValue != null && dbValue == 1L,
                        javaValue -> (javaValue != null && javaValue) ? 1L : 0L))
                .build();
    }
}
