package ru.yandex.direct.core.entity.banner.type.measurers;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.jooq.Condition;
import org.jooq.DSLContext;
import org.jooq.impl.DSL;
import org.jooq.util.mysql.MySQLDSL;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.banner.container.BannerRepositoryContainer;
import ru.yandex.direct.core.entity.banner.model.BannerMeasurer;
import ru.yandex.direct.core.entity.banner.model.BannerMeasurerSystem;
import ru.yandex.direct.core.entity.banner.model.BannerWithMeasurers;
import ru.yandex.direct.core.entity.banner.model.InternalBannerMeasurer;
import ru.yandex.direct.core.entity.banner.repository.AddOrUpdateAndDeleteByKeyContainer;
import ru.yandex.direct.core.entity.banner.repository.AddOrUpdateAndDeleteContainer;
import ru.yandex.direct.core.entity.banner.repository.type.AbstractMultiRowEntityRepositoryTypeSupport;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.model.AppliedChanges;

import static java.util.Collections.emptyList;
import static java.util.function.Function.identity;
import static org.apache.commons.collections4.CollectionUtils.isEmpty;
import static org.apache.commons.lang3.ObjectUtils.defaultIfNull;
import static ru.yandex.direct.core.entity.banner.model.BannerWithMeasurers.MEASURERS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_MEASURERS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.model.AppliedChanges.isChanged;
import static ru.yandex.direct.utils.FunctionalUtils.flatMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
@Component
public class BannerWithMeasurersRepositoryTypeSupport
        extends AbstractMultiRowEntityRepositoryTypeSupport
        <BannerWithMeasurers, InternalBannerMeasurer> {
    private final JooqMapperWithSupplier<InternalBannerMeasurer> mapper;

    protected BannerWithMeasurersRepositoryTypeSupport(DslContextProvider dslContextProvider) {
        super(dslContextProvider);
        this.mapper = createMapper();
    }

    @Override
    protected AddOrUpdateAndDeleteContainer<InternalBannerMeasurer> buildAddOrUpdateAndDeleteContainer(
            DSLContext context,
            BannerRepositoryContainer updateParameters,
            Collection<AppliedChanges<BannerWithMeasurers>> bannersChangesCollection) {
        AddOrUpdateAndDeleteByKeyContainer.Builder<InternalBannerMeasurer, BannerMeasurerSystem> builder =
                AddOrUpdateAndDeleteByKeyContainer.builder(InternalBannerMeasurer::getBannerMeasurerSystem)
                        .skipNewRowsByEquals();

        bannersChangesCollection.stream()
                .filter(isChanged(MEASURERS))
                .forEach(ac -> {
                    var old = mapToInternalList(ac.getOldValue(MEASURERS), ac.getModel().getId());
                    var newValues = mapToInternalList(ac.getNewValue(MEASURERS), ac.getModel().getId());
                    builder.addDiff(old, newValues);
                });

        return builder.build();
    }

    @Nonnull
    @Override
    protected List<InternalBannerMeasurer> extractNewRepositoryModels(Collection<BannerWithMeasurers> banners) {
        return flatMap(banners, BannerWithMeasurersRepositoryTypeSupport::mapToInternalList);
    }

    private static List<InternalBannerMeasurer> mapToInternalList(BannerWithMeasurers banner) {
        if (isEmpty(banner.getMeasurers())) {
            return emptyList();
        }
        return mapToInternalList(banner.getMeasurers(), banner.getId());
    }

    private static List<InternalBannerMeasurer> mapToInternalList(List<BannerMeasurer> list, Long bannerId) {
        if (isEmpty(list)) {
            return emptyList();
        }
        return mapList(list, it -> toInternalBannerMeasurer(it, bannerId));
    }

    private static BannerMeasurer toBannerMeasurer(InternalBannerMeasurer measurer) {
        return new BannerMeasurer()
                .withBannerMeasurerSystem(measurer.getBannerMeasurerSystem())
                .withParams(measurer.getParams())
                .withHasIntegration(measurer.getHasIntegration());
    }

    private static InternalBannerMeasurer toInternalBannerMeasurer(BannerMeasurer measurer, Long bannerId) {
        return new InternalBannerMeasurer()
                .withBannerId(bannerId)
                .withBannerMeasurerSystem(measurer.getBannerMeasurerSystem())
                .withParams(measurer.getParams())
                .withHasIntegration(measurer.getHasIntegration());
    }

    @Override
    protected void delete(DSLContext context, Collection<InternalBannerMeasurer> measurersToDelete) {
        if (isEmpty(measurersToDelete)) {
            return;
        }
        Condition condition = StreamEx.of(measurersToDelete)
                .map(m -> BANNER_MEASURERS.BID.in(m.getBannerId())
                        .and(BANNER_MEASURERS.MEASURER_SYSTEM.eq(BannerMeasurerSystem.toSource(m.getBannerMeasurerSystem()))))
                .reduce(Condition::or)
                .orElse(DSL.falseCondition());
        context.deleteFrom(BANNER_MEASURERS)
                .where(condition)
                .execute();
    }

    @Override
    protected void addOrUpdate(DSLContext context, Collection<InternalBannerMeasurer> repositoryModelsToAddOrUpdate) {
        new InsertHelper<>(context, BANNER_MEASURERS)
                .addAll(mapper, repositoryModelsToAddOrUpdate)
                .onDuplicateKeyUpdate()
                .set(BANNER_MEASURERS.PARAMS, MySQLDSL.values(BANNER_MEASURERS.PARAMS))
                .set(BANNER_MEASURERS.HAS_INTEGRATION, MySQLDSL.values(BANNER_MEASURERS.HAS_INTEGRATION))
                .executeIfRecordsAdded();
    }

    @Override
    public Class<BannerWithMeasurers> getTypeClass() {
        return BannerWithMeasurers.class;
    }

    @Override
    public void enrichModelFromOtherTables(DSLContext dslContext, Collection<BannerWithMeasurers> banners) {
        Set<Long> bannerIds = listToSet(banners, BannerWithMeasurers::getId);
        var bannerMeasurers = dslContext
                .select(mapper.getFieldsToRead())
                .from(BANNER_MEASURERS)
                .where(BANNER_MEASURERS.BID.in(bannerIds))
                .fetch(mapper::fromDb);
        Map<Long, List<InternalBannerMeasurer>> bannerIdToMeasures = StreamEx.of(bannerMeasurers)
                .mapToEntry(InternalBannerMeasurer::getBannerId, identity())
                .grouping();
        for (BannerWithMeasurers banner : banners) {
            List<BannerMeasurer> measurers = mapList(defaultIfNull(bannerIdToMeasures.get(banner.getId()), emptyList()),
                    it -> toBannerMeasurer(it));
            banner.setMeasurers(measurers);
        }
    }

    private static JooqMapperWithSupplier<InternalBannerMeasurer> createMapper() {
        return JooqMapperWithSupplierBuilder.builder(InternalBannerMeasurer::new)
                .map(property(InternalBannerMeasurer.BANNER_ID, BANNER_MEASURERS.BID))
                .map(convertibleProperty(
                        BannerMeasurer.BANNER_MEASURER_SYSTEM,
                        BANNER_MEASURERS.MEASURER_SYSTEM,
                        BannerMeasurerSystem::fromSource,
                        BannerMeasurerSystem::toSource))
                .map(convertibleProperty(
                        BannerMeasurer.PARAMS,
                        BANNER_MEASURERS.PARAMS,
                        dbValue -> dbValue == null ? "" : dbValue,
                        javaValue -> "".equals(javaValue) ? null : javaValue))
                .map(convertibleProperty(
                        BannerMeasurer.HAS_INTEGRATION,
                        BANNER_MEASURERS.HAS_INTEGRATION,
                        dbValue -> dbValue != null && dbValue == 1L,
                        javaValue -> (javaValue != null && javaValue) ? 1L : 0L))
                .build();
    }
}
