package ru.yandex.direct.core.entity.banner.type.mobilecontent;

import java.util.Collection;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.jooq.util.mysql.MySQLDSL;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.banner.model.BannerWithMobileContent;
import ru.yandex.direct.core.entity.banner.model.NewMobileContentPrimaryAction;
import ru.yandex.direct.core.entity.banner.repository.type.AbstractFlatRelatedEntityUpsertRepositoryTypeSupport;
import ru.yandex.direct.dbschema.ppc.tables.records.BannersMobileContentRecord;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapper;
import ru.yandex.direct.jooqmapper.JooqMapperBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.model.AppliedChanges;

import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS_MOBILE_CONTENT;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.jooqmapper.write.WriterBuilders.fromProperty;

@Component
@ParametersAreNonnullByDefault
public class BannerWithMobileContentRepositoryTypeSupport
        extends AbstractFlatRelatedEntityUpsertRepositoryTypeSupport
        <BannerWithMobileContent, BannersMobileContentRecord> {

    private static final Logger logger =
            LoggerFactory.getLogger(BannerWithMobileContentRepositoryTypeSupport.class);

    @Autowired
    public BannerWithMobileContentRepositoryTypeSupport(DslContextProvider dslContextProvider) {
        super(dslContextProvider, BANNERS_MOBILE_CONTENT.BID, createMapper());
    }

    @Override
    protected void upsertEntity(DSLContext context, Collection<BannerWithMobileContent> banners) {
        var insertHelper = new InsertHelper<>(context, BANNERS_MOBILE_CONTENT);
        insertHelper.addAll(getJooqMapper(), banners);
        insertHelper.onDuplicateKeyUpdate()
                .set(BANNERS_MOBILE_CONTENT.PRIMARY_ACTION, MySQLDSL.values(BANNERS_MOBILE_CONTENT.PRIMARY_ACTION))
                .set(BANNERS_MOBILE_CONTENT.REFLECTED_ATTRS, MySQLDSL.values(BANNERS_MOBILE_CONTENT.REFLECTED_ATTRS))
                .set(BANNERS_MOBILE_CONTENT.IMPRESSION_URL, MySQLDSL.values(BANNERS_MOBILE_CONTENT.IMPRESSION_URL));
        insertHelper.executeIfRecordsAdded();
    }

    @Override
    protected boolean isAddEntity(BannerWithMobileContent model) {
        return model.getPrimaryAction() != null && model.getReflectedAttributes() != null;
    }

    @Override
    protected boolean isUpsertEntity(AppliedChanges<BannerWithMobileContent> appliedChange) {
        return appliedChange.changedAndNotDeleted(BannerWithMobileContent.PRIMARY_ACTION)
                || appliedChange.changedAndNotDeleted(BannerWithMobileContent.REFLECTED_ATTRIBUTES)
                || appliedChange.changed(BannerWithMobileContent.IMPRESSION_URL);
    }

    @Override
    protected boolean isDeleteEntity(AppliedChanges<BannerWithMobileContent> appliedChange) {
        if (appliedChange.deleted(BannerWithMobileContent.PRIMARY_ACTION) ||
                appliedChange.deleted(BannerWithMobileContent.REFLECTED_ATTRIBUTES)) {
            logger.error("Try to delete mobile content for bid: {}. Delete entity is not supported for " +
                    "BannerWithMobileContent", appliedChange.getModel().getId());
        }
        return false;
    }

    private static JooqMapper<BannerWithMobileContent> createMapper() {
        return JooqMapperBuilder.<BannerWithMobileContent>builder()
                .map(convertibleProperty(BannerWithMobileContent.REFLECTED_ATTRIBUTES,
                        BANNERS_MOBILE_CONTENT.REFLECTED_ATTRS,
                        BannerWithMobileContentUtils::reflectedAttrsFromDbConverter,
                        BannerWithMobileContentUtils::reflectedAttrsToDbConverter))
                .map(convertibleProperty(BannerWithMobileContent.PRIMARY_ACTION,
                        BANNERS_MOBILE_CONTENT.PRIMARY_ACTION,
                        NewMobileContentPrimaryAction::fromSource, NewMobileContentPrimaryAction::toSource))
                .map(property(BannerWithMobileContent.IMPRESSION_URL, BANNERS_MOBILE_CONTENT.IMPRESSION_URL))
                .writeField(BANNERS_MOBILE_CONTENT.BID, fromProperty(BannerWithMobileContent.ID))
                .build();
    }

    @Override
    public Class<BannerWithMobileContent> getTypeClass() {
        return BannerWithMobileContent.class;
    }
}
