package ru.yandex.direct.core.entity.banner.type.multicard

import org.jooq.DSLContext
import org.jooq.util.mysql.MySQLDSL
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.banner.container.BannerRepositoryContainer
import ru.yandex.direct.core.entity.banner.model.BannerMulticard
import ru.yandex.direct.core.entity.banner.model.BannerMulticardInternal
import ru.yandex.direct.core.entity.banner.model.BannerMulticardsCurrency
import ru.yandex.direct.core.entity.banner.model.BannerWithMulticardSetMulticards
import ru.yandex.direct.core.entity.banner.repository.AddOrUpdateAndDeleteByKeyContainer
import ru.yandex.direct.core.entity.banner.repository.AddOrUpdateAndDeleteContainer
import ru.yandex.direct.core.entity.banner.repository.type.AbstractMultiRowEntityRepositoryTypeSupport
import ru.yandex.direct.dbschema.ppc.Tables.BANNER_MULTICARDS
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder
import ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty
import ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property
import ru.yandex.direct.jooqmapperhelper.InsertHelper
import ru.yandex.direct.model.AppliedChanges

@Component
class BannerWithMulticardSetMulticardsRepositoryTypeSupport(
    private val shardHelper: ShardHelper,
    dslContextProvider: DslContextProvider,
) : AbstractMultiRowEntityRepositoryTypeSupport<BannerWithMulticardSetMulticards, BannerMulticardInternal>(
    dslContextProvider
) {
    private val bannerMulticardMapper = JooqMapperWithSupplierBuilder.builder { BannerMulticardInternal() }
        .map(property(BannerMulticardInternal.MULTICARD_ID, BANNER_MULTICARDS.MULTICARD_ID))
        .map(property(BannerMulticardInternal.BID, BANNER_MULTICARDS.BID))
        .map(property(BannerMulticardInternal.ORDER, BANNER_MULTICARDS.ORDER_NUM))
        .map(property(BannerMulticardInternal.TEXT, BANNER_MULTICARDS.TEXT))
        .map(property(BannerMulticardInternal.IMAGE_HASH, BANNER_MULTICARDS.IMAGE_HASH))
        .map(property(BannerMulticardInternal.HREF, BANNER_MULTICARDS.HREF))
        .map(property(BannerMulticardInternal.PRICE_OLD, BANNER_MULTICARDS.PRICE_OLD))
        .map(property(BannerMulticardInternal.PRICE, BANNER_MULTICARDS.PRICE))
        .map(convertibleProperty(
            BannerMulticardInternal.CURRENCY, BANNER_MULTICARDS.CURRENCY,
            BannerMulticardsCurrency::fromSource,
            BannerMulticardsCurrency::toSource))
        .build()

    override fun getTypeClass() = BannerWithMulticardSetMulticards::class.java

    override fun enrichModelFromOtherTables(
        dslContext: DSLContext,
        models: Collection<BannerWithMulticardSetMulticards>
    ) {
        val bannerIds = models.map { it.id }.toSet()
        val multicardsInternal = dslContext
            .select(bannerMulticardMapper.fieldsToRead)
            .from(BANNER_MULTICARDS)
            .where(BANNER_MULTICARDS.BID.`in`(bannerIds))
            .orderBy(BANNER_MULTICARDS.BID, BANNER_MULTICARDS.ORDER_NUM)
            .fetch(bannerMulticardMapper::fromDb)

        val multicardsByBid = multicardsInternal
            .groupBy { it.bid }
            // сортировка для уверенности, groupBy должен сохранять порядок сам по себе
            .mapValues { (_, multicards) -> multicards.sortedBy { it.order } }

        models.forEach { banner ->
            val multicards = multicardsByBid[banner.id]
            if (multicards != null) {
                banner.multicards = fromInternalMulticards(multicards)
            }
        }
    }

    override fun extractNewRepositoryModels(
        banners: Collection<BannerWithMulticardSetMulticards>
    ): List<BannerMulticardInternal> {
        val multicards = banners.flatMap { banner ->
            val multicards = banner.multicards.orEmpty()
            toInternalMulticards(banner.id, multicards)
        }
        enrichWithCardId(multicards)
        return multicards
    }

    private fun enrichWithCardId(bannerMulticards: Collection<BannerMulticardInternal>) {
        if (bannerMulticards.isEmpty()) return
        val ids = shardHelper.generateBannerMulticardIds(bannerMulticards.size)
        bannerMulticards.zip(ids).forEach { (bannerMulticard, cardId) ->
            bannerMulticard.multicardId = cardId
        }
    }

    override fun buildAddOrUpdateAndDeleteContainer(
        context: DSLContext,
        updateParameters: BannerRepositoryContainer,
        bannersChangesCollection: Collection<AppliedChanges<BannerWithMulticardSetMulticards>>,
    ): AddOrUpdateAndDeleteContainer<BannerMulticardInternal> {
        val oldRows: MutableList<BannerMulticardInternal> = mutableListOf()
        val newRows: MutableList<BannerMulticardInternal> = mutableListOf()

        bannersChangesCollection
            .filter { ac -> ac.changed(BannerWithMulticardSetMulticards.MULTICARDS) }
            .map { ac ->
                val oldBannerRows = ac.getOldValue(BannerWithMulticardSetMulticards.MULTICARDS) ?: listOf()
                val newBannerRows = ac.getNewValue(BannerWithMulticardSetMulticards.MULTICARDS) ?: listOf()

                oldRows += toInternalMulticards(ac.model.id, oldBannerRows)
                newRows += toInternalMulticards(ac.model.id, newBannerRows)
            }

        val rowsWithoutId = newRows.filter { it.multicardId == null }
        enrichWithCardId(rowsWithoutId)

        return AddOrUpdateAndDeleteByKeyContainer
            .builder(BannerMulticardInternal::getMulticardId)
            .skipNewRowsByEquals()
            .addDiff(oldRows, newRows)
            .build()
    }

    override fun addOrUpdate(
        context: DSLContext,
        repositoryModelsToAddOrUpdate: Collection<BannerMulticardInternal>
    ) {
        InsertHelper(context, BANNER_MULTICARDS)
            .addAll(bannerMulticardMapper, repositoryModelsToAddOrUpdate)
            .onDuplicateKeyUpdate()
            .set(BANNER_MULTICARDS.ORDER_NUM, MySQLDSL.values(BANNER_MULTICARDS.ORDER_NUM))
            .set(BANNER_MULTICARDS.TEXT, MySQLDSL.values(BANNER_MULTICARDS.TEXT))
            .set(BANNER_MULTICARDS.IMAGE_HASH, MySQLDSL.values(BANNER_MULTICARDS.IMAGE_HASH))
            .set(BANNER_MULTICARDS.HREF, MySQLDSL.values(BANNER_MULTICARDS.HREF))
            .set(BANNER_MULTICARDS.PRICE, MySQLDSL.values(BANNER_MULTICARDS.PRICE))
            .set(BANNER_MULTICARDS.PRICE_OLD, MySQLDSL.values(BANNER_MULTICARDS.PRICE_OLD))
            .set(BANNER_MULTICARDS.CURRENCY, MySQLDSL.values(BANNER_MULTICARDS.CURRENCY))
            .executeIfRecordsAdded()
    }

    override fun delete(context: DSLContext, repositoryModelsToDelete: Collection<BannerMulticardInternal>) {
        val cardIds = repositoryModelsToDelete.map { it.multicardId }
        context.deleteFrom(BANNER_MULTICARDS)
            .where(BANNER_MULTICARDS.MULTICARD_ID.`in`(cardIds))
            .execute()
    }

    private fun toInternalMulticards(bid: Long, multicards: List<BannerMulticard>): List<BannerMulticardInternal> {
        return multicards.mapIndexed { index, multicard ->
            BannerMulticardInternal()
                .withMulticardId(multicard.multicardId)
                .withBid(bid)
                .withOrder(index.toLong())
                .withText(multicard.text)
                .withImageHash(multicard.imageHash)
                .withHref(multicard.href)
                .withPrice(multicard.price)
                .withPriceOld(multicard.priceOld)
                .withCurrency(multicard.currency)
        }
    }

    private fun fromInternalMulticards(multicards: List<BannerMulticardInternal>): List<BannerMulticard> {
        return multicards.map {
            BannerMulticard()
                .withMulticardId(it.multicardId)
                .withText(it.text)
                .withImageHash(it.imageHash)
                .withHref(it.href)
                .withPrice(it.price)
                .withPriceOld(it.priceOld)
                .withCurrency(it.currency)
        }
    }
}
