package ru.yandex.direct.core.entity.banner.type.multicard

import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.banner.container.BannersUpdateOperationContainer
import ru.yandex.direct.core.entity.banner.model.BannerMulticard
import ru.yandex.direct.core.entity.banner.model.BannerWithMulticardSet
import ru.yandex.direct.core.entity.banner.service.validation.type.update.AbstractBannerUpdateValidationTypeSupport
import ru.yandex.direct.model.ModelChanges
import ru.yandex.direct.validation.builder.Validator
import ru.yandex.direct.validation.builder.When
import ru.yandex.direct.validation.constraint.CommonConstraints.inSet
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.direct.validation.util.item
import ru.yandex.direct.validation.util.validateList
import ru.yandex.direct.validation.util.validateModel
import ru.yandex.direct.validation.util.validateModelChanges

@Component
class BannerWithMulticardSetUpdateValidationTypeSupport(
    private val bannerWithMulticardSetValidatorProvider: BannerWithMulticardSetValidatorProvider,
) : AbstractBannerUpdateValidationTypeSupport<BannerWithMulticardSet>() {
    override fun getTypeClass() = BannerWithMulticardSet::class.java

    override fun validateBeforeApply(
        container: BannersUpdateOperationContainer,
        vr: ValidationResult<List<ModelChanges<BannerWithMulticardSet>>, Defect<*>>,
        unmodifiedModels: Map<Long, BannerWithMulticardSet>
    ): ValidationResult<List<ModelChanges<BannerWithMulticardSet>>, Defect<*>> {
        return validateList(vr) {
            checkEachBy { banner ->
                validateExistingIds(banner, unmodifiedModels[banner.id])
            }
        }
    }

    private fun validateExistingIds(
        mc: ModelChanges<BannerWithMulticardSet>,
        oldBanner: BannerWithMulticardSet?
    ): ValidationResult<ModelChanges<BannerWithMulticardSet>, Defect<*>> {
        if (oldBanner == null) {
            return ValidationResult.success(mc)
        }

        val oldMulticardIds: Set<Long> = oldBanner.multicards
            ?.map { it.multicardId }?.toSet()
            .orEmpty()

        return validateModelChanges(mc) {
            item(BannerWithMulticardSet.MULTICARDS) {
                checkBy(bannerMulticardsValidator(oldMulticardIds), When.notNull())
            }
        }
    }

    private fun bannerMulticardsValidator(oldMulticardIds: Set<Long>) =
        Validator { multicards: List<BannerMulticard> ->
            validateList(multicards) {
                checkEachBy { multicard ->
                    validateModel(multicard) {
                        item(BannerMulticard.MULTICARD_ID) {
                            check(inSet(oldMulticardIds))
                        }
                    }
                }
            }
        }

    override fun validate(
        container: BannersUpdateOperationContainer,
        vr: ValidationResult<List<BannerWithMulticardSet>, Defect<*>>
    ): ValidationResult<List<BannerWithMulticardSet>, Defect<*>> {
        return validateList(vr) {
            checkEachBy(
                bannerWithMulticardSetValidatorProvider.bannerWithMulticardValidator(container, vr.value)
            )
        }
    }
}
