package ru.yandex.direct.core.entity.banner.type.multicard

import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType
import ru.yandex.direct.core.entity.banner.container.BannersOperationContainer
import ru.yandex.direct.core.entity.banner.model.BannerMulticard
import ru.yandex.direct.core.entity.banner.model.BannerWithMulticardSet
import ru.yandex.direct.core.entity.banner.model.CpmBanner
import ru.yandex.direct.core.entity.banner.model.ImageType
import ru.yandex.direct.core.entity.banner.model.TextBanner
import ru.yandex.direct.core.entity.banner.service.validation.BannerImageHashValidator
import ru.yandex.direct.core.entity.banner.service.validation.BannerTextValidator
import ru.yandex.direct.core.entity.banner.type.href.BannerHrefValidator
import ru.yandex.direct.core.entity.banner.type.multicard.BannerWithMulticardSetConstants.MAX_CPM_BANNER_MULTICARDS_COUNT
import ru.yandex.direct.core.entity.banner.type.multicard.BannerWithMulticardSetConstants.MAX_CPM_BANNER_TEXT_LENGTH
import ru.yandex.direct.core.entity.banner.type.multicard.BannerWithMulticardSetConstants.MAX_TEXT_BANNER_MULTICARDS_COUNT
import ru.yandex.direct.core.entity.banner.type.multicard.BannerWithMulticardSetConstants.MAX_TEXT_BANNER_TEXT_LENGTH
import ru.yandex.direct.core.entity.banner.type.multicard.BannerWithMulticardSetConstants.MAX_WORD_LENGTH
import ru.yandex.direct.core.entity.banner.type.multicard.BannerWithMulticardSetConstants.MIN_CPM_BANNER_MULTICARDS_COUNT
import ru.yandex.direct.core.entity.banner.type.multicard.BannerWithMulticardSetConstants.MIN_TEXT_BANNER_MULTICARDS_COUNT
import ru.yandex.direct.core.entity.image.repository.BannerImageFormatRepository
import ru.yandex.direct.feature.FeatureName
import ru.yandex.direct.validation.builder.Constraint.fromPredicate
import ru.yandex.direct.validation.builder.Validator
import ru.yandex.direct.validation.builder.When
import ru.yandex.direct.validation.constraint.CollectionConstraints.listSize
import ru.yandex.direct.validation.constraint.CollectionConstraints.unique
import ru.yandex.direct.validation.constraint.CommonConstraints
import ru.yandex.direct.validation.defect.CommonDefects
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.util.item
import ru.yandex.direct.validation.util.list
import ru.yandex.direct.validation.util.validateList
import ru.yandex.direct.validation.util.validateModel

@Component
class BannerWithMulticardSetValidatorProvider(
    private val bannerImageFormatRepository: BannerImageFormatRepository,
) {
    private val textBannerTextValidator = BannerTextValidator
        .builder(MAX_TEXT_BANNER_TEXT_LENGTH, MAX_WORD_LENGTH)
        .withTemplateDisallowed()
        .build()

    private val cpmBannerTextValidator = BannerTextValidator
        .builder(MAX_CPM_BANNER_TEXT_LENGTH, MAX_WORD_LENGTH)
        .withTemplateDisallowed()
        .build()

    fun bannerWithMulticardValidator(
        container: BannersOperationContainer,
        banners: List<BannerWithMulticardSet>,
    ): Validator<BannerWithMulticardSet, Defect<*>> {
        val isTgoMulticardEnabled = container.isFeatureEnabledForClient(FeatureName.ASSET_TGO_MULTICARD)
        val isCpmVideoMulticardEnabled = container.isFeatureEnabledForClient(FeatureName.ASSET_CPM_VIDEO_MULTICARD)

        val allMulticards = banners.flatMap { it.multicards ?: listOf() }
        val imageHashes = allMulticards.map { it.imageHash }.toSet()

        val imageTypeByHash = bannerImageFormatRepository
            .getExistingImageHashesWithType(container.shard, container.clientId, imageHashes)

        return Validator { banner ->
            val isMulticardSetAllowed = when (banner) {
                is TextBanner -> isTgoMulticardEnabled
                is CpmBanner -> isCpmVideoMulticardEnabled
                        && container.getAdGroup(banner)?.type == AdGroupType.CPM_VIDEO
                else -> false
            }

            validateModel(banner) {
                list(BannerWithMulticardSet.MULTICARDS) {
                    check(fromPredicate({ isMulticardSetAllowed }, CommonDefects.isNull()))

                    checkBy(multicardCountValidator(banner), When.isValid())
                    checkEach(unique(BannerMulticard::getMulticardId), When.isValid())
                    checkEachBy(bannerMulticardValidator(banner, imageTypeByHash), When.isValid())
                }
            }
        }
    }

    private fun multicardCountValidator(banner: BannerWithMulticardSet) =
        Validator { multicards: List<BannerMulticard>? ->
            validateList(multicards) {
                when (banner) {
                    is TextBanner ->
                        check(listSize(MIN_TEXT_BANNER_MULTICARDS_COUNT, MAX_TEXT_BANNER_MULTICARDS_COUNT), When.isValid())
                    is CpmBanner ->
                        check(listSize(MIN_CPM_BANNER_MULTICARDS_COUNT, MAX_CPM_BANNER_MULTICARDS_COUNT), When.isValid())
                }
            }
        }

    private fun bannerMulticardValidator(banner: BannerWithMulticardSet, imageTypes: Map<String, ImageType>) =
        Validator { multicard: BannerMulticard ->
            validateModel(multicard) {
                item(BannerMulticard.TEXT) {
                    when (banner) {
                        is TextBanner -> checkBy(textBannerTextValidator, When.notNull())
                        is CpmBanner -> checkBy(cpmBannerTextValidator)
                    }
                }
                item(BannerMulticard.IMAGE_HASH) {
                    check(CommonConstraints.notNull())
                    checkBy(BannerImageHashValidator(imageTypes, true, false))
                }
                item(BannerMulticard.HREF) {
                    when (banner) {
                        is TextBanner -> check(CommonConstraints.isNull())
                        is CpmBanner -> {
                            check(CommonConstraints.notNull())
                            checkBy(BannerHrefValidator())
                        }
                    }
                }
                item(BannerMulticard.PRICE) {
                    when (banner) {
                        is TextBanner -> check(CommonConstraints.isNull())
                        is CpmBanner -> {
                            check(CommonConstraints.notNull(), When.isTrue(multicard.currency != null))
                        }
                    }
                }
                item(BannerMulticard.PRICE_OLD) {
                    when (banner) {
                        is TextBanner -> check(CommonConstraints.isNull())
                        is CpmBanner -> {
                            check(CommonConstraints.isNull(), When.isTrue(multicard.price == null))
                        }
                    }
                }
                item(BannerMulticard.CURRENCY) {
                    when (banner) {
                        is TextBanner -> check(CommonConstraints.isNull())
                        is CpmBanner -> {
                            check(CommonConstraints.notNull(), When.isTrue(multicard.price != null))
                        }
                    }
                }
            }
        }
}
