package ru.yandex.direct.core.entity.banner.type.organization;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import com.google.common.collect.Sets;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jooq.DSLContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.stereotype.Component;

import ru.yandex.altay.model.language.LanguageOuterClass;
import ru.yandex.direct.core.entity.banner.container.BannersAddOperationContainer;
import ru.yandex.direct.core.entity.banner.model.BannerWithOrganization;
import ru.yandex.direct.core.entity.banner.model.BannerWithOrganizationAndVcard;
import ru.yandex.direct.core.entity.banner.repository.BannerRelationsRepository;
import ru.yandex.direct.core.entity.banner.service.type.add.AbstractBannerAddOperationTypeSupport;
import ru.yandex.direct.core.entity.campaign.model.MetrikaCounter;
import ru.yandex.direct.core.entity.campaign.repository.CampMetrikaCountersRepository;
import ru.yandex.direct.core.entity.campaign.service.CampMetrikaCountersService;
import ru.yandex.direct.core.entity.organizations.OrganizationsClientTranslatableException;
import ru.yandex.direct.core.entity.organizations.repository.OrganizationRepository;
import ru.yandex.direct.core.entity.organizations.service.OrganizationService;
import ru.yandex.direct.feature.FeatureName;

import static java.util.Collections.emptySet;
import static ru.yandex.direct.core.entity.organizations.util.PermalinkUtils.newCalculatePreferVCardOverPermalink;
import static ru.yandex.direct.feature.FeatureName.ADDING_ORGANIZATIONS_COUNTERS_TO_CAMPAIGN_ON_ADDING_ORGANIZATIONS_TO_ADS;
import static ru.yandex.direct.feature.FeatureName.IS_ENABLE_PREFER_V_CARD_OVER_PERMALINK;
import static ru.yandex.direct.organizations.swagger.OrganizationsClient.getLanguageByName;
import static ru.yandex.direct.utils.FunctionalUtils.flatMap;
import static ru.yandex.direct.utils.FunctionalUtils.flatMapToSet;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
public class BannerWithOrganizationAddOperationTypeSupport
        extends AbstractBannerAddOperationTypeSupport<BannerWithOrganization> {

    private static final Logger logger =
            LoggerFactory.getLogger(BannerWithOrganizationAddOperationTypeSupport.class);

    private final OrganizationRepository organizationRepository;
    private final BannerRelationsRepository bannerRelationsRepository;
    private final CampMetrikaCountersRepository campMetrikaCountersRepository;
    private final OrganizationService organizationService;
    private final CampMetrikaCountersService campMetrikaCountersService;

    @Autowired
    public BannerWithOrganizationAddOperationTypeSupport(OrganizationRepository organizationRepository,
                                                         BannerRelationsRepository bannerRelationsRepository,
                                                         CampMetrikaCountersRepository campMetrikaCountersRepository,
                                                         OrganizationService organizationService,
                                                         CampMetrikaCountersService campMetrikaCountersService) {
        this.organizationRepository = organizationRepository;
        this.bannerRelationsRepository = bannerRelationsRepository;
        this.campMetrikaCountersRepository = campMetrikaCountersRepository;
        this.organizationService = organizationService;
        this.campMetrikaCountersService = campMetrikaCountersService;
    }

    @Override
    public Class<BannerWithOrganization> getTypeClass() {
        return BannerWithOrganization.class;
    }

    @Override
    public void beforeExecution(BannersAddOperationContainer addContainer, List<BannerWithOrganization> models) {
        var featureEnabled = addContainer.isFeatureEnabledForClient(IS_ENABLE_PREFER_V_CARD_OVER_PERMALINK);
        models.stream()
                .filter(b -> b.getPermalinkId() != null)
                .forEach(b -> {
                    if (featureEnabled && b instanceof BannerWithOrganizationAndVcard) {
                        b.setPreferVCardOverPermalink(newCalculatePreferVCardOverPermalink(
                                (BannerWithOrganizationAndVcard) b));
                    } else {
                        b.setPreferVCardOverPermalink(false);
                    }
                });
    }

    @Override
    public void updateRelatedEntitiesInTransaction(
            DSLContext dslContext,
            BannersAddOperationContainer addModelContainer,
            List<BannerWithOrganization> models) {
        addOrganizationCountersToBannersCampaigns(dslContext, addModelContainer, models);
        organizationRepository.addOrUpdateOrganizations(dslContext,
                addModelContainer.getClientOrganizations().values());
    }

    /**
     * При создании баннера с организацией добавляем к кампании счетчик организации
     */
    public void addOrganizationCountersToBannersCampaigns(
            DSLContext dslContext,
            BannersAddOperationContainer addModelContainer,
            Collection<BannerWithOrganization> banners) {
        Set<Long> enabledAddingOrganizationCounterToCampaignByBannerId =
                getBannerIdsWithEnabledAddingOrganizationCounterToCampaign(
                        mapList(banners, BannerWithOrganization::getId),
                        addModelContainer);

        List<BannerWithOrganization> bannersWithEnabledOrganizationCountersAdding = StreamEx.of(banners)
                .mapToEntry(BannerWithOrganization::getId)
                .filterValues(enabledAddingOrganizationCounterToCampaignByBannerId::contains)
                .keys()
                .toList();

        if (bannersWithEnabledOrganizationCountersAdding.isEmpty()) {
            return;
        }

        Set<Long> changingCampaignIds = listToSet(bannersWithEnabledOrganizationCountersAdding,
                banner -> addModelContainer.getCampaign(banner).getId());

        Map<Long, Set<Long>> addingPermalinkIdsByCampaignId = StreamEx.of(bannersWithEnabledOrganizationCountersAdding)
                .mapToEntry(banner -> addModelContainer.getCampaign(banner).getId(),
                        BannerWithOrganization::getPermalinkId)
                .nonNullValues()
                .sortedBy(Map.Entry::getKey)
                .collapseKeys()
                .mapValues(Set::copyOf)
                .toMap();

        List<Long> addingPermalinkIds = flatMap(addingPermalinkIdsByCampaignId.values(), Function.identity());

        Map<Long, Set<Long>> currentPermalinkIdsByCampaignId = getCurrentPermalinksByCampaignId(dslContext,
                changingCampaignIds);

        Set<Long> currentPermalinkIds = flatMapToSet(currentPermalinkIdsByCampaignId.values(), Function.identity());

        Set<Long> newPermalinkIds = StreamEx.of(currentPermalinkIds)
                .append(addingPermalinkIds)
                .toSet();

        Map<Long, Long> counterIdByPermalinkId;
        try {
            counterIdByPermalinkId = organizationService.getMetrikaCountersByOrganizationsIds(
                    getLanguageByName(LocaleContextHolder.getLocale().getLanguage()).orElse(LanguageOuterClass.Language.EN),
                    newPermalinkIds);
        } catch (OrganizationsClientTranslatableException e) {
            // игнорируем обновление счетчиков при недоступности Справочника при отвязке организации
            if (StreamEx.of(bannersWithEnabledOrganizationCountersAdding).allMatch(c -> c.getPermalinkId() == null)) {
                logger.warn("Do not update organization counters for campaigns due to organization client problem");
                return;
            }
            throw e;
        }

        Map<Long, Set<Long>> currentPermalinkCounterIdsByCampaignId = StreamEx.of(changingCampaignIds)
                .mapToEntry(cid -> Sets.union(addingPermalinkIdsByCampaignId.getOrDefault(cid, emptySet()),
                        currentPermalinkIdsByCampaignId.getOrDefault(cid, emptySet())))
                .mapValues(campaignPermalinkIds -> StreamEx.of(campaignPermalinkIds)
                        .map(counterIdByPermalinkId::get)
                        .nonNull()
                        .toSet())
                .toMap();

        Map<Long, List<MetrikaCounter>> actualMetrikaCounterByCampaignId = campMetrikaCountersRepository
                .getMetrikaCounterByCid(dslContext, changingCampaignIds);

        Map<Long, Set<Long>> actualMetrikaCounterIdsByCampaignId = EntryStream.of(actualMetrikaCounterByCampaignId)
                .mapValues(counters -> listToSet(counters,
                        MetrikaCounter::getId))
                .toMap();

        Map<Long, Set<Long>> newMetrikaCounterIdsByCampaignId = StreamEx.of(changingCampaignIds)
                .mapToEntry(cid -> getNewMetrikaCounterIds(
                        actualMetrikaCounterIdsByCampaignId.getOrDefault(cid, emptySet()),
                        emptySet(),
                        currentPermalinkCounterIdsByCampaignId.getOrDefault(cid, emptySet())))
                .toMap();

        Map<Long, List<MetrikaCounter>> countersByCampaignId =
                CampMetrikaCountersService.
                        calculateCountersByCampaignIdForNewMetrikaCounterIds(newMetrikaCounterIdsByCampaignId,
                                actualMetrikaCounterByCampaignId,
                                Set.copyOf(counterIdByPermalinkId.values()));

        if (addModelContainer.isFeatureEnabledForClient(FeatureName.TOGETHER_UPDATING_STRATEGY_AND_CAMPAIGN_METRIKA_COUNTERS)) {

            campMetrikaCountersService.updateMetrikaCounterForStrategies(
                    addModelContainer.getShard(),
                    addModelContainer.getClientId(),
                    countersByCampaignId,
                    List.copyOf(countersByCampaignId.keySet()),
                    EntryStream.of(newMetrikaCounterIdsByCampaignId)
                            .mapValues(List::copyOf)
                            .toMap()
            );

        } else {
            campMetrikaCountersRepository.updateMetrikaCounters(
                    dslContext,
                    countersByCampaignId);
        }
    }

    /**
     * Получить для присланные id баннеров, доступна ли фича для их клиента, по которой включается автоматическое
     * добавление счетчиков метрики организаций
     */
    public Set<Long> getBannerIdsWithEnabledAddingOrganizationCounterToCampaign(
            List<Long> changedBannerIds,
            BannersAddOperationContainer container) {
        return getBannerIdsForClientsWithFeature(
                ADDING_ORGANIZATIONS_COUNTERS_TO_CAMPAIGN_ON_ADDING_ORGANIZATIONS_TO_ADS,
                changedBannerIds,
                container);
    }

    public Set<Long> getBannerIdsForClientsWithFeature(FeatureName featureName, List<Long> changedBannerIds,
                                                       BannersAddOperationContainer container) {
        if (container.isFeatureEnabledForClient(featureName)) {
            return new HashSet<>(changedBannerIds);
        } else {
            return Set.of();
        }
    }

    private Map<Long, Set<Long>> getCurrentPermalinksByCampaignId(DSLContext dslContext,
                                                                  Set<Long> campaignIds) {
        Map<Long, List<Long>> allBannerIdsByUpdatingCampaignIds =
                bannerRelationsRepository.getBannerIdsByCampaignIdsMap(dslContext, campaignIds);

        List<Long> updatingCampaignsBannerIds = flatMap(allBannerIdsByUpdatingCampaignIds.values(),
                Function.identity());

        Map<Long, Long> permalinkIdByBannerId = organizationRepository.getPermalinkIdsByBannerIds(dslContext,
                updatingCampaignsBannerIds);

        return EntryStream.of(allBannerIdsByUpdatingCampaignIds)
                .mapValues(campaignBannerIds -> StreamEx.of(campaignBannerIds)
                        .map(permalinkIdByBannerId::get)
                        .nonNull()
                        .toSet())
                .toMap();
    }

    private static Set<Long> getNewMetrikaCounterIds(Set<Long> actualCounters, Set<Long> deletingCounters,
                                                     Set<Long> addingCounters) {
        return StreamEx.of(actualCounters)
                .remove(deletingCounters::contains)
                .append(addingCounters)
                .toSet();
    }

}
