package ru.yandex.direct.core.entity.banner.type.organization;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jetbrains.annotations.NotNull;
import org.jooq.DSLContext;
import org.jooq.JoinType;
import org.jooq.util.mysql.MySQLDSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.banner.container.BannerRepositoryContainer;
import ru.yandex.direct.core.entity.banner.container.BannersOperationContainer;
import ru.yandex.direct.core.entity.banner.model.BannerWithOrganization;
import ru.yandex.direct.core.entity.banner.repository.type.AbstractFlatRelatedEntityUpsertRepositoryTypeSupport;
import ru.yandex.direct.core.entity.banner.repository.type.ModifiedPaths;
import ru.yandex.direct.dbschema.ppc.tables.records.BannerPermalinksRecord;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapper;
import ru.yandex.direct.jooqmapper.JooqMapperBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.multitype.entity.JoinQuery;
import ru.yandex.grut.objects.proto.client.Schema;

import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.booleanProperty;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_PERMALINKS;
import static ru.yandex.direct.dbschema.ppc.enums.BannerPermalinksPermalinkAssignType.manual;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.jooqmapper.write.WriterBuilders.fromProperty;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

@Component
@ParametersAreNonnullByDefault
public class BannerWithOrganizationRepositoryTypeSupport
        extends AbstractFlatRelatedEntityUpsertRepositoryTypeSupport
        <BannerWithOrganization, BannerPermalinksRecord> {

    private static final Set<ModelProperty<? super BannerWithOrganization, ?>> NEW_BANNER_ORGANIZATION_PROPERTIES =
            Set.of(BannerWithOrganization.PREFER_V_CARD_OVER_PERMALINK);

    private static final Set<ModelProperty<? super BannerWithOrganization, ?>> GRUT_PROPERTIES =
            Set.of(
                    BannerWithOrganization.PERMALINK_ID,
                    BannerWithOrganization.PREFER_V_CARD_OVER_PERMALINK
            );

    @Autowired
    public BannerWithOrganizationRepositoryTypeSupport(DslContextProvider dslContextProvider) {
        super(dslContextProvider, BANNER_PERMALINKS.BID, createMapper());
    }

    @Override
    protected void upsertEntity(DSLContext context, Collection<BannerWithOrganization> banners) {
        var helper = new InsertHelper<>(context, BANNER_PERMALINKS);
        banners.forEach(b -> {
            helper.add(getJooqMapper(), b);
            helper.set(BANNER_PERMALINKS.PERMALINK_ASSIGN_TYPE, manual);
            helper.newRecord();
        });

        helper.onDuplicateKeyUpdate()
                .set(BANNER_PERMALINKS.PERMALINK_ASSIGN_TYPE, MySQLDSL.values(BANNER_PERMALINKS.PERMALINK_ASSIGN_TYPE))
                .set(BANNER_PERMALINKS.PREFER_VCARD_OVER_PERMALINK,
                        MySQLDSL.values(BANNER_PERMALINKS.PREFER_VCARD_OVER_PERMALINK))
                // Для типа manual поддерживаем значение поля is_change_to_manual_rejected равным 0.
                // Явно проставляем полю 0, т.к. в нём может храниться 1 в случае ручного добавления
                // той организации, которая ранее была добавлена автоматически и отклонена.
                .set(BANNER_PERMALINKS.IS_CHANGE_TO_MANUAL_REJECTED, 0L);

        helper.executeIfRecordsAdded();
    }

    @Override
    protected void deleteEntities(DSLContext context, BannerRepositoryContainer updateParameters,
                                  Collection<AppliedChanges<BannerWithOrganization>> appliedChanges) {
        Set<Long> bannerIds = listToSet(appliedChanges, ac -> ac.getModel().getId());

        context
                .deleteFrom(BANNER_PERMALINKS)
                .where(BANNER_PERMALINKS.BID.in(bannerIds))
                .and(BANNER_PERMALINKS.PERMALINK_ASSIGN_TYPE.eq(manual))
                .execute();
    }

    @Override
    protected boolean isAddEntity(BannerWithOrganization model) {
        return model.getPermalinkId() != null;
    }

    @Override
    protected boolean isUpsertEntity(AppliedChanges<BannerWithOrganization> ac) {
        return ac.changedAndNotDeleted(BannerWithOrganization.PERMALINK_ID)
                || (ac.getNewValue(BannerWithOrganization.PERMALINK_ID) != null
                && NEW_BANNER_ORGANIZATION_PROPERTIES.stream().anyMatch(ac::changed));
    }

    @Override
    protected boolean isDeleteEntity(AppliedChanges<BannerWithOrganization> appliedChange) {
        // Если пермалинк поменялся, то старый (если он есть) нужно удалить, а новый (если он есть) будет добавлен в
        // isUpsertEntity. В этом репозитории если баннер changedAndNotDeleted - он попадёт сразу и в isUpsertEntity
        // и в isDeleteEntity.
        return appliedChange.changed(BannerWithOrganization.PERMALINK_ID) &&
                !appliedChange.assigned(BannerWithOrganization.PERMALINK_ID);
    }

    @Override
    public List<JoinQuery> joinQuery() {
        return List.of(new JoinQuery(BANNER_PERMALINKS,
                JoinType.LEFT_OUTER_JOIN,
                BANNER_PERMALINKS.BID.eq(BANNERS.BID)
                        .and(BANNER_PERMALINKS.PERMALINK_ASSIGN_TYPE.eq(manual))));
    }

    @Override
    public Class<BannerWithOrganization> getTypeClass() {
        return BannerWithOrganization.class;
    }

    private static JooqMapper<BannerWithOrganization> createMapper() {
        return JooqMapperBuilder.<BannerWithOrganization>builder()
                .writeField(BANNER_PERMALINKS.BID, fromProperty(BannerWithOrganization.ID))
                .map(property(BannerWithOrganization.PERMALINK_ID, BANNER_PERMALINKS.PERMALINK))
                .map(booleanProperty(BannerWithOrganization.PREFER_V_CARD_OVER_PERMALINK,
                        BANNER_PERMALINKS.PREFER_VCARD_OVER_PERMALINK))
                .build();
    }

    @Override
    public Set<ModelProperty<? super BannerWithOrganization, ?>> getGrutSupportedProperties() {
        return GRUT_PROPERTIES;
    }

    @Override
    public Map<Long, ModifiedPaths> applyToGrutObjects(
            @NotNull Map<Long, Schema.TBannerV2.Builder> bannerBuilders,
            @NotNull Collection<AppliedChanges<BannerWithOrganization>> appliedChangesList,
            @NotNull BannersOperationContainer operationContainer) {
        Map<Long, ModifiedPaths> modifiedPathsMap = new HashMap<>();
        for (var appliedChanges : appliedChangesList) {
            Set<String> setPaths = new HashSet<>();
            Set<String> removePaths = new HashSet<>();
            Long id = appliedChanges.getModel().getId();
            if (appliedChanges.getPropertiesForUpdate().contains(BannerWithOrganization.PERMALINK_ID)) {
                var newValue = appliedChanges.getNewValue(BannerWithOrganization.PERMALINK_ID);
                if (newValue != null) {
                    Schema.TBannerV2.Builder bannerBuilder = bannerBuilders.get(id);
                    bannerBuilder.getSpecBuilder().getPermalinkBuilder().setId(newValue);
                    setPaths.add("/spec/permalink/id");
                } else {
                    removePaths.add("/spec/permalink/id");
                }
            }
            if (appliedChanges.getPropertiesForUpdate().contains(BannerWithOrganization.PREFER_V_CARD_OVER_PERMALINK)) {
                var newValue = appliedChanges.getNewValue(BannerWithOrganization.PREFER_V_CARD_OVER_PERMALINK);
                if (newValue != null) {
                    Schema.TBannerV2.Builder bannerBuilder = bannerBuilders.get(id);
                    bannerBuilder.getSpecBuilder().getPermalinkBuilder().setPreferVcardOverPermalink(newValue);
                    setPaths.add("/spec/permalink/prefer_vcard_over_permalink");
                } else {
                    removePaths.add("/spec/permalink/prefer_vcard_over_permalink");
                }
            }
            if (!setPaths.isEmpty() || !removePaths.isEmpty()) {
                modifiedPathsMap.put(id, new ModifiedPaths(setPaths, removePaths));
            }
        }
        return modifiedPathsMap;
    }
}
