package ru.yandex.direct.core.entity.banner.type.organization;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jooq.DSLContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.stereotype.Component;

import ru.yandex.altay.model.language.LanguageOuterClass;
import ru.yandex.direct.core.entity.banner.container.BannersModerationContainer;
import ru.yandex.direct.core.entity.banner.container.BannersUpdateOperationContainer;
import ru.yandex.direct.core.entity.banner.model.BannerWithOrganization;
import ru.yandex.direct.core.entity.banner.model.BannerWithOrganizationAndVcard;
import ru.yandex.direct.core.entity.banner.repository.BannerRelationsRepository;
import ru.yandex.direct.core.entity.banner.service.type.update.AbstractBannerUpdateOperationTypeSupport;
import ru.yandex.direct.core.entity.campaign.AvailableCampaignSources;
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign;
import ru.yandex.direct.core.entity.campaign.model.MetrikaCounter;
import ru.yandex.direct.core.entity.campaign.repository.CampMetrikaCountersRepository;
import ru.yandex.direct.core.entity.campaign.service.CampMetrikaCountersService;
import ru.yandex.direct.core.entity.organizations.OrganizationsClientTranslatableException;
import ru.yandex.direct.core.entity.organizations.repository.OrganizationRepository;
import ru.yandex.direct.core.entity.organizations.service.OrganizationService;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.model.AppliedChanges;

import static java.util.Collections.emptySet;
import static ru.yandex.direct.core.entity.banner.model.BannerWithOrganization.PERMALINK_ID;
import static ru.yandex.direct.core.entity.banner.model.BannerWithOrganization.PREFER_V_CARD_OVER_PERMALINK;
import static ru.yandex.direct.feature.FeatureName.ADDING_ORGANIZATIONS_COUNTERS_TO_CAMPAIGN_ON_ADDING_ORGANIZATIONS_TO_ADS;
import static ru.yandex.direct.organizations.swagger.OrganizationsClient.getLanguageByName;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.flatMap;
import static ru.yandex.direct.utils.FunctionalUtils.flatMapToSet;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

@Component
public class BannerWithOrganizationUpdateOperationTypeSupport
        extends AbstractBannerUpdateOperationTypeSupport<BannerWithOrganization> {
    private static final Logger logger =
            LoggerFactory.getLogger(BannerWithOrganizationUpdateOperationTypeSupport.class);

    private final OrganizationRepository organizationRepository;
    private final BannerRelationsRepository bannerRelationsRepository;
    private final CampMetrikaCountersRepository campMetrikaCountersRepository;
    private final CampMetrikaCountersService campMetrikaCountersService;
    private final OrganizationService organizationService;

    @Autowired
    public BannerWithOrganizationUpdateOperationTypeSupport(
            OrganizationRepository organizationRepository,
            BannerRelationsRepository bannerRelationsRepository,
            CampMetrikaCountersRepository campMetrikaCountersRepository,
            CampMetrikaCountersService campMetrikaCountersService,
            OrganizationService organizationService) {
        this.organizationRepository = organizationRepository;
        this.bannerRelationsRepository = bannerRelationsRepository;
        this.campMetrikaCountersRepository = campMetrikaCountersRepository;
        this.campMetrikaCountersService = campMetrikaCountersService;
        this.organizationService = organizationService;
    }

    @Override
    public Class<BannerWithOrganization> getTypeClass() {
        return BannerWithOrganization.class;
    }

    @Override
    public void beforeExecution(BannersUpdateOperationContainer updateContainer,
                                List<AppliedChanges<BannerWithOrganization>> appliedChanges) {
        appliedChanges.stream()
                .filter(ac -> ac.getNewValue(PERMALINK_ID) != null)
                .forEach(ac -> {
                    if (!(ac.getModel() instanceof BannerWithOrganizationAndVcard)) {
                        ac.modify(PREFER_V_CARD_OVER_PERMALINK, false);
                    }
                });
    }

    @Override
    public boolean needModeration(BannersModerationContainer container,
                                  AppliedChanges<BannerWithOrganization> appliedChanges) {
        // Не отправляем на модерацию при отрыве от организации
        CommonCampaign campaign = container.getCampaign(appliedChanges.getModel());
        if (AvailableCampaignSources.INSTANCE.isUC(campaign.getSource())
                && appliedChanges.deleted(BannerWithOrganization.PERMALINK_ID)) {
            return false;
        }

        return appliedChanges.changed(BannerWithOrganization.PERMALINK_ID)
                || appliedChanges.changed(PREFER_V_CARD_OVER_PERMALINK);
    }

    @Override
    public boolean needBsResync(AppliedChanges<BannerWithOrganization> appliedChanges) {
        return appliedChanges.changed(BannerWithOrganization.PERMALINK_ID)
                || appliedChanges.changed(PREFER_V_CARD_OVER_PERMALINK);
    }

    @Override
    public boolean needLastChangeReset(AppliedChanges<BannerWithOrganization> appliedChanges) {
        return appliedChanges.changed(BannerWithOrganization.PERMALINK_ID)
                || appliedChanges.changed(PREFER_V_CARD_OVER_PERMALINK);
    }

    @Override
    public void updateRelatedEntitiesInTransaction(DSLContext dslContext,
                                                   BannersUpdateOperationContainer updateContainer,
                                                   List<AppliedChanges<BannerWithOrganization>> appliedChanges) {
        var featureEnabled = updateContainer.isFeatureEnabledForClient(
                ADDING_ORGANIZATIONS_COUNTERS_TO_CAMPAIGN_ON_ADDING_ORGANIZATIONS_TO_ADS);
        if (featureEnabled) {
            updateOrgaizationsCountersForBannersCampaigns(dslContext, appliedChanges, updateContainer);
        }
        organizationRepository.addOrUpdateOrganizations(dslContext,
                updateContainer.getClientOrganizations().values());
    }

    /**
     * При изменении организации у баннера обновляем счетчики на кампании
     */
    private void updateOrgaizationsCountersForBannersCampaigns(
            DSLContext dslContext,
            List<AppliedChanges<BannerWithOrganization>> appliedChanges,
            BannersUpdateOperationContainer updateContainer) {
        List<AppliedChanges<BannerWithOrganization>> bannerChangesShouldUpdateCampaignCounters =
                filterList(appliedChanges, this::shouldUpdateCampaignCounter);

        if (bannerChangesShouldUpdateCampaignCounters.isEmpty()) {
            return;
        }

        Set<Long> changingCampaignIds = listToSet(bannerChangesShouldUpdateCampaignCounters,
                ac -> updateContainer.getCampaign(ac.getModel()).getId());

        Map<Long, Set<Long>> currentPermalinkIdsByCampaignId = getCurrentPermalinksByCampaignId(dslContext,
                changingCampaignIds);

        //на текущий момент не можем узнать напрямую источник счетчика метрики в кампании

        //на первом этапе вычисляем пермалинки, которые были сброшены при обновлении баннеров
        Map<Long, Set<Long>> deletingPermalinkIdsByCampaignId =
                getDeletingPermalinkIdsByCampaignId(bannerChangesShouldUpdateCampaignCounters,
                        currentPermalinkIdsByCampaignId, updateContainer);

        //для всех удаляющихся и добавляющиихся пермалинков получаем счетчики метрики
        Set<Long> permalinkIdsToFetchCounters = StreamEx.of(currentPermalinkIdsByCampaignId.values())
                .flatMap(StreamEx::of)
                .append(flatMapToSet(deletingPermalinkIdsByCampaignId.values(), Function.identity()))
                .toSet();

        Map<Long, Long> counterIdByPermalinkId;
        try {
            counterIdByPermalinkId = organizationService.getMetrikaCountersByOrganizationsIds(
                    getLanguageByName(LocaleContextHolder.getLocale().getLanguage()).orElse(LanguageOuterClass.Language.EN),
                    permalinkIdsToFetchCounters);
        } catch (OrganizationsClientTranslatableException e) {
            // игнорируем обновление счетчиков при недоступности Справочника при отвязке организации
            if (StreamEx.of(bannerChangesShouldUpdateCampaignCounters)
                    .allMatch(c -> c.getNewValue(BannerWithOrganization.PERMALINK_ID) == null)) {
                logger.warn("Do not update organization counters for campaigns due to organization client problem");
                return;
            }
            throw e;
        }

        Map<Long, List<MetrikaCounter>> actualMetrikaCounterByCampaignId = campMetrikaCountersRepository
                .getMetrikaCounterByCid(dslContext, changingCampaignIds);

        Map<Long, Set<Long>> actualMetrikaCounterIdsByCampaignId = EntryStream.of(actualMetrikaCounterByCampaignId)
                .mapValues(counters -> listToSet(counters, MetrikaCounter::getId))
                .toMap();

        Map<Long, Set<Long>> currentPermalinkCounterIdsByCampaignId =
                getCounterIdsByCampaignIdForPermalinksByCampaignId(
                        currentPermalinkIdsByCampaignId, counterIdByPermalinkId);

        //счетчики, которые принадлежали удаленным из баннеров пермалинкам нужно удалить, если у других баннеров не было
        //таких же организаций
        Map<Long, Set<Long>> deletingPermalinkCounterIdsByCampaignId =
                getCounterIdsByCampaignIdForPermalinksByCampaignId(
                        deletingPermalinkIdsByCampaignId, counterIdByPermalinkId);

        Map<Long, Set<Long>> newMetrikaCounterIdsByCampaignId = StreamEx.of(changingCampaignIds)
                .mapToEntry(cid -> getNewMetrikaCounterIds(
                        actualMetrikaCounterIdsByCampaignId.getOrDefault(cid, emptySet()),
                        deletingPermalinkCounterIdsByCampaignId.getOrDefault(cid, emptySet()),
                        currentPermalinkCounterIdsByCampaignId.getOrDefault(cid, emptySet())))
                .toMap();

        updateCampaignMetrikaCounters(dslContext, newMetrikaCounterIdsByCampaignId, actualMetrikaCounterByCampaignId,
                updateContainer,
                changingCampaignIds,
                Set.copyOf(counterIdByPermalinkId.values()));
    }

    private boolean shouldUpdateCampaignCounter(AppliedChanges<BannerWithOrganization> ac) {
        return ac.changed(BannerWithOrganization.PERMALINK_ID);
    }

    private Map<Long, Set<Long>> getCurrentPermalinksByCampaignId(DSLContext dslContext,
                                                                  Set<Long> campaignIds) {
        Map<Long, List<Long>> allBannerIdsByUpdatingCampaignIds =
                bannerRelationsRepository.getBannerIdsByCampaignIdsMap(dslContext, campaignIds);

        List<Long> updatingCampaignsBannerIds = flatMap(allBannerIdsByUpdatingCampaignIds.values(),
                Function.identity());

        Map<Long, Long> permalinkIdByBannerId = organizationRepository.getPermalinkIdsByBannerIds(dslContext,
                updatingCampaignsBannerIds);

        return EntryStream.of(allBannerIdsByUpdatingCampaignIds)
                .mapValues(campaignBannerIds -> StreamEx.of(campaignBannerIds)
                        .map(permalinkIdByBannerId::get)
                        .nonNull()
                        .toSet())
                .toMap();
    }

    private Map<Long, Set<Long>> getDeletingPermalinkIdsByCampaignId(
            Collection<AppliedChanges<BannerWithOrganization>> appliedChanges,
            Map<Long, Set<Long>> currentPermalinkIdsByCampaignId,
            BannersUpdateOperationContainer updateContainer) {
        return StreamEx.of(appliedChanges)
                .mapToEntry(ac -> updateContainer.getCampaign(ac.getModel()).getId(),
                        ac -> ac.getOldValue(BannerWithOrganization.PERMALINK_ID))
                .nonNullValues()
                .removeKeyValue((campaignId, permalinkId) ->
                        currentPermalinkIdsByCampaignId.get(campaignId).contains(permalinkId))
                .sortedBy(Map.Entry::getKey)
                .collapseKeys()
                .mapValues(Set::copyOf)
                .toMap();
    }

    private static Map<Long, Set<Long>> getCounterIdsByCampaignIdForPermalinksByCampaignId(
            Map<Long, Set<Long>> permalinkIdsByCampaignId,
            Map<Long, Long> counterIdByPermalinkId) {
        return EntryStream.of(permalinkIdsByCampaignId)
                .mapValues(campaignPermalinkIds -> StreamEx.of(campaignPermalinkIds)
                        .map(counterIdByPermalinkId::get)
                        .nonNull()
                        .toSet())
                .toMap();
    }

    public static Set<Long> getNewMetrikaCounterIds(Set<Long> actualCounters, Set<Long> deletingCounters,
                                                    Set<Long> addingCounters) {
        return StreamEx.of(actualCounters)
                .remove(deletingCounters::contains)
                .append(addingCounters)
                .toSet();
    }

    private void updateCampaignMetrikaCounters(DSLContext dslContext,
                                               Map<Long, Set<Long>> newMetrikaCounterIdsByCampaignId,
                                               Map<Long, List<MetrikaCounter>> actualMetrikaCounterByCampaignId,
                                               BannersUpdateOperationContainer updateContainer,
                                               Set<Long> changingCampaignIds,
                                               Set<Long> organizationCounterIds) {
        List<Long> metrikaCounterCampaignIdsToDelete = EntryStream.of(newMetrikaCounterIdsByCampaignId)
                .filterValues(Set::isEmpty)
                .keys()
                .toList();

        campMetrikaCountersRepository.deleteMetrikaCounters(dslContext, metrikaCounterCampaignIdsToDelete);

        Map<Long, List<MetrikaCounter>> countersByCampaignId =
                CampMetrikaCountersService.
                        calculateCountersByCampaignIdForNewMetrikaCounterIds(newMetrikaCounterIdsByCampaignId,
                                actualMetrikaCounterByCampaignId,
                                organizationCounterIds);

        if (updateContainer.isFeatureEnabledForClient(FeatureName.TOGETHER_UPDATING_STRATEGY_AND_CAMPAIGN_METRIKA_COUNTERS)) {
            campMetrikaCountersService.updateMetrikaCounterForStrategies(
                    updateContainer.getShard(),
                    updateContainer.getClientId(),
                    countersByCampaignId,
                    List.copyOf(changingCampaignIds),
                    EntryStream.of(newMetrikaCounterIdsByCampaignId)
                            .mapValues(List::copyOf)
                            .toMap()
            );
        } else {
            campMetrikaCountersRepository.updateMetrikaCounters(dslContext, countersByCampaignId);
        }
    }

}
