package ru.yandex.direct.core.entity.banner.type.performance;

import java.util.List;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.banner.container.BannersAddOperationContainer;
import ru.yandex.direct.core.entity.banner.model.PerformanceBannerMain;
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository;
import ru.yandex.direct.core.entity.banner.service.validation.type.add.AbstractBannerAddValidationTypeSupport;
import ru.yandex.direct.core.entity.banner.type.performance.defects.PerformanceBannerMainDefects;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.unique;

@Component
public class PerformanceBannerMainAddValidationTypeSupport
        extends AbstractBannerAddValidationTypeSupport<PerformanceBannerMain> {

    private final BannerTypedRepository bannerTypedRepository;
    private final FeatureService featureService;
    private final ShardHelper shardHelper;

    @Autowired
    public PerformanceBannerMainAddValidationTypeSupport(BannerTypedRepository bannerTypedRepository,
                                                         FeatureService featureService,
                                                         ShardHelper shardHelper) {
        this.bannerTypedRepository = bannerTypedRepository;
        this.featureService = featureService;
        this.shardHelper = shardHelper;
    }

    @Override
    public Class<PerformanceBannerMain> getTypeClass() {
        return PerformanceBannerMain.class;
    }

    @Override
    public ValidationResult<List<PerformanceBannerMain>, Defect> validate(BannersAddOperationContainer container,
                                                                          ValidationResult<List<PerformanceBannerMain>, Defect> vr) {
        List<Long> adGroupsIds = mapList(vr.getValue(), PerformanceBannerMain::getAdGroupId);
        if (adGroupsIds.isEmpty()) {
            return vr;
        }

        if (!featureService.isEnabledForClientId(container.getClientId(), FeatureName.SMART_NO_CREATIVES)) {
            return new ListValidationBuilder<>(vr)
                    .checkEach(fromPredicate(t -> false,
                            PerformanceBannerMainDefects.bannersWithoutCreativeNotEnabled()))
                    .getResult();
        }

        int shard = shardHelper.getShardByClientId(container.getClientId());
        var bannersByGroupIds = bannerTypedRepository.getBannersByGroupIds(shard, adGroupsIds,
                PerformanceBannerMain.class);
        var bannersByAdgroupIdsMap = listToMap(bannersByGroupIds, PerformanceBannerMain::getAdGroupId);

        return new ListValidationBuilder<>(vr)
                // нельзя добавить больше 1 баннера на группу
                .checkEach(unique(PerformanceBannerMain::getAdGroupId))
                // при этом надо учесть уже имеющиеся в базе баннеры
                .checkEach(fromPredicate(b -> !bannersByAdgroupIdsMap.containsKey(b.getAdGroupId()),
                        PerformanceBannerMainDefects.bannerInThisAdgroupAlreadyExists()))
                .getResult();
    }
}
