package ru.yandex.direct.core.entity.banner.type.phone;

import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.core.entity.clientphone.ClientPhoneIdsByTypeContainer;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.organizations.validation.OrganizationDefects.hasNoAccessToOrganization;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;

@ParametersAreNonnullByDefault
public class PhoneIdValidator implements Validator<Long, Defect> {

    private final Long permalinkId;
    private final Set<Long> accessibleOrganizationsPermalinkId;
    private final ClientPhoneIdsByTypeContainer phoneIdsContainer;
    private final Boolean isCopy;
    private final Boolean needToCheckPermalinkAccess;

    public PhoneIdValidator(
            Long permalinkId,
            Set<Long> accessibleOrganizationsPermalinkId,
            ClientPhoneIdsByTypeContainer phoneIdsContainer,
            boolean needToCheckPermalinkAccess,
            boolean isCopy) {
        this.permalinkId = permalinkId;
        this.accessibleOrganizationsPermalinkId = accessibleOrganizationsPermalinkId;
        this.phoneIdsContainer = phoneIdsContainer;
        this.needToCheckPermalinkAccess = needToCheckPermalinkAccess;
        this.isCopy = isCopy;
    }

    @Override
    public ValidationResult<Long, Defect> apply(Long phoneId) {
        return ItemValidationBuilder.<Long, Defect>of(phoneId)
                .check(checkPermalinkAccess())
                .check(checkValidPhoneId(permalinkId), When.isValid())
                .getResult();
    }

    private Constraint<Long, Defect> checkPermalinkAccess() {
        return fromPredicate(
                phoneId -> !needToCheckPermalinkAccess || accessibleOrganizationsPermalinkId.contains(permalinkId),
                hasNoAccessToOrganization()
        );
    }

    private Constraint<Long, Defect> checkValidPhoneId(Long permalinkId) {
        return fromPredicate(
                phoneId -> {
                    Set<Long> notManualPhoneIds = phoneIdsContainer.getNotManualPhoneIds(permalinkId);
                    Set<Long> manualPhoneIds = phoneIdsContainer.getManualPhoneIds();
                    return isCopy || notManualPhoneIds.contains(phoneId) || manualPhoneIds.contains(phoneId);
                },
                CommonDefects.objectNotFound()
        );
    }

}
