package ru.yandex.direct.core.entity.banner.type.pixels;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.result.Defect;

import static ru.yandex.direct.core.entity.adgroup.model.AdGroupType.CPM_AUDIO;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.equalPixelProvidersLimitExceeded;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.invalidPixelFormat;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.noRightsToAudiencePixel;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.noRightsToPixel;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.pixelNumberNonYaAudienceLimitExceeded;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.pixelNumberYaAudienceLimitExceeded;
import static ru.yandex.direct.core.entity.banner.type.pixels.BannerPixelsConstants.MAX_EQUAL_PROVIDERS_ON_BANNER;
import static ru.yandex.direct.core.entity.banner.type.pixels.BannerPixelsConstants.MAX_NOT_YA_AUDIENCE_PIXELS_COUNT_ON_BANNER_DEFAULT;
import static ru.yandex.direct.core.entity.banner.type.pixels.BannerPixelsConstants.MAX_NOT_YA_AUDIENCE_PIXELS_COUNT_ON_BANNER_VIDEO_AUDIO;
import static ru.yandex.direct.core.entity.banner.type.pixels.BannerPixelsConstants.MAX_YA_AUDIENCE_PIXELS_COUNT_ON_BANNER;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;

public class BannerPixelsConstraints {
    private BannerPixelsConstraints() {
    }

    /**
     * Ограничение на число пикселей Я.Аудиторий
     */
    public static Constraint<List<String>, Defect> maxNumberOfYaAudiencePixels() {
        return fromPredicate(pixelUrls -> StreamEx.of(pixelUrls)
                        .filter(Objects::nonNull)
                        .map(PixelProvider::fromUrl)
                        .filter(Objects::nonNull)
                        .filter(PixelProvider::isYaAudiencePixelProvider)
                        .count() <= MAX_YA_AUDIENCE_PIXELS_COUNT_ON_BANNER,
                pixelNumberYaAudienceLimitExceeded());
    }

    /**
     * Ограничение на число пикселей, не являющихся пикселями Я.Аудиторий
     */
    public static Constraint<List<String>, Defect> maxNumberOfNotYaAudiencePixels(AdGroupType adGroupType) {
        int nonAudiencePixelLimits =
                (adGroupType == CPM_AUDIO) ?
                        MAX_NOT_YA_AUDIENCE_PIXELS_COUNT_ON_BANNER_VIDEO_AUDIO
                        : MAX_NOT_YA_AUDIENCE_PIXELS_COUNT_ON_BANNER_DEFAULT;
        return fromPredicate(pixelUrls -> StreamEx.of(pixelUrls)
                        .filter(Objects::nonNull)
                        .map(PixelProvider::fromUrl)
                        .filter(Objects::nonNull)
                        .filter(pixelProvider -> !pixelProvider.isYaAudiencePixelProvider())
                        .count() <= nonAudiencePixelLimits,
                pixelNumberNonYaAudienceLimitExceeded());
    }

    /**
     * Ограничение, что на баннере не может быть много пикселей одного и того же провайдера
     * Выдаём ошибку только по тем провайдерам, которые разрешены клиенту, так как по остальным ошибка другого характера
     * {@link ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects#noRightsToPixel}
     *
     * @param accessibleProviderNames имена разрешённых клиенту провайдеров
     */
    public static Constraint<List<String>, Defect> maxNumberOfSamePixelProviders(
            Set<String> accessibleProviderNames) {
        return pixelUrls -> {
            if (pixelUrls == null) {
                return null;
            }

            Map<String, List<PixelProvider>> pixelUrlsByProvider = StreamEx.of(pixelUrls)
                    .filter(Objects::nonNull)
                    .map(PixelProvider::fromUrl)
                    .filter(Objects::nonNull)
                    .groupingBy(PixelProvider::getProviderName);
            List<String> exceededLimitsPixelProviderNames = EntryStream.of(pixelUrlsByProvider)
                    .filterValues(t -> t.size() > MAX_EQUAL_PROVIDERS_ON_BANNER)
                    .keys()
                    .filter(accessibleProviderNames::contains)
                    .toList();

            return exceededLimitsPixelProviderNames.isEmpty() ? null :
                    equalPixelProvidersLimitExceeded(exceededLimitsPixelProviderNames);
        };
    }

    /**
     * Проверка на то, что все добавляемые пиксели в разрешённом списке
     */
    public static Constraint<String, Defect> validPixelFormat() {
        return pixelUrl -> PixelProvider.fromUrl(pixelUrl) != null ? null : invalidPixelFormat();
    }

    public static Constraint<String, Defect> validPermissionsForYaAudienceProvider(Set<PixelProvider> accessibleProviders) {
        return pixelUrl -> {
            PixelProvider pixelProvider = PixelProvider.fromUrl(pixelUrl);
            return pixelProvider.isYaAudiencePixelProvider() && !accessibleProviders.contains(pixelProvider) ?
                    noRightsToAudiencePixel(pixelUrl) : null;
        };
    }

    public static Constraint<String, Defect> validPermissionsForProvider(Set<PixelProvider> accessibleProviders) {
        return pixelUrl -> {
            PixelProvider pixelProvider = PixelProvider.fromUrl(pixelUrl);
            return !pixelProvider.isYaAudiencePixelProvider() && !accessibleProviders.contains(pixelProvider) ?
                    noRightsToPixel(pixelUrl, accessibleProviders) : null;
        };
    }
}
