package ru.yandex.direct.core.entity.banner.type.pixels;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.jooq.DSLContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.banner.model.BannerPixel;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;

import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS_TNS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_PIXELS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Repository
@ParametersAreNonnullByDefault
public class BannerPixelsRepository {

    private final DslContextProvider dslContextProvider;
    private final JooqMapperWithSupplier<BannerPixel> bannerPixelsMapper = createBannerPixelsMapper();

    @Autowired
    public BannerPixelsRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
    }


    public void deleteBannerPixelsByBannerIds(DSLContext context, Collection<Long> bannerIds) {
        context.deleteFrom(BANNER_PIXELS)
                .where(BANNER_PIXELS.BID.in(bannerIds))
                .execute();
    }


    /**
     * Получение списка пикселей баннеров
     *
     * @param shard     шард для запроса
     * @param bannerIds список id баннеров
     * @return мапа bannerId -> список pixel_url
     */
    public Map<Long, List<String>> getPixelsByBannerIds(int shard, Collection<Long> bannerIds) {
        List<BannerPixel> pixelsResult = dslContextProvider.ppc(shard)
                .select(BANNER_PIXELS.BID, BANNER_PIXELS.PIXEL_URL)
                .from(BANNER_PIXELS)
                .where(BANNER_PIXELS.BID.in(bannerIds))
                .fetch()
                .map(bannerPixelsMapper::fromDb);

        return StreamEx.of(pixelsResult)
                .mapToEntry(BannerPixel::getBannerId, identity())
                .collapseKeys(toList())
                .mapValues(bannerPixels -> mapList(bannerPixels, BannerPixel::getPixelUrl))
                .toMap();
    }

    /**
     * Получение списка tns баннеров
     *
     * @param shard     шард для запроса
     * @param bannerIds список id баннеров
     * @return мапа bannerId -> список tns_id
     */
    public Map<Long, String> getTnsByIds(int shard, Collection<Long> bannerIds) {
        return dslContextProvider.ppc(shard)
                .select(BANNERS_TNS.BID, BANNERS_TNS.TNS_ID)
                .from(BANNERS_TNS)
                .where(BANNERS_TNS.BID.in(bannerIds))
                .fetchMap(BANNERS_TNS.BID, BANNERS_TNS.TNS_ID);
    }

    private static JooqMapperWithSupplier<BannerPixel> createBannerPixelsMapper() {
        return JooqMapperWithSupplierBuilder.builder(BannerPixel::new)
                .map(property(BannerPixel.BANNER_ID, BANNER_PIXELS.BID))
                .map(property(BannerPixel.PIXEL_ID, BANNER_PIXELS.PIXEL_ID))
                .map(property(BannerPixel.PIXEL_URL, BANNER_PIXELS.PIXEL_URL))
                .build();
    }
}
