package ru.yandex.direct.core.entity.banner.type.pixels;

import java.util.IdentityHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import one.util.streamex.EntryStream;
import one.util.streamex.IntStreamEx;
import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupForBannerOperation;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.banner.container.BannerWithPixelsValidationContainer;
import ru.yandex.direct.core.entity.banner.container.BannerWithPixelsValidationContainerImpl;
import ru.yandex.direct.core.entity.banner.model.Banner;
import ru.yandex.direct.core.entity.banner.model.BannerWithAdGroupId;
import ru.yandex.direct.core.entity.banner.model.BannerWithPixels;
import ru.yandex.direct.core.entity.banner.service.validation.BannerValidationInfo;
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.utils.CommonUtils;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.DefectInfo;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Collections.emptyList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.result.PathHelper.index;

/**
 * Валидация cpm-баннеров при изменениях объектов, от которых зависят права на пиксели
 */
public class BannerWithPixelsInterconnectionsValidator {

    private Map<Integer, List<Defect>> bannerValidationErrorsByIndex;
    private Map<Integer, List<Defect>> bannerValidationWarningsByIndex;
    private Map<Integer, List<Integer>> bannerIndexesByEntityIndex;

    private final BannerWithPixelsValidatorProvider bannerWithPixelsValidatorProvider;
    private final FeatureService featureService;
    private final AdGroupRepository adGroupRepository;
    private final CampaignTypedRepository campaignTypedRepository;

    public BannerWithPixelsInterconnectionsValidator(BannerWithPixelsValidatorProvider bannerWithPixelsValidatorProvider,
                                                        FeatureService featureService,
                                                        AdGroupRepository adGroupRepository,
                                                        CampaignTypedRepository campaignTypedRepository) {

        this.bannerWithPixelsValidatorProvider = bannerWithPixelsValidatorProvider;
        this.featureService = featureService;
        this.adGroupRepository = adGroupRepository;
        this.campaignTypedRepository = campaignTypedRepository;
    }

    /**
     * Предобработка данных и осуществление валидации при изменении объектов, от которых зависит валидация пикселей
     * Ошибки и предупреждения, полученные при валидации, группируются по индексу сущности в списке, подаваемом на
     * валидацию
     *
     * @param shard                        номер шарда
     * @param clientId                     идентификатор клиента
     * @param bannersByEntityIndex         список баннеров по индексу сущности
     * @param entityList                   список сущностей, которые надо будет провалидировать
     * @param bannerValidationInfoByEntity метод получения BannerValidationInfo, соответствующего баннеру, по сущности,
     *                                     с ним связанной
     */
    public <E> void processValidation(int shard, ClientId clientId,
                                      Map<Integer, List<BannerWithPixels>> bannersByEntityIndex,
                                      List<E> entityList,
                                      Function<List<E>, BannerValidationInfo> bannerValidationInfoByEntity,
                                      boolean skipPixelValidation) {
        List<BannerWithPixels> bannerList = getBannersList(bannersByEntityIndex);

        bannerIndexesByEntityIndex =
                groupBannerIndexesByEntityIndexes(bannerList, bannersByEntityIndex, entityList);

        Map<Integer, BannerValidationInfo> bannerValidationInfoMap =
                getBannerValidationInfoMap(bannerList, entityList, bannerValidationInfoByEntity);

        ValidationResult<List<BannerWithPixels>, Defect> listValidationResult;
        if (bannerList.isEmpty()) {
            listValidationResult = new ValidationResult<>(emptyList());
        } else {

            BannerWithPixelsValidationContainer container =
                    createContainer(shard, clientId, bannerList, bannerValidationInfoMap);

            listValidationResult =
                    ListValidationBuilder.of(bannerList, Defect.class)
                            .checkEachBy(bannerWithPixelsValidatorProvider.validator(container, bannerList),
                                    When.isTrue(!skipPixelValidation))
                            .getResult();
        }

        bannerValidationErrorsByIndex = IntStreamEx.range(listValidationResult.getValue().size()).boxed()
                .mapToEntry(ind -> listValidationResult.getSubResults().get(index(ind)).flattenErrors())
                .mapValues(defectInfoList -> mapList(defectInfoList, DefectInfo::getDefect))
                .toMap();
        bannerValidationWarningsByIndex = IntStreamEx.range(listValidationResult.getValue().size()).boxed()
                .mapToEntry(ind -> listValidationResult.getSubResults().get(index(ind)).flattenWarnings())
                .mapValues(defectInfoList -> mapList(defectInfoList, DefectInfo::getDefect))
                .toMap();
    }

    private BannerWithPixelsValidationContainer createContainer(int shard,
                                                                ClientId clientId,
                                                                List<BannerWithPixels> bannerList,
                                                                Map<Integer, BannerValidationInfo> bannerValidationInfoMap) {

        Set<String> clientEnabledFeatures = featureService.getEnabledForClientId(clientId);

        IdentityHashMap<Banner, Integer> bannerToIndexMap = EntryStream.of(bannerList)
                .mapValues(b -> (Banner) b)
                .invert()
                .toCustomMap(IdentityHashMap::new);

        Map<Integer, AdGroupForBannerOperation> indexToAdGroupMap = getIndexToAdGroupMap(shard, bannerList);
        Map<Integer, CommonCampaign> campaignsMap = getIndexToCampaignMap(shard, indexToAdGroupMap);

        return new BannerWithPixelsValidationContainerImpl(
                shard,
                clientId,
                clientEnabledFeatures,
                indexToAdGroupMap,
                bannerToIndexMap,
                campaignsMap,
                bannerValidationInfoMap
        );
    }

    /**
     * Извлекаем информацию о кампаниях из базы
     *
     * @return позиция баннера -> CommonCampaign
     */
    private Map<Integer, CommonCampaign> getIndexToCampaignMap(int shard,
                                                               Map<Integer, AdGroupForBannerOperation> indexToAdGroupMap) {
        Set<Long> campaignIds = listToSet(indexToAdGroupMap.values(), AdGroupForBannerOperation::getCampaignId);

        List<CommonCampaign> campaignList =
                campaignTypedRepository.getStrictly(shard, campaignIds, CommonCampaign.class);

        Map<Long, CommonCampaign> campaigns = listToMap(campaignList, CommonCampaign::getId);
        return EntryStream.of(indexToAdGroupMap)
                .mapValues(adGroup -> campaigns.get(adGroup.getCampaignId()))
                .nonNullValues()
                .toMap();
    }


    /**
     * Извлекаем информацию о группах из базы
     *
     * @return позиция баннера -> AdGroupForBannerOperation
     */
    private Map<Integer, AdGroupForBannerOperation> getIndexToAdGroupMap(int shard,
                                                                         List<BannerWithPixels> bannersList) {

        Set<Long> adGroupIds = bannersList.stream()
                .map(x -> (BannerWithAdGroupId) x)
                .map(BannerWithAdGroupId::getAdGroupId)
                .filter(CommonUtils::isValidId)
                .collect(Collectors.toSet());

        Map<Long, AdGroupForBannerOperation> adGroupsInfo =
                adGroupRepository.getAdGroupsForBannerOperation(shard, adGroupIds, null);

        return EntryStream.of(bannersList)
                .selectValues(BannerWithAdGroupId.class)
                .mapValues(banner -> adGroupsInfo.get(banner.getAdGroupId()))
                .nonNullValues()
                .toMap();
    }

    /**
     * Получение списка баннеров с уникальными идентификаторами по мапе, в списках значений которых они фигурируют
     */
    private List<BannerWithPixels> getBannersList(Map<Integer, List<BannerWithPixels>> bannersByEntityIndex) {
        return StreamEx.of(bannersByEntityIndex.values())
                .flatMap(StreamEx::of)
                .distinct(BannerWithPixels::getId)
                .toList();
    }

    /**
     * Получение мапы индексов баннеров в подаваемом списке по индексу сущности в списке сущностей
     *
     * @param bannerList           список баннеров с пикселями
     * @param bannersByEntityIndex список баннеров по индексу сущности
     * @param entityList           список сущностей, которые надо будет провалидировать
     */
    private <E> Map<Integer, List<Integer>> groupBannerIndexesByEntityIndexes(List<BannerWithPixels> bannerList,
                                                                              Map<Integer, List<BannerWithPixels>> bannersByEntityIndex,
                                                                              List<E> entityList) {
        Map<Integer, Long> bannerIdByIndex = EntryStream.of(bannerList)
                .mapValues(BannerWithPixels::getId)
                .toMap();
        Map<Long, Integer> bannerIndexById = EntryStream.of(bannerIdByIndex)
                .collect(Collectors.toMap(Map.Entry::getValue, Map.Entry::getKey));

        return EntryStream.of(entityList)
                .mapToValue((ind, entity) -> bannersByEntityIndex.get(ind))
                .filterValues(Objects::nonNull)
                .mapValues(banners -> mapList(banners, BannerWithPixels::getId))
                .mapValues(t -> mapList(t, bannerIndexById::get))
                .toMap();
    }

    /**
     * Получение BannerValidationInfo для каждого баннера, отправляемого на валидацию
     *
     * @param bannerList                   список баннеров с пикселями
     * @param bannerValidationInfoByEntity метод получения BannerValidationInfo, соответствующего баннеру, по списку
     *                                     сущностей,
     *                                     с ним связанным
     * @param entityList                   список сущностей, которые надо будет провалидировать
     * @return мапа индексов баннеров в BannerValidationInfo, им соответствующих
     */
    private <E> Map<Integer, BannerValidationInfo> getBannerValidationInfoMap(List<BannerWithPixels> bannerList,
                                                                              List<E> entityList,
                                                                              Function<List<E>, BannerValidationInfo> bannerValidationInfoByEntity) {

        Map<Integer, List<Integer>> entityIndexesByBannerIndex = EntryStream.of(bannerIndexesByEntityIndex)
                .flatMapValues(StreamEx::of)
                .invert()
                .grouping();
        return EntryStream.of(bannerList)
                .mapToValue((index, bannerWithPixels) -> bannerValidationInfoByEntity.apply(
                        mapList(entityIndexesByBannerIndex.get(index), entityList::get)))
                .toMap();
    }

    /**
     * Получение результатов валидации баннеров по сущности и её индексу в списке
     *
     * @param entity      изменяемая сущность
     * @param entityIndex индекс сущности в списке
     * @return Результат валидации Entity содержит все ошибки баннеров в виде списка
     */
    public <E> ValidationResult<E, Defect> getValidationResultForEntity(E entity, Integer entityIndex) {
        List<Integer> bannerIndexesForEntity = bannerIndexesByEntityIndex.get(entityIndex);
        List<Defect> errorsList = bannerIndexesForEntity == null ? emptyList() :
                StreamEx.of(bannerIndexesForEntity)
                        .map(bannerValidationErrorsByIndex::get)
                        .toFlatList(t -> t);
        List<Defect> warningsList = bannerIndexesForEntity == null ? emptyList() :
                StreamEx.of(bannerIndexesForEntity)
                        .map(bannerValidationWarningsByIndex::get)
                        .toFlatList(t -> t);
        return new ValidationResult<>(entity, errorsList, warningsList);
    }
}
