package ru.yandex.direct.core.entity.banner.type.pixels;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.jetbrains.annotations.NotNull;
import org.jooq.Condition;
import org.jooq.DSLContext;
import org.jooq.Record2;
import org.jooq.Result;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.banner.container.BannerRepositoryContainer;
import ru.yandex.direct.core.entity.banner.container.BannersOperationContainer;
import ru.yandex.direct.core.entity.banner.model.BannerPixel;
import ru.yandex.direct.core.entity.banner.model.BannerWithPixels;
import ru.yandex.direct.core.entity.banner.repository.AddOrUpdateAndDeleteByKeyContainer;
import ru.yandex.direct.core.entity.banner.repository.AddOrUpdateAndDeleteContainer;
import ru.yandex.direct.core.entity.banner.repository.type.AbstractMultiRowEntityRepositoryTypeSupport;
import ru.yandex.direct.core.entity.banner.repository.type.ModifiedPaths;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.grut.objects.proto.client.Schema;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptySet;
import static org.apache.commons.collections4.CollectionUtils.isEmpty;
import static org.apache.commons.lang3.ObjectUtils.defaultIfNull;
import static ru.yandex.direct.core.entity.banner.model.BannerWithPixels.PIXELS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_PIXELS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.model.AppliedChanges.isChanged;
import static ru.yandex.direct.utils.FunctionalUtils.flatMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
@Component
public class BannerWithPixelsRepositoryTypeSupport
        extends AbstractMultiRowEntityRepositoryTypeSupport
        <BannerWithPixels, BannerPixel> {

    private static final String PROTO_PIXEL_PATH = "/spec/pixels";

    private final ShardHelper shardHelper;
    private final JooqMapperWithSupplier<BannerPixel> bannerPixelMapper;

    @Autowired
    public BannerWithPixelsRepositoryTypeSupport(
            DslContextProvider dslContextProvider,
            ShardHelper shardHelper) {
        super(dslContextProvider);
        this.bannerPixelMapper = createBannerPixelMapper();
        this.shardHelper = shardHelper;
    }

    @Override
    protected AddOrUpdateAndDeleteContainer<BannerPixel> buildAddOrUpdateAndDeleteContainer(
            DSLContext context,
            BannerRepositoryContainer updateParameters,
            Collection<AppliedChanges<BannerWithPixels>> bannersChangesCollection) {
        var builder = AddOrUpdateAndDeleteByKeyContainer.builder(BannerPixel::getPixelUrl)
                .skipNewRowsByKeyEquals();

        StreamEx.of(bannersChangesCollection)
                .filter(isChanged(PIXELS))
                .forEach(ac -> {
                    Long bannerId = ac.getModel().getId();
                    List<BannerPixel> dbPixels = createBannerPixels(bannerId, ac.getOldValue(PIXELS));
                    List<BannerPixel> actualPixels = createBannerPixels(bannerId, ac.getNewValue(PIXELS));
                    builder.addDiff(dbPixels, actualPixels);
                });

        return builder.build();
    }

    private List<BannerPixel> createBannerPixels(Long bannerId, @Nullable List<String> pixelUrls) {
        if (pixelUrls == null) {
            return emptyList();
        }
        return mapList(pixelUrls, url -> new BannerPixel().withBannerId(bannerId).withPixelUrl(url));
    }

    @Nonnull
    @Override
    protected List<BannerPixel> extractNewRepositoryModels(Collection<BannerWithPixels> banners) {
        return flatMap(banners, banner -> createBannerPixels(banner.getId(), banner.getPixels()));
    }

    @Override
    protected void delete(DSLContext context, Collection<BannerPixel> pixelsToDelete) {
        if (isEmpty(pixelsToDelete)) {
            return;
        }

        Condition condition = StreamEx.of(pixelsToDelete)
                .mapToEntry(BannerPixel::getBannerId, BannerPixel::getPixelUrl)
                .collapseKeys()
                .mapKeyValue((bannerId, pixelUrls) ->
                        BANNER_PIXELS.BID.eq(bannerId).and(BANNER_PIXELS.PIXEL_URL.in(pixelUrls)))
                .reduce(Condition::or)
                .orElse(DSL.falseCondition());

        context.deleteFrom(BANNER_PIXELS)
                .where(condition)
                .execute();
    }

    @Override
    protected void addOrUpdate(DSLContext context, Collection<BannerPixel> newBannerPixels) {
        if (isEmpty(newBannerPixels)) {
            return;
        }
        generatePixelIds(newBannerPixels);
        new InsertHelper<>(context, BANNER_PIXELS)
                .addAll(bannerPixelMapper, newBannerPixels)
                .execute();
    }

    private void generatePixelIds(Collection<BannerPixel> newBannerPixels) {
        List<Long> pixelIds = shardHelper.generatePixelIds(newBannerPixels.size());
        StreamEx.of(newBannerPixels).zipWith(pixelIds.stream())
                .forKeyValue(BannerPixel::setPixelId);
    }

    @Override
    public void enrichModelFromOtherTables(DSLContext dslContext, Collection<BannerWithPixels> banners) {
        Set<Long> bannerIds = listToSet(banners, BannerWithPixels::getId);
        Result<Record2<Long, String>> result = dslContext
                .select(BANNER_PIXELS.BID, BANNER_PIXELS.PIXEL_URL)
                .from(BANNER_PIXELS)
                .where(BANNER_PIXELS.BID.in(bannerIds))
                .fetch();
        Map<Long, List<String>> bannerIdToPixelUrls = StreamEx.of(result)
                .mapToEntry(Record2::value1, Record2::value2)
                .grouping();
        banners.forEach(banner -> {
            banner.setPixels(defaultIfNull(bannerIdToPixelUrls.get(banner.getId()), emptyList()));
        });
    }

    @Override
    public Class<BannerWithPixels> getTypeClass() {
        return BannerWithPixels.class;
    }

    private static JooqMapperWithSupplier<BannerPixel> createBannerPixelMapper() {
        return JooqMapperWithSupplierBuilder.builder(BannerPixel::new)
                .map(property(BannerPixel.BANNER_ID, BANNER_PIXELS.BID))
                .map(property(BannerPixel.PIXEL_ID, BANNER_PIXELS.PIXEL_ID))
                .map(property(BannerPixel.PIXEL_URL, BANNER_PIXELS.PIXEL_URL))
                .build();
    }

    @Override
    public Set<ModelProperty<? super BannerWithPixels, ?>> getGrutSupportedProperties() {
        return Set.of(PIXELS);
    }

    @Override
    public Map<Long, ModifiedPaths> applyToGrutObjects(
            @NotNull Map<Long, Schema.TBannerV2.Builder> bannerBuilders,
            @NotNull Collection<AppliedChanges<BannerWithPixels>> appliedChangesList,
            @NotNull BannersOperationContainer operationContainer) {

        Map<Long, ModifiedPaths> modifiedPathsMap = new HashMap<>();
        for (var appliedChanges : appliedChangesList) {
            if (!appliedChanges.getPropertiesForUpdate().contains(PIXELS)) {
                continue;
            }
            Long id = appliedChanges.getModel().getId();
            var newValue = appliedChanges.getNewValue(PIXELS);
            ModifiedPaths modifiedPaths;
            if (newValue != null) {
                Schema.TBannerV2.Builder bannerBuilder = bannerBuilders.get(id);
                bannerBuilder.getSpecBuilder().clearPixels();
                bannerBuilder.getSpecBuilder().addAllPixels(newValue);
                modifiedPaths = new ModifiedPaths(Set.of(PROTO_PIXEL_PATH));
            } else {
                modifiedPaths = new ModifiedPaths(emptySet(), Set.of(PROTO_PIXEL_PATH));
            }
            modifiedPathsMap.put(id, modifiedPaths);
        }
        return modifiedPathsMap;
    }
}
