package ru.yandex.direct.core.entity.banner.type.pixels;

import java.util.Collection;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import javax.annotation.Nullable;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupForBannerOperation;
import ru.yandex.direct.core.entity.adgroup.model.CpmBannerAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.CriterionType;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.banner.container.BannerWithPixelsValidationContainer;
import ru.yandex.direct.core.entity.banner.container.BannerWithPixelsValidationContainerBannerOperationAdapter;
import ru.yandex.direct.core.entity.banner.container.BannersOperationContainer;
import ru.yandex.direct.core.entity.banner.model.BannerWithPixels;
import ru.yandex.direct.core.entity.banner.model.pixels.ClientPixelProvider;
import ru.yandex.direct.core.entity.banner.service.validation.BannerValidationInfo;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign;
import ru.yandex.direct.core.entity.deal.model.Deal;
import ru.yandex.direct.core.entity.deal.model.DealAdfox;
import ru.yandex.direct.core.entity.deal.model.DealPlacement;
import ru.yandex.direct.core.entity.deal.model.StatusAdfox;
import ru.yandex.direct.core.entity.deal.service.DealService;
import ru.yandex.direct.core.entity.placements.model.Placement;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingConditionBase;
import ru.yandex.direct.core.entity.retargeting.repository.RetargetingConditionRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.function.Function.identity;
import static ru.yandex.direct.core.entity.banner.type.pixels.BannerPixelsValidator.bannerPixelsValidator;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
public class BannerWithPixelsValidatorProvider {
    private final ClientPixelProviderRepository clientPixelProviderRepository;
    private final RetargetingConditionRepository retargetingConditionRepository;
    private final AdGroupRepository adGroupRepository;
    private final DealService dealService;

    public BannerWithPixelsValidatorProvider(ClientPixelProviderRepository clientPixelProviderRepository,
                                                RetargetingConditionRepository retargetingConditionRepository,
                                                AdGroupRepository adGroupRepository,
                                                DealService dealService) {
        this.clientPixelProviderRepository = clientPixelProviderRepository;
        this.retargetingConditionRepository = retargetingConditionRepository;
        this.adGroupRepository = adGroupRepository;
        this.dealService = dealService;
    }

    public Validator<BannerWithPixels, Defect> validator(BannersOperationContainer container,
                                                         List<? extends BannerWithPixels> banners) {
        return validator(new BannerWithPixelsValidationContainerBannerOperationAdapter(container), banners);
    }

    /**
     * Метод возвращает IdentityHashMap
     */
    public Map<BannerWithPixels, NewPixelsValidationContext> collectPixelsValidationContext(
            BannersOperationContainer container, List<? extends BannerWithPixels> banners) {
        return collectPixelsValidationContext(new BannerWithPixelsValidationContainerBannerOperationAdapter(container), banners);
    }

    public Validator<BannerWithPixels, Defect> validator(BannerWithPixelsValidationContainer container,
                                                         List<? extends BannerWithPixels> banners) {
        var pixelsValidationContextMap = collectPixelsValidationContext(container, banners);
        return banner -> {
            NewPixelsValidationContext context = pixelsValidationContextMap.get(banner);
            ModelItemValidationBuilder<BannerWithPixels> builder = ModelItemValidationBuilder.of(banner);
            builder.item(BannerWithPixels.PIXELS)
                    .checkBy(bannerPixelsValidator(context));
            return builder.getResult();
        };
    }

    /**
     * Метод возвращает IdentityHashMap
     */
    public Map<BannerWithPixels, NewPixelsValidationContext> collectPixelsValidationContext(
            BannerWithPixelsValidationContainer container, List<? extends BannerWithPixels> banners) {
        List<PixelPermissionInfo> permissionInfosForClient = getPermissionInfosForClient(container);
        var campaignTypeByCampaignIdMap = getCampaignTypeMap(container);
        var retargetingConditionsByAdGroupIdMap = getRetConditionsByAdGroupIds(container);
        var retConditionsByRetConditionIds = getRetConditionsByIds(container);
        var placementByCampaignIdMap = getPlacementsByCampaignIdsMap(container);
        Set<Long> cpmAdGroupsWithKeywords = whichCpmAdGroupsCanTargetKeywords(container.getShard(),
                mapList(container.getUniqueAdGroups(), AdGroupForBannerOperation::getId));

        Map<BannerWithPixels, NewPixelsValidationContext> res = new IdentityHashMap<>();
        for (BannerWithPixels banner : banners) {
            var adGroup = container.getAdGroup(banner);
            var adGroupId = adGroup.getId();
            var campaignId = adGroup.getCampaignId();

            var bannerValidationInfo = container.getBannersValidationInfo(banner);
            var retConditionsFromDb = retargetingConditionsByAdGroupIdMap.getOrDefault(adGroupId, emptyList());
            var retConditionCollected = collectRetargetingConditions(
                    retConditionsFromDb,
                    bannerValidationInfo,
                    retConditionsByRetConditionIds);
            var hasKeywordsOnAdGroup = getHasKeywordsOnAdGroup(bannerValidationInfo, cpmAdGroupsWithKeywords,
                    adGroupId);
            NewPixelsValidationContext cntx = new NewPixelsValidationContext(
                    permissionInfosForClient,
                    campaignTypeByCampaignIdMap.get(campaignId),
                    nvl(placementByCampaignIdMap.get(campaignId), emptyList()),
                    retConditionCollected,
                    hasKeywordsOnAdGroup,
                    adGroup.getType());
            res.put(banner, cntx);
        }
        return res;
    }

    //список условий ретаргетинга по группе
    private Map<Long, List<RetargetingCondition>> getRetConditionsByAdGroupIds(BannerWithPixelsValidationContainer container) {
        List<Long> adGroupIds = mapList(container.getUniqueAdGroups(), AdGroupForBannerOperation::getId);
        return retargetingConditionRepository.getRetConditionsByAdGroupIds(container.getShard(),
                container.getClientId(), adGroupIds);
    }

    private Map<Long, RetargetingCondition> getRetConditionsByIds(BannerWithPixelsValidationContainer container) {
        if (container.getBannersValidationInfoMap() == null) {
            return emptyMap();
        }
        List<Long> retConditionIds = StreamEx.of(container.getBannersValidationInfoMap().values())
                .map(BannerValidationInfo::getRetConditionIds)
                .nonNull()
                .toFlatList(t -> t);
        return StreamEx.of(retargetingConditionRepository.getConditions(container.getShard(), retConditionIds))
                .toMap(RetargetingConditionBase::getId, identity());
    }

    /**
     * @param retConditionsFromDb            данные по ретаргетингам из базы
     * @param bannerValidationInfo           текущее значение экземпляра BannerValidationInfo
     * @param retConditionsByRetConditionIds мапа идентификаторов условий ретаргетинга в сами условия
     */
    private List<RetargetingCondition> collectRetargetingConditions(
            List<RetargetingCondition> retConditionsFromDb,
            @Nullable BannerValidationInfo bannerValidationInfo,
            Map<Long, RetargetingCondition> retConditionsByRetConditionIds) {
        if (bannerValidationInfo != null && bannerValidationInfo.getRetargetingConditions() != null) {
            return bannerValidationInfo.getRetargetingConditions();
        }
        if (bannerValidationInfo != null && bannerValidationInfo.getRetConditionIds() != null) {
            return mapList(bannerValidationInfo.getRetConditionIds(), retConditionsByRetConditionIds::get);
        }
        return retConditionsFromDb;
    }

    private Boolean getHasKeywordsOnAdGroup(BannerValidationInfo bannerValidationInfo,
                                            Set<Long> cpmAdGroupsWithKeywords,
                                            Long adGroupId) {
        if (bannerValidationInfo != null && bannerValidationInfo.getCpmAdGroupWithKeywords() != null) {
            return bannerValidationInfo.getCpmAdGroupWithKeywords();
        }

        return cpmAdGroupsWithKeywords.contains(adGroupId);
    }

    private Map<Long, CampaignType> getCampaignTypeMap(BannerWithPixelsValidationContainer container) {
        return StreamEx.of(container.getCampaigns())
                .mapToEntry(CommonCampaign::getId, CommonCampaign::getType)
                .distinctKeys()
                .toMap();
    }

    //список placement'ов сделок, привязанных к кампании
    private Map<Long, List<Placement>> getPlacementsByCampaignIdsMap(BannerWithPixelsValidationContainer container) {
        return getPlacementsByCampaignIdsMap(container.getShard(), container.getClientId(), getCampaignIds(container));
    }

    private Set<Long> getCampaignIds(BannerWithPixelsValidationContainer container) {
        return listToSet(container.getCampaigns(), CommonCampaign::getId);
    }

    //информация о разрешённых клиенту пикселях
    private List<PixelPermissionInfo> getPermissionInfosForClient(BannerWithPixelsValidationContainer container) {
        List<ClientPixelProvider> clientPixelProviders =
                clientPixelProviderRepository.getClientPixelProviders(container.getShard(),
                        container.getClientId().asLong());
        return mapList(clientPixelProviders, PixelPermissionInfo::fromDb);
    }

    /**
     * Метод получения информации о том, есть ли кейворды, привязанные к данным адгруппам
     *
     * @return мапа id группы объявлений - true, если это CPM-группа с типом нацеливаний "ключевые фразы", иначе false
     */
    private Set<Long> whichCpmAdGroupsCanTargetKeywords(int shard, Collection<Long> adGroupIds) {
        List<AdGroup> adGroupsWithTypes = adGroupRepository.getAdGroups(shard, adGroupIds);
        return StreamEx.of(adGroupsWithTypes)
                .select(CpmBannerAdGroup.class)
                .filter(cpmag -> cpmag.getCriterionType() == CriterionType.KEYWORD)
                .map(AdGroup::getId)
                .toSet();
    }

    /**
     * Метод получение списка плейсментов сделок, связанных с кампанией, по id кампаний
     */
    private Map<Long, List<Placement>> getPlacementsByCampaignIdsMap(int shard,
                                                                     ClientId clientId,
                                                                     Set<Long> campaignIds) {
        Map<Long, List<Long>> dealsByCampaignIds = dealService.getLinkedDealsByCampaignsIdsInShard(shard, campaignIds);
        List<Deal> dealsForCampaigns =
                dealService.getDealsByClientForAgency(shard, clientId, StreamEx.of(dealsByCampaignIds.values())
                        .toFlatList(identity()));
        Map<Long, Deal> dealByDealIdMap = StreamEx.of(dealsForCampaigns)
                .filter(t -> t.getAdfoxStatus() == StatusAdfox.CREATED || t.getAdfoxStatus() == StatusAdfox.ACTIVE)
                .toMap(DealAdfox::getId, identity());
        Map<Long, Placement> placementsByPageIdsMap =
                dealService.getPlacementsMapByDeals(dealByDealIdMap.values());
        Map<Long, List<Placement>> placementsByDealIdMap =
                EntryStream.of(dealByDealIdMap)
                        .mapValues(DealAdfox::getPlacements)
                        .filterValues(Objects::nonNull)
                        .mapValues(t -> mapList(t, DealPlacement::getPageId))
                        .mapValues(t -> mapList(t, placementsByPageIdsMap::get))
                        .toMap();
        return EntryStream.of(dealsByCampaignIds)
                .filterValues(Objects::nonNull)
                .mapValues(t -> StreamEx.of(t).flatCollection(placementsByDealIdMap::get).toList())
                .toMap();
    }
}
