package ru.yandex.direct.core.entity.banner.type.pixels;

import java.util.List;

import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.banner.model.pixels.ClientPixelProvider;
import ru.yandex.direct.core.entity.banner.model.pixels.CriterionType;
import ru.yandex.direct.core.entity.banner.model.pixels.PixelCampaignType;
import ru.yandex.direct.core.entity.banner.model.pixels.Provider;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;

import static ru.yandex.direct.dbschema.ppc.Tables.CLIENT_PIXEL_PROVIDERS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;

@Repository
public class ClientPixelProviderRepository {

    private final DslContextProvider dslContextProvider;
    public final JooqMapperWithSupplier<ClientPixelProvider> clientPixelProvidersMapper;

    public ClientPixelProviderRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
        clientPixelProvidersMapper = createClientPixelProvidersMapper();
    }

    /**
     * Получение из таблицы CLIENT_PIXEL_PROVIDERS по идентификатору клиента выданных ему разрешений
     * на добавляемые пиксели
     *
     * @param shard    Номер шарда
     * @param clientId id клиента
     * @return информация о разрешенных клиенту системах аудита
     */
    public List<ClientPixelProvider> getClientPixelProviders(int shard, Long clientId) {
        return dslContextProvider.ppc(shard)
                .select(CLIENT_PIXEL_PROVIDERS.CLIENT_ID, CLIENT_PIXEL_PROVIDERS.PROVIDER,
                        CLIENT_PIXEL_PROVIDERS.CAMPAIGN_TYPE, CLIENT_PIXEL_PROVIDERS.CRITERION_TYPE)
                .from(CLIENT_PIXEL_PROVIDERS)
                .where(CLIENT_PIXEL_PROVIDERS.CLIENT_ID.eq(clientId))
                .fetch()
                .map(clientPixelProvidersMapper::fromDb);
    }

    private static JooqMapperWithSupplier<ClientPixelProvider> createClientPixelProvidersMapper() {
        return JooqMapperWithSupplierBuilder.builder(ClientPixelProvider::new)
                .map(convertibleProperty(ClientPixelProvider.PROVIDER, CLIENT_PIXEL_PROVIDERS.PROVIDER,
                        Provider::fromSource, Provider::toSource))
                .map(convertibleProperty(ClientPixelProvider.PIXEL_CAMPAIGN_TYPE, CLIENT_PIXEL_PROVIDERS.CAMPAIGN_TYPE,
                        PixelCampaignType::fromSource, PixelCampaignType::toSource))
                .map(convertibleProperty(ClientPixelProvider.CRITERION_TYPE,
                        CLIENT_PIXEL_PROVIDERS.CRITERION_TYPE,
                        CriterionType::fromSource, CriterionType::toSource))
                .build();
    }
}
