package ru.yandex.direct.core.entity.banner.type.pixels;

import java.util.Map;

import javax.annotation.Nullable;

import com.google.common.collect.ImmutableMap;

import ru.yandex.direct.core.entity.banner.model.pixels.CriterionType;

import static ru.yandex.direct.utils.CommonUtils.nvl;

/**
 * Инвентарный тип баннера
 */
public enum InventoryType {
    NOT_DEAL,
    PRIVATE_CONDITIONS_FOREIGN_INVENTORY,
    PUBLIC_CONDITIONS_FOREIGN_INVENTORY,
    YANDEX_INVENTORY,
    UNKNOWN_INVENTORY;

    @Nullable
    public static InventoryType fromSource(CriterionType value) {
        if (value == null) {
            return null;
        }
        switch (value) {
            case NONE:
                return NOT_DEAL;
            case PRIVATE_:
                return PRIVATE_CONDITIONS_FOREIGN_INVENTORY;
            case PUBLIC_:
                return PUBLIC_CONDITIONS_FOREIGN_INVENTORY;
            case YANDEX:
                return YANDEX_INVENTORY;
            default:
                throw new IllegalStateException("No such value: " + value);
        }
    }

    @Nullable
    public static CriterionType toSource(InventoryType value) {
        if (value == null) {
            return null;
        }
        switch (value) {
            case NOT_DEAL:
                return CriterionType.NONE;
            case PRIVATE_CONDITIONS_FOREIGN_INVENTORY:
                return CriterionType.PRIVATE_;
            case PUBLIC_CONDITIONS_FOREIGN_INVENTORY:
                return CriterionType.PUBLIC_;
            case YANDEX_INVENTORY:
                return CriterionType.YANDEX;
            case UNKNOWN_INVENTORY:
                //значение UNKNOWN_INVENTORY может получиться только при вычислении инвентарного типа по баннеру
                //на этапе валидации и в базу записываться не должно
                throw new IllegalStateException("No equivalent type for UNKNOWN_INVENTORY in database");
            default:
                throw new IllegalStateException("No such value: " + value);
        }
    }

    private static final Map<PixelProvider, InventoryType> DEFAULT_INVENTORY_TYPE_BY_PROVIDER =
            ImmutableMap.<PixelProvider, InventoryType>builder()
                    .put(PixelProvider.YANDEXAUDIENCE, InventoryType.PRIVATE_CONDITIONS_FOREIGN_INVENTORY)
                    .put(PixelProvider.WEBORAMA, InventoryType.YANDEX_INVENTORY)
                    .put(PixelProvider.ADRIVER, InventoryType.YANDEX_INVENTORY)
                    .put(PixelProvider.SIZMEK, InventoryType.YANDEX_INVENTORY)
                    .put(PixelProvider.DCM, InventoryType.YANDEX_INVENTORY)
                    .put(PixelProvider.GEMIUS, InventoryType.YANDEX_INVENTORY)
                    .put(PixelProvider.TNS, InventoryType.YANDEX_INVENTORY)
                    .put(PixelProvider.ADJUST, InventoryType.YANDEX_INVENTORY)
                    .put(PixelProvider.MAIL_RU_TOP_100, InventoryType.PUBLIC_CONDITIONS_FOREIGN_INVENTORY)
                    .build();

    /**
     * Неизвестный тип инвентаря в зависимости от типа провайдера приравниваем к одному из других типов
     */
    public static InventoryType recalculateInventoryTypeIfUnknown(PixelProvider pixelProvider,
            InventoryType inventoryType)
    {
        if (inventoryType == InventoryType.UNKNOWN_INVENTORY) {
            return nvl(DEFAULT_INVENTORY_TYPE_BY_PROVIDER.get(pixelProvider), InventoryType.UNKNOWN_INVENTORY);
        }
        return inventoryType;
    }
}
