package ru.yandex.direct.core.entity.banner.type.pixels;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import ru.yandex.direct.core.entity.banner.model.pixels.Provider;

import static java.util.Arrays.asList;
import static org.thymeleaf.util.SetUtils.singletonSet;
import static ru.yandex.direct.validation.constraint.StringConstraints.validHref;
import static ru.yandex.misc.io.http.UrlUtils.extractHost;

/**
 * Класс, хранящий информацию о провайдере пикселя аудита
 */
public enum PixelProvider {
    TNS("TNS", "(?:\\w+\\.)?tns-counter.ru"),
    ADFOX("Adfox", "(?:ads\\.adfox\\.ru|amc\\.yandex\\.ru|mc\\.admetrica\\.ru)"),
    WEBORAMA("Weborama", "wcm(?:(?:-ru\\.frontend)|\\.solution)\\.weborama\\.fr"),
    ADRIVER("Adriver", "ad\\.adriver\\.ru"),
    SIZMEK("Sizmek", "bs\\.serving-sys\\.(com|ru)"),
    DCM("DCM", "ad\\.doubleclick\\.net"),
    GEMIUS("Gemius", "gdeby\\.hit\\.gemius\\.pl"),
    YANDEXAUDIENCE("Yandex.Audience", "mc\\.yandex\\.ru"),
    MEDIASCOPE("Mediascope", "(?:\\{DRND}\\.)?verify\\.yandex\\.ru"),
    MAIL_RU_TOP_100("Mail.ru top-100", "(?:[a-z0-9A-Z\\-]+\\.)?mail\\.ru"),
    DV("DoubleVerify", "tps\\.doubleverify\\.com"),
    IAS("IAS", "pixel\\.adsafeprotected\\.com"),
    APPSFLYER("APPSFLYER", "impression\\.appsflyer\\.com"),
    ADLOOX("adloox", "pixel\\.adlooxtracking\\.(com|ru)"),
    ADJUST("Adjust", "view\\.adjust\\.com");

    public static final String MEDIASCOPE_RANDOM_PREFIX = "{DRND}.";
    private final String providerName;
    //регулярное выражение на соответствие хоста url пикселя конкретному провайдеру
    private final Pattern expectedHostPattern;
    //регулярное выражение на проверку того, что пользователь не указал два раза подряд пиксель в одной строке
    //если встретилось два раза, возвращаем null в качестве провайдера
    private final Pattern forbiddenDuplicateRegexp;

    /**
     * Содержит типы инвентаря, разрешенные провайдерам
     */
    private static final Map<PixelProvider, Set<InventoryType>> PERMISSIONS_TABLE =
            constructPermissionsTable();

    private static final Pattern HTTP_PATTERN = Pattern.compile("^https://");
    private static final Pattern RANDOM_PLACEHOLDER_PATTERN = Pattern.compile("(%(?:25)?)(?i:(?:aw_)?random)\\1");

    PixelProvider(String providerName, String hostNamePattern) {
        this.providerName = providerName;
        this.expectedHostPattern = Pattern.compile("^" + hostNamePattern + "$");
        this.forbiddenDuplicateRegexp = Pattern.compile(hostNamePattern);
    }

    public String getProviderName() {
        return providerName;
    }

    @Nullable
    public static PixelProvider fromSource(Provider value) {
        if (value == null) {
            return null;
        }
        switch (value) {
            case ADRIVER:
                return ADRIVER;
            case SIZMEK:
                return SIZMEK;
            case DCM:
                return DCM;
            case WEBORAMA:
                return WEBORAMA;
            case YNDX_AUDIENCE:
                return YANDEXAUDIENCE;
            case GEMIUS:
                return GEMIUS;
            case TNS:
                return TNS;
            case MEDIASCOPE:
                return MEDIASCOPE;
            case ADJUST:
                return ADJUST;
            case MAIL_RU_TOP_100:
                return MAIL_RU_TOP_100;
            case DV:
                return DV;
            case IAS:
                return IAS;
            case APPSFLYER:
                return APPSFLYER;
            case ADLOOX:
                return ADLOOX;
            default:
                throw new IllegalStateException("No such value: " + value);
        }
    }

    @Nullable
    public static Provider toSource(PixelProvider value) {
        if (value == null) {
            return null;
        }
        switch (value) {
            case ADRIVER:
                return Provider.ADRIVER;
            case SIZMEK:
                return Provider.SIZMEK;
            case DCM:
                return Provider.DCM;
            case WEBORAMA:
                return Provider.WEBORAMA;
            case YANDEXAUDIENCE:
                return Provider.YNDX_AUDIENCE;
            case GEMIUS:
                return Provider.GEMIUS;
            case TNS:
                return Provider.TNS;
            case MEDIASCOPE:
                return Provider.MEDIASCOPE;
            case ADJUST:
                return Provider.ADJUST;
            case MAIL_RU_TOP_100:
                return Provider.MAIL_RU_TOP_100;
            case DV:
                return Provider.DV;
            case IAS:
                return Provider.IAS;
            case APPSFLYER:
                return Provider.APPSFLYER;
            case ADLOOX:
                return Provider.ADLOOX;
            default:
                throw new IllegalStateException("No such value: " + value);
        }
    }

    /**
     * @return Является ли данный пиксель пикселем Я.Аудиторий
     */
    public Boolean isYaAudiencePixelProvider() {
        return this.equals(YANDEXAUDIENCE);
    }

    /**
     * Возвращает типы инвентаря, разрешенные данному провайдеру
     */
    @Nonnull
    public Set<InventoryType> getAllowedInventoryTypes() {
        return PERMISSIONS_TABLE.get(this);
    }

    /**
     * Возвращает провайдера пикселя по url пикселя
     */
    @Nullable
    public static PixelProvider fromUrl(@Nonnull String pixelUrl) {
        String purePixelUrl = pixelUrl;
        String host = extractHost(pixelUrl);

        // Для Mediascope можно указывать пиксель в виде невалидного урла - https://{DRND}.verify.yandex.ru/...
        // На такой случай мы выкусываем "{DRND}." из исходного урла и проверяем то, что осталось
        if (host.startsWith(MEDIASCOPE_RANDOM_PREFIX)) {
            int index = pixelUrl.indexOf(MEDIASCOPE_RANDOM_PREFIX);
            purePixelUrl = pixelUrl.substring(0, index) + pixelUrl.substring(index + MEDIASCOPE_RANDOM_PREFIX.length());
        }

        //сначала проверяем на то, что передана ссылка
        if (validHref().apply(purePixelUrl) != null) {
            return null;
        }

        //Проверяем, что начинается с https
        if (!HTTP_PATTERN.matcher(purePixelUrl).find()) {
            return null;
        }

        //Теперь проверим, что хост пикселя относится к одному из допустимых провайдеров
        //Вернём первый подошедший
        PixelProvider[] pixelProviders = PixelProvider.values();
        for (PixelProvider pixelProvider : pixelProviders) {
            //проверяем, сколько раз в строке встречается данный провайдер
            Matcher duplicateMatcher = pixelProvider.forbiddenDuplicateRegexp.matcher(purePixelUrl);
            int numDuplicates = 0;
            while (duplicateMatcher.find()) {
                numDuplicates += 1;
                //если встретился более одного раза, вернём null
                if (numDuplicates > 1) {
                    return null;
                }
            }

            if (pixelProvider.expectedHostPattern.matcher(host).find()) {
                // для Mediascope проверяем не только хост, но и другие части ссылки,
                // и не проверяем RANDOM_PLACEHOLDER_PATTERN
                if (pixelProvider == MEDIASCOPE) {
                    if (verifyMediascopeUrl(purePixelUrl)) {
                        return pixelProvider;
                    } else {
                        return null;
                    }
                }

                //проверяем, что передан макрос RANDOM_PLACEHOLDER_PATTERN
                if (!RANDOM_PLACEHOLDER_PATTERN.matcher(purePixelUrl).find()) {
                    return null;
                }

                return pixelProvider;
            }
        }
        return null;
    }

    private static boolean verifyMediascopeUrl(String pixelUrl) {
        return List.of(pixelUrl.split("[?&]")).contains("platformid=1")
                && pixelUrl.startsWith("https://verify.yandex.ru/verify");
    }

    /**
     * Построение таблицы, по которой можно, зная провайдера пикселя аудита и инвентарный тип баннера выяснить,
     * разрешена ли операция по пикселю, или требуется ручное проставление разрешений в базе
     */
    private static Map<PixelProvider, Set<InventoryType>> constructPermissionsTable() {
        Map<PixelProvider, Set<InventoryType>> result = new HashMap<>();

        result.put(YANDEXAUDIENCE, new HashSet<>(asList(InventoryType.NOT_DEAL, InventoryType.YANDEX_INVENTORY)));

        Set<InventoryType> publicInventoryTypeSet =
                singletonSet(InventoryType.PUBLIC_CONDITIONS_FOREIGN_INVENTORY);
        asList(ADRIVER, WEBORAMA, SIZMEK, DCM, GEMIUS, TNS, MEDIASCOPE, ADJUST, DV, IAS, APPSFLYER, ADLOOX)
                .forEach(provider -> {
            result.put(provider, publicInventoryTypeSet);
        });

        Set<InventoryType> allInventoryTypesSet = new HashSet<>(asList(InventoryType.values()));
        result.put(ADFOX, allInventoryTypesSet);

        Set<InventoryType> emptyInventoryTypesSet = new HashSet<>();
        result.put(MAIL_RU_TOP_100, emptyInventoryTypesSet);
        return result;
    }

}
