package ru.yandex.direct.core.entity.banner.type.pixels;

import java.util.List;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.placements.model.Placement;

import static ru.yandex.direct.core.entity.banner.type.pixels.InventoryType.recalculateInventoryTypeIfUnknown;

public class PixelUtils {
    /**
     * Вычисляет провайдеров пикселей, разрешенных клиенту на данный инвентарный тип баннера.
     *
     * @param clientPermissions - разрешения на клиента (из базы)
     * @param inventoryType     инвентарный тип баннера, для которого хотим получить разрешенных провайдеров
     * @return список имен разрешенных провайдеров
     */
    public static Set<PixelProvider> getAccessibleClientProviders(List<PixelPermissionInfo> clientPermissions,
                                                                  CampaignType campaignType,
                                                                  InventoryType inventoryType)
    {
        //Получаем список провайдеров по разрешениям в базе
        List<PixelProvider> accessibleClientProviders = clientPermissions.stream()
                .filter(clientPermission -> clientPermission.getInventoryType() ==
                        recalculateInventoryTypeIfUnknown(clientPermission.getProvider(), inventoryType) &&
                        clientPermission.getCampaignType() == campaignType)
                .map(PixelPermissionInfo::getProvider)
                .collect(Collectors.toList());
        //Добавляем тех провайдеров, которые разрешены по дефолту
        //noinspection ConstantConditions
        return StreamEx.of(PixelProvider.values())
                .filter(pixelProvider -> pixelProvider
                        .getAllowedInventoryTypes()
                        .contains(inventoryType))
                .append(accessibleClientProviders)
                .distinct()
                .collect(Collectors.toSet());
    }

    /**
     * Получает инвентарный тип баннера, к которому пытаются привязать пиксель
     */
    public static InventoryType calculateInventoryInfo(List<Placement> bannerPlacements,
                                                       boolean hasPrivateSegments, boolean isCpmDeal) {
        if (!isCpmDeal) {
            return InventoryType.NOT_DEAL;
        }
        InventoryType inventoryTypeFromPlacements = getInventoryTypeByPlacements(bannerPlacements);
        if (inventoryTypeFromPlacements != null) {
            return inventoryTypeFromPlacements;
        }
        if (hasPrivateSegments) {
            return InventoryType.PRIVATE_CONDITIONS_FOREIGN_INVENTORY;
        } else {
            return InventoryType.PUBLIC_CONDITIONS_FOREIGN_INVENTORY;
        }
    }

    /**
     * Пытается установить InventoryType по списку Placement'ов сделок, привязанных к кампании баннера
     *
     * @return UNKNOWN_INVENTORY, если неизвестный (смешанный) тип
     * YANDEX_INVENTORY, если все площадки яндекса
     * null, если только чужой инвентарь
     */
    private static InventoryType getInventoryTypeByPlacements(List<Placement> bannerPlacements) {
        if (bannerPlacements.contains(null)) {
            return InventoryType.UNKNOWN_INVENTORY;
        }
        Predicate<Placement> isYandexPlacement = placement -> placement.getIsYandexPage() == 1;
        if (bannerPlacements.isEmpty()) {
            return InventoryType.UNKNOWN_INVENTORY;
        } else if (bannerPlacements.stream().allMatch(isYandexPlacement)) {
            return InventoryType.YANDEX_INVENTORY;
        } else if (bannerPlacements.stream().anyMatch(isYandexPlacement)) {
            return InventoryType.UNKNOWN_INVENTORY;
        } else {
            return null;
        }
    }
}
