package ru.yandex.direct.core.entity.banner.type.pricepackage;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.banner.model.BannerWithPricePackage;
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository;
import ru.yandex.direct.core.entity.campaign.model.CpmPriceCampaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository;
import ru.yandex.direct.core.entity.campaign.service.CampaignWithPricePackageUtils;
import ru.yandex.direct.core.entity.creative.model.Creative;
import ru.yandex.direct.core.entity.creative.repository.CreativeRepository;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Collections.emptyList;
import static java.util.function.Function.identity;
import static ru.yandex.direct.core.entity.campaign.service.CampaignWithPricePackageUtils.isCampaignFullWithBanners;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;


/**
 * Валидатор проверяет, что исключение баннеров excludeBanners не нарушает полноту прайсововй кампании.
 * Кампания считается полной, если в её дефолтной группе есть хотя бы один баннер
 * под каждый формат морды (campaign.flightTargetingsSnapshot.viewTypes).
 * Если кампания уже крутится в БК, то исключение каких то баннеров может нарушить полноту => кампания остановится.
 * <p>
 * Разрешается исключать баннеры (условия работают по ИЛИ):
 * - в специфичных группах
 * - в кампаниях которые ещё не запустились в БК
 * - если исключение баннеров (в совокунпности) на нарушают полноты кампании
 */
public abstract class BannerWithPricePackageFullnessValidatorBase<T> implements Validator<T, Defect> {

    private final CampaignTypedRepository campaignTypedRepository;
    private final AdGroupRepository adGroupRepository;
    private final BannerTypedRepository bannerRepository;
    private final CreativeRepository creativeRepository;

    private final Set<Long> breakingBannerIds;

    BannerWithPricePackageFullnessValidatorBase(
            int shard,
            List<BannerWithPricePackage> excludeBanners,
            CampaignTypedRepository campaignTypedRepository,
            AdGroupRepository adGroupRepository,
            BannerTypedRepository bannerRepository,
            CreativeRepository creativeRepository) {
        this.campaignTypedRepository = campaignTypedRepository;
        this.adGroupRepository = adGroupRepository;
        this.bannerRepository = bannerRepository;
        this.creativeRepository = creativeRepository;

        this.breakingBannerIds = prepareBreakingBannerIds(shard, excludeBanners);
    }

    private Set<Long> prepareBreakingBannerIds(int shard, List<BannerWithPricePackage> excludeBanners) {
        Set<Long> bannerIds = listToSet(excludeBanners, BannerWithPricePackage::getId);
        Set<Long> adGroupIds = listToSet(excludeBanners, BannerWithPricePackage::getAdGroupId);
        Set<Long> campaignIds = listToSet(excludeBanners, BannerWithPricePackage::getCampaignId);

        Map<Long, CpmPriceCampaign> campaignsToCheck = getStartedCpmPriceCampaigns(shard, campaignIds);
        Map<Long, Long> campaignIdToDefaultAdGroupIdToCheck =
                getCampaignIdToDefaultAdGroupIdToCheck(shard, campaignsToCheck, adGroupIds);
        Set<Long> adGroupIdsToCheck = new HashSet<>(campaignIdToDefaultAdGroupIdToCheck.values());

        Map<Long, List<BannerWithPricePackage>> remainingBannersInCampaigns =
                getRemainingBannersInDefaultAdGroupByCampaignId(shard, bannerIds, campaignIdToDefaultAdGroupIdToCheck);
        Set<Long> remainingBannerIds = remainingBannersInCampaigns.values().stream()
                .flatMap(List::stream)
                .map(BannerWithPricePackage::getId)
                .collect(Collectors.toSet());
        Map<Long, Creative> remainingBannerIdToCreative =
                creativeRepository.getCreativesByBannerIds(shard, remainingBannerIds);

        Set<Long> breakingCampaignIds = EntryStream.of(remainingBannersInCampaigns)
                .filterKeyValue((campaignId, defaultAdGroupRemainingBanners) -> !isCampaignFullWithBanners(
                        campaignsToCheck.get(campaignId), defaultAdGroupRemainingBanners, remainingBannerIdToCreative))
                .keys()
                .toSet();

        return StreamEx.of(excludeBanners)
                .filter(banner -> breakingCampaignIds.contains(banner.getCampaignId()))
                .filter(banner -> adGroupIdsToCheck.contains(banner.getAdGroupId()))
                .map(BannerWithPricePackage::getId)
                .toSet();
    }

    private Map<Long, CpmPriceCampaign> getStartedCpmPriceCampaigns(
            int shard, Set<Long> campaignIds) {
        List<CpmPriceCampaign> cmpPriceCampaigns =
                campaignTypedRepository.getSafely(shard, campaignIds, CpmPriceCampaign.class);
        return StreamEx.of(cmpPriceCampaigns)
                .filter(CampaignWithPricePackageUtils::isCampaignStarted)
                .mapToEntry(CpmPriceCampaign::getId, identity())
                .toMap();
    }

    private Map<Long, Long> getCampaignIdToDefaultAdGroupIdToCheck(
            int shard, Map<Long, CpmPriceCampaign> campaignsToCheck, Set<Long> adGroupIdsToCheck) {
        Map<Long, Long> campaignIdToDefaultAdGroupId =
                adGroupRepository.getDefaultPriceSalesAdGroupIdByCampaignId(shard, campaignsToCheck.keySet());
        return EntryStream.of(campaignIdToDefaultAdGroupId)
                .filterValues(adGroupIdsToCheck::contains)
                .toMap();
    }

    private Map<Long, List<BannerWithPricePackage>> getRemainingBannersInDefaultAdGroupByCampaignId(
            int shard, Set<Long> bannerIdsToStop, Map<Long, Long> campaignIdToDefaultAdGroupId) {
        Collection<Long> defaultAdGroupIds = campaignIdToDefaultAdGroupId.values();

        Map<Long, List<BannerWithPricePackage>> bannersInDefaultAdGroups =
                StreamEx.of(bannerRepository.getBannersByGroupIds(shard, defaultAdGroupIds))
                        .select(BannerWithPricePackage.class)
                        .groupingBy(BannerWithPricePackage::getAdGroupId);

        return EntryStream.of(campaignIdToDefaultAdGroupId)
                .mapValues(defaultAdGroupId ->
                        bannersInDefaultAdGroups.getOrDefault(defaultAdGroupId, emptyList()))
                .mapValues(bannersInDefaultAdGroup ->
                        filterList(bannersInDefaultAdGroup, banner -> !bannerIdsToStop.contains(banner.getId())))
                .toMap();
    }

    abstract Constraint<T, Defect> canExcludeBanner(Set<Long> breakingBannerIds);

    @Override
    public ValidationResult<T, Defect> apply(T t) {
        return ItemValidationBuilder.<T, Defect>of(t)
                .check(canExcludeBanner(breakingBannerIds))
                .getResult();
    }
}
