package ru.yandex.direct.core.entity.banner.type.sitelink;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.jooq.DSLContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.banner.container.BannerAdditionalActionsContainer;
import ru.yandex.direct.core.entity.banner.container.BannersUpdateOperationContainer;
import ru.yandex.direct.core.entity.banner.model.Banner;
import ru.yandex.direct.core.entity.banner.model.BannerStatusModerate;
import ru.yandex.direct.core.entity.banner.model.BannerStatusSitelinksModerate;
import ru.yandex.direct.core.entity.banner.model.BannerWithSitelinksModeration;
import ru.yandex.direct.core.entity.moderation.service.ModerationService;
import ru.yandex.direct.model.AppliedChanges;

import static ru.yandex.direct.core.entity.banner.model.BannerWithSitelinksModeration.SITELINKS_SET_ID;
import static ru.yandex.direct.core.entity.banner.model.BannerWithSitelinksModeration.STATUS_MODERATE;
import static ru.yandex.direct.core.entity.banner.model.BannerWithSitelinksModeration.STATUS_SITELINKS_MODERATE;
import static ru.yandex.direct.core.entity.banner.service.BannerModerationUtils.bannerBecameDraft;
import static ru.yandex.direct.core.entity.banner.service.BannerUtils.getCampaignIdToBannerIds;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
public class DefaultBannerWithSitelinksModerationProcessor
        implements BannerWithSitelinksModerationProcessor<BannerWithSitelinksModeration> {

    private final ModerationService moderationService;

    @Autowired
    public DefaultBannerWithSitelinksModerationProcessor(ModerationService moderationService) {
        this.moderationService = moderationService;
    }

    @Override
    public Class<BannerWithSitelinksModeration> getProcessedClass() {
        return BannerWithSitelinksModeration.class;
    }

    @Override
    public void process(DSLContext dsl,
                        BannerAdditionalActionsContainer additionalActionsContainer,
                        BannersUpdateOperationContainer container,
                        List<AppliedChanges<BannerWithSitelinksModeration>> appliedChanges) {
        List<BannerWithSitelinksModeration> bannersToClearModerationData = new ArrayList<>();
        for (AppliedChanges<BannerWithSitelinksModeration> changes : appliedChanges) {
            if (!changes.hasActuallyChangedProps() && container.getModerationMode().isDefault()) {
                continue;
            }

            if (needToResetModeration(changes)) {
                changes.modify(STATUS_SITELINKS_MODERATE, BannerStatusSitelinksModerate.NEW);
            } else if (readyForModeration(changes)) {
                changes.modify(STATUS_SITELINKS_MODERATE, BannerStatusSitelinksModerate.READY);
            }

            if (needToClearModerationData(changes)) {
                bannersToClearModerationData.add(changes.getModel());
            }
        }
        clearModerationData(dsl, container, bannersToClearModerationData);
    }

    private boolean needToResetModeration(AppliedChanges<BannerWithSitelinksModeration> changes) {
        return bannerBecameDraft(changes) || changes.deleted(SITELINKS_SET_ID);
    }

    private boolean readyForModeration(AppliedChanges<BannerWithSitelinksModeration> changes) {
        if (changes.getNewValue(SITELINKS_SET_ID) == null) {
            return false;
        }

        if (changes.getNewValue(STATUS_MODERATE) == BannerStatusModerate.READY) {
            return true;
        }

        if (changes.changed(SITELINKS_SET_ID)) {
            return true;
        }

        if (changes.getNewValue(STATUS_SITELINKS_MODERATE) == BannerStatusSitelinksModerate.NO) {
            boolean hasChangesBesidesStatusSitelinksModerate =
                    (changes.getActuallyChangedProps().size() > 1) || !changes.changed(STATUS_SITELINKS_MODERATE);
            return hasChangesBesidesStatusSitelinksModerate;
        }

        return false;
    }

    private boolean needToClearModerationData(AppliedChanges<BannerWithSitelinksModeration> changes) {
        if (changes.deleted(SITELINKS_SET_ID)) {
            return true;
        }

        if (bannerBecameDraft(changes)
                && changes.replaced(SITELINKS_SET_ID)
                && changes.getOldValue(STATUS_SITELINKS_MODERATE) != BannerStatusSitelinksModerate.NEW) {
            return true;
        }

        return false;
    }

    private void clearModerationData(DSLContext dsl,
                                     BannersUpdateOperationContainer container,
                                     List<BannerWithSitelinksModeration> bannersToClearModerationData) {
        List<Long> bannerIdsSitelinksSetToModeration = mapList(bannersToClearModerationData, Banner::getId);
        Map<Long, List<Long>> campaignIdToBannerIdsWithDeletedSitelinks =
                getCampaignIdToBannerIds(bannersToClearModerationData, banner -> banner.getSitelinksSetId() == null);
        moderationService.clearSitelinksSetsModeration(dsl, container.getClientId(),
                bannerIdsSitelinksSetToModeration, campaignIdToBannerIdsWithDeletedSitelinks);
    }
}
