package ru.yandex.direct.core.entity.banner.type.title;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.adgroup.model.ContentPromotionAdgroupType;
import ru.yandex.direct.core.entity.banner.container.BannersOperationContainer;
import ru.yandex.direct.core.entity.banner.model.BannerWithFixedTitle;
import ru.yandex.direct.core.entity.banner.model.BannerWithTitle;
import ru.yandex.direct.core.entity.banner.model.ContentPromotionBanner;
import ru.yandex.direct.core.entity.banner.model.CpcVideoBanner;
import ru.yandex.direct.core.entity.banner.model.CpmBanner;
import ru.yandex.direct.core.entity.banner.model.ImageBanner;
import ru.yandex.direct.core.entity.banner.model.MobileAppBanner;
import ru.yandex.direct.core.entity.banner.model.TextBanner;
import ru.yandex.direct.core.entity.banner.service.validation.BannerTextValidator;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackage;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.banner.service.validation.BannerTextConstants.MAX_NUMBER_OF_NARROW_CHARACTERS;
import static ru.yandex.direct.core.entity.banner.service.validation.BannerTextConstraints.charsAreAllowed;
import static ru.yandex.direct.core.entity.banner.service.validation.BannerTextConstraints.stringIsNotBlank;
import static ru.yandex.direct.core.entity.banner.type.title.BannerConstantsService.MAX_LENGTH_CONTENT_PROMOTION_TITLE;
import static ru.yandex.direct.core.entity.banner.type.title.BannerConstantsService.MAX_LENGTH_TITLE_WORD;
import static ru.yandex.direct.core.entity.banner.type.title.BannerConstantsService.MAX_LENGTH_VIDEO_FRONTPAGE_TITLE;
import static ru.yandex.direct.validation.constraint.CommonConstraints.isNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.StringConstraints.maxStringLength;
import static ru.yandex.direct.validation.util.ConstraintsValidator.emptyValidator;
import static ru.yandex.direct.validation.util.ConstraintsValidator.oneConstraintValidator;

@Component
public class BannerWithTitleValidatorProvider {

    private final BannerConstantsService bannerConstantsService;

    @Autowired
    public BannerWithTitleValidatorProvider(BannerConstantsService bannerConstantsService) {
        this.bannerConstantsService = bannerConstantsService;
    }

    public Validator<BannerWithTitle, Defect> bannerWithTitleValidator(BannersOperationContainer container) {
        return bannerWithTitle -> {
            ModelItemValidationBuilder<BannerWithTitle> ivb = ModelItemValidationBuilder.of(bannerWithTitle);

            ivb.item(BannerWithTitle.TITLE)
                    .checkBy(titleValidator(bannerWithTitle, container));

            return ivb.getResult();
        };
    }

    private Validator<String, Defect> titleValidator(BannerWithTitle banner, BannersOperationContainer container) {
        boolean isUniversalCampaign = container.isUniversalCampaignBanner(banner);
        boolean useNewValidation = container.isFeatureEnabledForClient(FeatureName.INCREASE_AD_TEXT_LIMITS)
                || (isUniversalCampaign && container.isFeatureEnabledForClient(FeatureName.UNIVERSAL_CAMPAIGNS_INCREASE_AD_TEXT_LIMITS));
        int maxTitleLength = bannerConstantsService.getMaxLengthTitle(useNewValidation);

        if (banner instanceof TextBanner) {
            var builder = BannerTextValidator.builder(maxTitleLength, MAX_LENGTH_TITLE_WORD);

            if (!useNewValidation) {
                builder.withMaxNumberOfNarrowCharacters(MAX_NUMBER_OF_NARROW_CHARACTERS);
            }

            return builder.build();
        }

        if (banner instanceof CpmBanner || banner instanceof ImageBanner) {
            // если это видео на главной, то для неё специфичный лимит на заголовок и заголовок обязательный
            PricePackage pricePackage = container.getPricePackage(banner);
            boolean isControlByPackage = pricePackage != null && pricePackage.isFrontpageVideoPackage();
            if (isControlByPackage) {
                maxTitleLength = MAX_LENGTH_VIDEO_FRONTPAGE_TITLE;
            }
            if (banner.getTitle() == null && !isControlByPackage) {
                return emptyValidator();
            }

            var builder = BannerTextValidator.builder(maxTitleLength, MAX_LENGTH_TITLE_WORD);

            if (!useNewValidation) {
                builder.withMaxNumberOfNarrowCharacters(MAX_NUMBER_OF_NARROW_CHARACTERS);
            }

            return builder.build();
        }

        if (banner instanceof MobileAppBanner) {
            return BannerTextValidator.builder(maxTitleLength, MAX_LENGTH_TITLE_WORD).build();
        }

        if (banner instanceof ContentPromotionBanner) {
            ContentPromotionAdgroupType contentPromotionAdgroupType = container.getContentPromotionAdGroupType(banner);

            // невозможно провалидировать тайтл, когда тип контента группы неизвестен.
            // эта проблема должна быть поймана в другом месте.
            if (contentPromotionAdgroupType == null) {
                return emptyValidator();
            }

            switch (contentPromotionAdgroupType) {
                case VIDEO:
                    return BannerTextValidator
                            .builder(MAX_LENGTH_CONTENT_PROMOTION_TITLE, MAX_LENGTH_TITLE_WORD)
                            .withMaxNumberOfNarrowCharacters(MAX_NUMBER_OF_NARROW_CHARACTERS)
                            .build();
                case SERVICE:
                case EDA:
                    // В баннере продвижения Я.Услуг и Я.еды заголовок нигде не показываем, поэтому валидируем по-минимуму:
                    // чтоб был, чтоб был не очень длинным и не содержал запрещенных символов
                    return getWeakenedTitleValidatorContentPromotion();
            }
        }

        // в этом типе баннера тайтл является системным полем и поэтому не валидируется
        if (banner instanceof CpcVideoBanner || banner instanceof BannerWithFixedTitle) {
            return emptyValidator();
        }

        // валидатор по умолчанию запрещает устанавливать тайтл;
        // таким образом, если для нового типа баннера забыли включить валидацию тайтла,
        // это обнаружится самым простым тестом
        return oneConstraintValidator(isNull());
    }

    private Validator<String, Defect> getWeakenedTitleValidatorContentPromotion() {
        return title -> {
            ItemValidationBuilder<String, Defect> vb = ItemValidationBuilder.of(title);
            vb
                    .check(notNull())
                    .check(stringIsNotBlank())
                    .check(charsAreAllowed())
                    .check(maxStringLength(MAX_LENGTH_CONTENT_PROMOTION_TITLE));
            return vb.getResult();
        };
    }
}
