package ru.yandex.direct.core.entity.banner.type.turboapp;

import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.banner.container.BannersUpdateOperationContainer;
import ru.yandex.direct.core.entity.banner.model.BannerPrice;
import ru.yandex.direct.core.entity.banner.model.BannerTurboAppType;
import ru.yandex.direct.core.entity.banner.model.BannerWithHrefAndPriceAndTurboApp;
import ru.yandex.direct.core.entity.banner.model.BannerWithPrice;
import ru.yandex.direct.core.entity.banner.model.BannerWithTurboApp;
import ru.yandex.direct.core.entity.banner.model.TurboAppInfo;
import ru.yandex.direct.core.entity.banner.service.BannerTurboAppService;
import ru.yandex.direct.core.entity.banner.service.type.update.AbstractBannerUpdateOperationTypeSupport;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.turboapps.client.model.TurboAppInfoResponse;

import static ru.yandex.direct.core.entity.banner.type.turboapp.BannerWithHrefAndPriceAndTurboAppUtils.countBannerTurboAppType;
import static ru.yandex.direct.core.entity.banner.type.turboapp.BannerWithTurboAppConstants.NEW_BANNER_TURBO_APP_PROPERTIES;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;

@Component
@ParametersAreNonnullByDefault
public class BannerWithHrefAndPriceAndTurboAppUpdateOperationTypeSupport
        extends AbstractBannerUpdateOperationTypeSupport<BannerWithHrefAndPriceAndTurboApp> {

    private final BannerTurboAppService bannerTurboAppService;

    @Autowired
    public BannerWithHrefAndPriceAndTurboAppUpdateOperationTypeSupport(BannerTurboAppService bannerTurboAppService) {
        this.bannerTurboAppService = bannerTurboAppService;
    }

    @Override
    public Class<BannerWithHrefAndPriceAndTurboApp> getTypeClass() {
        return BannerWithHrefAndPriceAndTurboApp.class;
    }

    @Override
    public void beforeExecution(BannersUpdateOperationContainer updateContainer,
                                List<AppliedChanges<BannerWithHrefAndPriceAndTurboApp>> appliedChanges) {
        if (updateContainer.isFeatureEnabledForClient(FeatureName.TURBO_APP_ALLOWED)) {
            var bannersWithTurboAppsAllowed = filterList(appliedChanges,
                    ac -> updateContainer.getCampaign(ac.getModel()).getHasTurboApp());

            var bannersWithUpdatedHref = filterList(bannersWithTurboAppsAllowed,
                    ac -> ac.changed(BannerWithHrefAndPriceAndTurboApp.HREF));

            updateBannerTurboApps(updateContainer.getShard(), updateContainer.getClientId(), bannersWithUpdatedHref);
        }

        updateTurboAppTypes(appliedChanges);
    }

    public void updateBannerTurboApps(int shard, ClientId clientId,
                                      List<AppliedChanges<BannerWithHrefAndPriceAndTurboApp>> banners) {
        if (banners.isEmpty()) {
            return;
        }

        Map<Long, String> bannerHrefByBid = StreamEx.of(banners)
                .map(AppliedChanges::getModel)
                .filter(banner -> banner.getHref() != null)
                .toMap(BannerWithHrefAndPriceAndTurboApp::getId, BannerWithHrefAndPriceAndTurboApp::getHref);

        Map<Long, TurboAppInfoResponse> turboAppsResponse = bannerTurboAppService.getTurboAppsInfo(bannerHrefByBid);

        // Удаляем турбо-аппы для баннеров, у которых нету хрефа или ручка не вернула результат
        for (var ac : banners) {
            if (!turboAppsResponse.containsKey(ac.getModel().getId())) {
                NEW_BANNER_TURBO_APP_PROPERTIES.forEach(prop -> ac.modify(prop, null));
            }
        }

        Map<Long, TurboAppInfo> turboAppByAppId = bannerTurboAppService
                .upsertTurboApps(shard, clientId, turboAppsResponse.values());

        Map<Long, TurboAppInfo> turboAppByBid = EntryStream.of(turboAppsResponse)
                .mapValues(TurboAppInfoResponse::getAppId)
                .mapValues(turboAppByAppId::get)
                .toMap();

        // Обновляем турбо-аппы, если ручка вернула результат
        for (var ac : banners) {
            Long bid = ac.getModel().getId();
            TurboAppInfo turboApp = turboAppByBid.get(bid);
            TurboAppInfoResponse turboAppResponse = turboAppsResponse.get(bid);

            if (turboAppResponse != null) {
                ac.modify(BannerWithTurboApp.TURBO_APP_INFO_ID, turboApp.getTurboAppInfoId());
                ac.modify(BannerWithTurboApp.TURBO_APP_CONTENT, turboAppResponse.getContent());

                BannerTurboAppType turboAppType = countBannerTurboAppType(ac.getModel().getBannerPrice());
                ac.modify(BannerWithTurboApp.TURBO_APP_TYPE, turboAppType);
            }
        }
    }

    public void updateTurboAppTypes(List<AppliedChanges<BannerWithHrefAndPriceAndTurboApp>> appliedChanges) {
        for (var ac : appliedChanges) {
            if (ac.changed(BannerWithPrice.BANNER_PRICE) &&
                    ac.getModel().getTurboAppInfoId() != null) {
                BannerPrice price = ac.getModel().getBannerPrice();
                BannerTurboAppType type = countBannerTurboAppType(price);
                ac.modify(BannerWithHrefAndPriceAndTurboApp.TURBO_APP_TYPE, type);
            }
        }
    }
}
