package ru.yandex.direct.core.entity.banner.type.turboapp;

import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.jooq.util.mysql.MySQLDSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.banner.model.TurboAppInfo;
import ru.yandex.direct.dbschema.ppc.tables.records.TurboAppsInfoRecord;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static ru.yandex.direct.dbschema.ppc.tables.BannerTurboApps.BANNER_TURBO_APPS;
import static ru.yandex.direct.dbschema.ppc.tables.TurboAppsInfo.TURBO_APPS_INFO;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;

@Repository
public class TurboAppsInfoRepository {

    private final DslContextProvider dslContextProvider;
    private final ShardHelper shardHelper;

    private static final JooqMapperWithSupplier<TurboAppInfo> MAPPER = createMapper();

    @Autowired
    public TurboAppsInfoRepository(DslContextProvider dslContextProvider, ShardHelper shardHelper) {
        this.dslContextProvider = dslContextProvider;
        this.shardHelper = shardHelper;
    }

    private static JooqMapperWithSupplier<TurboAppInfo> createMapper() {
        return JooqMapperWithSupplierBuilder.builder(TurboAppInfo::new)
                .map(property(TurboAppInfo.TURBO_APP_INFO_ID, TURBO_APPS_INFO.TURBO_APP_INFO_ID))
                .map(property(TurboAppInfo.TURBO_APP_ID, TURBO_APPS_INFO.TURBO_APP_ID))
                .map(property(TurboAppInfo.CLIENT_ID, TURBO_APPS_INFO.CLIENT_ID))
                .map(property(TurboAppInfo.CONTENT, TURBO_APPS_INFO.CONTENT))
                .build();
    }

    public List<TurboAppInfo> getTurboAppInfoByTurboAppIds(int shard, Collection<Long> turboAppIds,
                                                           Collection<ClientId> clientIds) {
        return dslContextProvider.ppc(shard)
                .select(MAPPER.getFieldsToRead())
                .from(TURBO_APPS_INFO)
                .where(TURBO_APPS_INFO.TURBO_APP_ID.in(turboAppIds))
                .and(TURBO_APPS_INFO.CLIENT_ID.in(clientIds))
                .fetch(MAPPER::fromDb);
    }

    public List<TurboAppInfo> getTurboAppInfoByInfoIds(int shard, Collection<Long> turboAppInfoIds) {
        return dslContextProvider.ppc(shard)
                .select(MAPPER.getFieldsToRead())
                .from(TURBO_APPS_INFO)
                .where(TURBO_APPS_INFO.TURBO_APP_INFO_ID.in(turboAppInfoIds))
                .fetch(MAPPER::fromDb);
    }

    public Map<Long, TurboAppInfo> getTurboAppInfoByBannerIds(int shard, Collection<Long> bannerIds) {
        return dslContextProvider.ppc(shard)
                .select(BANNER_TURBO_APPS.BID, TURBO_APPS_INFO.TURBO_APP_ID, TURBO_APPS_INFO.TURBO_APP_INFO_ID,
                        TURBO_APPS_INFO.CONTENT, TURBO_APPS_INFO.CLIENT_ID)
                .from(BANNER_TURBO_APPS)
                .join(TURBO_APPS_INFO).on(BANNER_TURBO_APPS.TURBO_APP_INFO_ID.eq(TURBO_APPS_INFO.TURBO_APP_INFO_ID))
                .where(BANNER_TURBO_APPS.BID.in(bannerIds))
                .fetchMap(BANNER_TURBO_APPS.BID, MAPPER::fromDb);
    }

    public List<TurboAppInfo> getTurboAppInfoByClientId(int shard, ClientId clientId) {
        return dslContextProvider.ppc(shard)
                .select(MAPPER.getFieldsToRead())
                .from(TURBO_APPS_INFO)
                .where(TURBO_APPS_INFO.CLIENT_ID.in(clientId.asLong()))
                .fetch(MAPPER::fromDb);
    }

    public Map<Long, TurboAppInfo> getTurboAppInfoByBannerIds(Collection<Long> bannerIds) {
        Map<Long, TurboAppInfo> result = new HashMap<>();
        shardHelper.forEachShard(shard -> result.putAll(getTurboAppInfoByBannerIds(shard, bannerIds)));
        return result;
    }

    public void addOrUpdateTurboAppInfo(int shard, Collection<TurboAppInfo> turboAppInfoList) {
        if(turboAppInfoList.isEmpty()) {
            return;
        }
        generateAndSetTurboAppInfoIds(turboAppInfoList);
        InsertHelper<TurboAppsInfoRecord> helper = new InsertHelper<>(dslContextProvider.ppc(shard), TURBO_APPS_INFO);
        helper.addAll(MAPPER, turboAppInfoList);
        if (helper.hasAddedRecords()) {
            helper.onDuplicateKeyUpdate()
                    .set(TURBO_APPS_INFO.CONTENT, MySQLDSL.values(TURBO_APPS_INFO.CONTENT));
        }
        helper.executeIfRecordsAdded();
    }

    private void generateAndSetTurboAppInfoIds(Collection<TurboAppInfo> turboAppInfoList) {
        List<TurboAppInfo> turboAppInfoWithoutIds = filterList(turboAppInfoList, t -> t.getTurboAppInfoId() == null);
        Iterator<Long> turboAppInfoIds = generateTurboAppInfoIds(turboAppInfoWithoutIds).iterator();
        turboAppInfoWithoutIds.forEach(turbo -> turbo.setTurboAppInfoId(turboAppInfoIds.next()));
    }

    private List<Long> generateTurboAppInfoIds(List<TurboAppInfo> turboAppInfoList) {
        return shardHelper.generateTurboAppInfoIds(turboAppInfoList.size());
    }
}
