package ru.yandex.direct.core.entity.banner.type.turbolanding;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.banner.container.BannersOperationContainer;
import ru.yandex.direct.core.entity.banner.model.Banner;
import ru.yandex.direct.core.entity.banner.model.BannerWithTurboLanding;
import ru.yandex.direct.core.entity.banner.model.CpcVideoBanner;
import ru.yandex.direct.core.entity.banner.model.CpmBanner;
import ru.yandex.direct.core.entity.banner.model.ImageBanner;
import ru.yandex.direct.core.entity.banner.model.TextBanner;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign;
import ru.yandex.direct.core.entity.turbolanding.model.TurboLanding;
import ru.yandex.direct.core.entity.turbolanding.repository.TurboLandingRepository;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.banner.model.BannerWithTurboLanding.TURBO_LANDING_ID;
import static ru.yandex.direct.core.entity.banner.type.turbolanding.BannerWithTurbolandingConstants.ALLOWED_TURBO_PRESETS_BY_GROUP_TYPE;
import static ru.yandex.direct.core.entity.banner.type.turbolanding.BannerWithTurbolandingConstants.DEFAULT_ALLOWED_PRESETS;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.inconsistentCampaignType;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapToSet;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.util.ConstraintsValidator.emptyValidator;

@Component
public class BannerWithTurboLandingValidatorProvider {

    private static final Set<CampaignType> NO_TURBOLANDING_CAMPAIGN_TYPES = Set.of(CampaignType.CPM_PRICE,
            CampaignType.CPM_DEALS, CampaignType.MOBILE_CONTENT);

    private final TurboLandingRepository turboLandingRepository;

    @Autowired
    public BannerWithTurboLandingValidatorProvider(TurboLandingRepository turboLandingRepository) {
        this.turboLandingRepository = turboLandingRepository;
    }

    Validator<BannerWithTurboLanding, Defect> bannerWithTurboLandingValidator(
            BannersOperationContainer container,
            List<BannerWithTurboLanding> unvalidatedBanners) {

        Map<Long, TurboLanding> turboLandingById = getClientTurbolandings(container, unvalidatedBanners);

        return bannerWithTurboLanding -> {
            ModelItemValidationBuilder<BannerWithTurboLanding> ivb =
                    ModelItemValidationBuilder.of(bannerWithTurboLanding);
            boolean turboLandingIsSet = bannerWithTurboLanding.getTurboLandingId() != null
                    || bannerWithTurboLanding.getTurboLandingStatusModerate() != null;
            ivb.check(isBannerFromTurbolandingCampaign(container::getCampaign), When.isTrue(turboLandingIsSet));
            ivb.item(TURBO_LANDING_ID)
                    .checkBy(turboLandingValidator(bannerWithTurboLanding, container, turboLandingById),
                            When.isValid());
            return ivb.getResult();
        };
    }

    private Validator<Long, Defect> turboLandingValidator(BannerWithTurboLanding banner,
                                                          BannersOperationContainer container,
                                                          Map<Long, TurboLanding> turboLandingById) {
        var adGroup = container.getAdGroup(banner);
        if (adGroup == null || adGroup.getType() == AdGroupType.MOBILE_CONTENT) {
            // группа может быть равна null, если указанная в баннере группа не принадлежит клиенту
            return emptyValidator();
        }

        var validPresets = ALLOWED_TURBO_PRESETS_BY_GROUP_TYPE
                .getOrDefault(adGroup.getType(), DEFAULT_ALLOWED_PRESETS);

        if (banner instanceof CpmBanner) {
            return turboLandingId -> {
                ItemValidationBuilder<Long, Defect> builder = ItemValidationBuilder.of(turboLandingId);
                builder
                        .check(notNull(), When.isTrue(adGroup.getType() == AdGroupType.CPM_GEOPRODUCT))
                        .checkBy(new BannerTurboLandingValidator(validPresets, turboLandingById));
                return builder.getResult();
            };
        }

        if (banner instanceof TextBanner ||
                banner instanceof ImageBanner ||
                banner instanceof CpcVideoBanner) {

            return turboLandingId -> {
                ItemValidationBuilder<Long, Defect> builder = ItemValidationBuilder.of(turboLandingId);
                builder
                        .checkBy(new BannerTurboLandingValidator(validPresets, turboLandingById));
                return builder.getResult();
            };
        }

        return new BannerTurboLandingValidator(validPresets, turboLandingById);
    }

    private static Constraint<BannerWithTurboLanding, Defect> isBannerFromTurbolandingCampaign(
            Function<Banner, CommonCampaign> getCampaignFunction) {
        return fromPredicate(banner ->
                {
                    CommonCampaign campaign = getCampaignFunction.apply(banner);
                    return campaign == null || !NO_TURBOLANDING_CAMPAIGN_TYPES.contains(campaign.getType());
                },
                inconsistentCampaignType());
    }

    private Map<Long, TurboLanding> getClientTurbolandings(
            BannersOperationContainer container,
            List<BannerWithTurboLanding> unvalidatedBanners) {
        Set<Long> turboLandingIds = filterAndMapToSet(unvalidatedBanners,
                b -> b.getTurboLandingId() != null,
                BannerWithTurboLanding::getTurboLandingId);
        var turboLandings = turboLandingRepository
                .getClientTurboLandingsbyId(container.getShard(), container.getClientId(), turboLandingIds);
        return listToMap(turboLandings, TurboLanding::getId);
    }

}
