package ru.yandex.direct.core.entity.banner.type.vcard;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import com.google.common.collect.ImmutableSet;
import org.jooq.DSLContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.banner.container.BannerAdditionalActionsContainer;
import ru.yandex.direct.core.entity.banner.container.BannersUpdateOperationContainer;
import ru.yandex.direct.core.entity.banner.model.Banner;
import ru.yandex.direct.core.entity.banner.model.BannerStatusModerate;
import ru.yandex.direct.core.entity.banner.model.BannerVcardStatusModerate;
import ru.yandex.direct.core.entity.banner.model.BannerWithVcardModeration;
import ru.yandex.direct.core.entity.moderation.service.ModerationService;
import ru.yandex.direct.core.entity.vcard.model.Vcard;
import ru.yandex.direct.core.entity.vcard.repository.VcardRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelProperty;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.function.Predicate.not;
import static ru.yandex.direct.core.entity.banner.model.BannerWithVcardModeration.STATUS_MODERATE;
import static ru.yandex.direct.core.entity.banner.model.BannerWithVcardModeration.VCARD_ID;
import static ru.yandex.direct.core.entity.banner.model.BannerWithVcardModeration.VCARD_STATUS_MODERATE;
import static ru.yandex.direct.core.entity.banner.service.BannerModerationUtils.bannerBecameDraft;
import static ru.yandex.direct.core.entity.banner.service.BannerTextChangeUtil.isBannerTextChangedSignificantly;
import static ru.yandex.direct.core.entity.banner.service.BannerUtils.getCampaignIdToBannerIds;
import static ru.yandex.direct.core.entity.vcard.model.Vcard.CAMPAIGN_ID;
import static ru.yandex.direct.core.entity.vcard.model.Vcard.EMAIL;
import static ru.yandex.direct.core.entity.vcard.model.Vcard.ID;
import static ru.yandex.direct.core.entity.vcard.model.Vcard.INSTANT_MESSENGER;
import static ru.yandex.direct.core.entity.vcard.model.Vcard.LAST_CHANGE;
import static ru.yandex.direct.core.entity.vcard.model.Vcard.LAST_DISSOCIATION;
import static ru.yandex.direct.core.entity.vcard.model.Vcard.UID;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
public class DefaultBannerWithVcardModerationProcessor
        implements BannerWithVcardModerationProcessor<BannerWithVcardModeration> {

    private static final Set<ModelProperty> IGNORED_VCARD_PROPERTIES = ImmutableSet.of(
            ID,
            CAMPAIGN_ID,
            UID,
            LAST_CHANGE,
            LAST_DISSOCIATION,
            EMAIL,
            INSTANT_MESSENGER);

    private final VcardRepository vcardRepository;
    private final ModerationService moderationService;

    @Autowired
    public DefaultBannerWithVcardModerationProcessor(
            VcardRepository vcardRepository,
            ModerationService moderationService) {
        this.vcardRepository = vcardRepository;
        this.moderationService = moderationService;
    }

    @Override
    public Class<BannerWithVcardModeration> getProcessedClass() {
        return BannerWithVcardModeration.class;
    }

    @Override
    public void process(DSLContext dsl,
                        BannerAdditionalActionsContainer additionalActionsContainer,
                        BannersUpdateOperationContainer container,
                        List<AppliedChanges<BannerWithVcardModeration>> appliedChanges) {
        Map<Long, Vcard> vcards = container.getVcardIdToData();

        List<BannerWithVcardModeration> bannersToClearModerationData = new ArrayList<>();
        Set<Long> vcardIdsToDissociate = new HashSet<>();

        for (AppliedChanges<BannerWithVcardModeration> changes : appliedChanges) {
            if (needToResetModeration(changes)) {
                changes.modify(VCARD_STATUS_MODERATE, BannerVcardStatusModerate.NEW);
            } else if (readyForModeration(changes, vcards)) {
                changes.modify(VCARD_STATUS_MODERATE, BannerVcardStatusModerate.READY);
            }

            if (needToClearModerationData(changes)) {
                bannersToClearModerationData.add(changes.getModel());
            }
            if (needToDissociateOldVcard(changes)) {
                vcardIdsToDissociate.add(changes.getOldValue(VCARD_ID));
            }
        }

        clearModerationData(dsl, container.getClientId(), bannersToClearModerationData);
        vcardRepository.setDissociateLastChange(dsl, vcardIdsToDissociate);
    }

    private boolean needToResetModeration(AppliedChanges<BannerWithVcardModeration> changes) {
        return bannerBecameDraft(changes) || changes.deleted(VCARD_ID);
    }

    private boolean readyForModeration(AppliedChanges<BannerWithVcardModeration> changes, Map<Long, Vcard> vcards) {
        if (changes.getNewValue(VCARD_ID) == null) {
            return false;
        }

        if (changes.changed(VCARD_ID) && vcardsDifferSignificantly(changes, vcards)) {
            return true;
        }

        if (isBannerTextChangedSignificantly(changes)) {
            return true;
        }

        if (changes.getNewValue(STATUS_MODERATE) == BannerStatusModerate.READY
                && changes.getNewValue(VCARD_STATUS_MODERATE) == BannerVcardStatusModerate.NEW) {
            return true;
        }

        return false;
    }

    private boolean vcardsDifferSignificantly(AppliedChanges<BannerWithVcardModeration> changes,
                                              Map<Long, Vcard> vcards) {
        Long newVcardId = changes.getNewValue(VCARD_ID);
        Long oldVcardId = changes.getOldValue(VCARD_ID);

        if (newVcardId == null && oldVcardId == null) {
            return false;
        } else if (newVcardId == null || oldVcardId == null) {
            return true;
        }

        Vcard newVcard = checkNotNull(vcards.get(newVcardId));
        Vcard oldVcard = checkNotNull(vcards.get(oldVcardId));

        return Vcard.allModelProperties().stream()
                .filter(not(IGNORED_VCARD_PROPERTIES::contains))
                .anyMatch(property -> !Objects.equals(property.getRaw(newVcard), property.getRaw(oldVcard)));
    }

    private boolean needToClearModerationData(AppliedChanges<BannerWithVcardModeration> changes) {
        if (changes.deleted(VCARD_ID)) {
            return true;
        }

        if (bannerBecameDraft(changes)
                && changes.replaced(VCARD_ID)
                && changes.getOldValue(VCARD_STATUS_MODERATE) != BannerVcardStatusModerate.NEW) {
            return true;
        }

        return false;
    }

    private boolean needToDissociateOldVcard(AppliedChanges<BannerWithVcardModeration> changes) {
        return changes.changed(VCARD_ID) && changes.getOldValue(VCARD_ID) != null;
    }

    private void clearModerationData(DSLContext dsl, ClientId clientId,
                                     List<BannerWithVcardModeration> bannersToClearModerationData) {
        List<Long> bannerIds = mapList(bannersToClearModerationData, Banner::getId);
        Map<Long, List<Long>> campaignIdToBannerIdsWithDeletedVcards =
                getCampaignIdToBannerIds(bannersToClearModerationData, banner -> banner.getVcardId() == null);
        moderationService.clearVcardsModeration(dsl, clientId, bannerIds, campaignIdToBannerIdsWithDeletedVcards);
    }
}
