package ru.yandex.direct.core.entity.bidmodifiers.expression;

import java.util.Collections;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.core.entity.bidmodifier.model.BidModifierExpressionOperator;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Описание параметра
 * Может быть, имеет смысл добавить ещё и "кастомный валидатор"
 */
@ParametersAreNonnullByDefault
public class ParameterInfo {
    private ParameterInfo(ParameterType type,
                          @Nullable Integer min,
                          @Nullable Integer max,
                          @Nullable Set<String> allowedValues,
                          Set<BidModifierExpressionOperator> allowedOperators) {
        this.type = type;
        this.min = min;
        this.max = max;
        this.allowedValues = allowedValues != null ? Collections.unmodifiableSet(allowedValues) : null;
        this.allowedOperators = Collections.unmodifiableSet(allowedOperators);
    }

    /**
     * Тип значений, которые может принимать параметр
     */
    private final ParameterType type;

    /**
     * Минимальное integer-значение, которое может указать пользователь; null если можно любые целые числа.
     * Используется только для параметров с type=INTEGER
     */
    private final Integer min;

    /**
     * Максимальное integer-значение, которое может указать пользователь; null, если можно любые целые числа.
     * Используется только для параметров с type=INTEGER
     */
    private final Integer max;

    /**
     * Допустимые значения (для type=ENUM обязательно, для STRING опционально, для INTEGER не используется).
     */
    private final Set<String> allowedValues;

    /**
     * Набор допустимых операций (обязателен для всех типов параметров).
     */
    private final Set<BidModifierExpressionOperator> allowedOperators;

    public ParameterType getType() {
        return type;
    }

    @Nullable
    public Integer getMin() {
        return min;
    }

    @Nullable
    public Integer getMax() {
        return max;
    }

    @Nullable
    public Set<String> getAllowedValues() {
        return allowedValues;
    }

    public Set<BidModifierExpressionOperator> getAllowedOperators() {
        return allowedOperators;
    }

    public static ParameterInfo integerParameter(Integer min, Integer max,
                                                 Set<BidModifierExpressionOperator> allowedOperators) {
        return new ParameterInfo(ParameterType.INTEGER, min, max, null, allowedOperators);
    }

    public static ParameterInfo enumParameter(Set<String> allowedValues,
                                              Set<BidModifierExpressionOperator> allowedOperators) {
        checkNotNull(allowedValues);
        return new ParameterInfo(ParameterType.ENUM, null, null, allowedValues, allowedOperators);
    }

    public static ParameterInfo stringParameter(@Nullable Set<String> allowedValues,
                                                Set<BidModifierExpressionOperator> allowedOperators) {
        return new ParameterInfo(ParameterType.STRING, null, null, allowedValues, allowedOperators);
    }
}
