package ru.yandex.direct.core.entity.bidmodifiers.repository;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Iterables;
import com.google.common.collect.Maps;
import com.google.common.collect.Multimap;
import com.google.common.collect.Multimaps;
import com.google.common.collect.Sets;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jooq.Condition;
import org.jooq.DSLContext;
import org.jooq.Operator;
import org.jooq.Record;
import org.jooq.Result;
import org.jooq.SelectConditionStep;
import org.jooq.SelectFinalStep;
import org.jooq.SelectForUpdateStep;
import org.jooq.impl.DSL;
import org.jooq.types.ULong;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierType;
import ru.yandex.direct.core.entity.bidmodifiers.container.AddedBidModifierInfo;
import ru.yandex.direct.core.entity.bidmodifiers.container.BidModifierKey;
import ru.yandex.direct.core.entity.bidmodifiers.repository.typesupport.BidModifierMultipleValuesTypeSupport;
import ru.yandex.direct.core.entity.bidmodifiers.repository.typesupport.BidModifierTypeSupport;
import ru.yandex.direct.core.entity.bidmodifiers.repository.typesupport.BidModifierTypeSupportDispatcher;
import ru.yandex.direct.core.entity.container.CampaignIdAndAdGroupIdPair;
import ru.yandex.direct.dbschema.ppc.Indexes;
import ru.yandex.direct.dbschema.ppc.tables.HierarchicalMultipliers;
import ru.yandex.direct.dbschema.ppc.tables.records.HierarchicalMultipliersRecord;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder;
import ru.yandex.direct.model.AppliedChanges;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.emptySet;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.core.entity.bidmodifiers.Constants.ALL_LEVELS;
import static ru.yandex.direct.core.entity.bidmodifiers.Constants.ALL_TYPES_INTERNAL;
import static ru.yandex.direct.core.entity.bidmodifiers.repository.mapper.Common.ALL_BIDMODIFIER_FIELDS;
import static ru.yandex.direct.core.entity.bidmodifiers.repository.mapper.Common.computeSyntheticKeyHash;
import static ru.yandex.direct.dbschema.ppc.Tables.HIERARCHICAL_MULTIPLIERS;
import static ru.yandex.direct.dbschema.ppc.Tables.PHRASES;

@Repository
@ParametersAreNonnullByDefault
public class BidModifierRepository implements BidModifierRepositoryInterface {

    private static final int BID_MODIFIERS_SELECT_CHUNK = 5000;
    private final DslContextProvider dslContextProvider;
    private final BidModifierTypeSupportDispatcher typeSupportDispatcher;

    @Autowired
    public BidModifierRepository(DslContextProvider dslContextProvider,
                                 BidModifierTypeSupportDispatcher typeSupportDispatcher) {
        this.dslContextProvider = dslContextProvider;
        this.typeSupportDispatcher = typeSupportDispatcher;
    }

    /**
     * Получает список корректировок по идентификаторам.
     * Если adGroupIdsOnly не пустой, то добавляется фильтрация по campaignIdsOnly+adGroupIdsOnly
     * Если adGroupIdsOnly пустой, но campaignIds не пустой, то добавляется фильтрация только по campaignIdsOnly
     */
    public List<BidModifier> getByIds(int shard, Multimap<BidModifierType, Long> idsByType,
                                      Set<Long> campaignIdsOnly, Set<Long> adGroupIdsOnly,
                                      Set<BidModifierType> types, Set<BidModifierLevel> levels) {
        checkArgument(!idsByType.isEmpty(), "idsByType shouldn't be empty");
        checkArgument(!types.isEmpty(), "types shouldn't be empty");
        checkArgument(!levels.isEmpty(), "levels shouldn't be empty");

        // Загружаем записи из дочерних таблиц, группируя их по hierarchical_multiplier_id
        //noinspection unchecked
        Map<Long, List<BidModifierAdjustment>> adjustmentsByBidModifierId = EntryStream.of(idsByType.asMap())
                .filterKeys(types::contains) //фильтруем типы которые ограничивают выборку
                .mapKeys(typeSupportDispatcher::getTypeSupport)
                .filterKeys(BidModifierTypeSupport::isMultiValues) //оставляем только multi-value keys
                .selectKeys(BidModifierMultipleValuesTypeSupport.class)
                .mapKeyValue(
                        (typeSupport, ids) -> (Map<Long, List<BidModifierAdjustment>>) typeSupport.getAdjustmentsByIds(
                                dslContextProvider.ppc(shard), ids))
                .flatMap(map -> map.entrySet().stream())
                .collect(toMap(Map.Entry::getKey, Map.Entry::getValue));

        // Идентификаторы сквозные, поэтому по всем запрошенным idsByType можно смело выбирать
        // hierarchical_multiplier_id
        // Зачем так: новые мобильные корректировки могут быть как просто в hierarchical_multiplier,
        // так и в mobile_multipliers_values и объединение идентификатором -- самый простой способ получить всё
        Set<Long> parentIds = EntryStream.of(idsByType.asMap())
                .values()
                .flatMap(StreamEx::of)
                // Добавляем те parentId, которые были найдены из дочерних записей
                .append(adjustmentsByBidModifierId.keySet())
                .toSet();

        // Получаем все видимые пользователю наборы из запрошенных
        List<BidModifier> bidModifiers =
                getEmptyBidModifiersByIds(shard, parentIds, campaignIdsOnly, adGroupIdsOnly, levels);

        // Соединяем bidModifiers и adjustments для multi-valued типов корректировок
        //noinspection unchecked
        EntryStream.of(bidModifiers.stream().collect(groupingBy(BidModifier::getType)))
                .mapKeys(typeSupportDispatcher::getTypeSupport)
                .selectKeys(BidModifierMultipleValuesTypeSupport.class)
                .flatMapValues(Collection::stream)
                .filterValues(modifier -> adjustmentsByBidModifierId.get(modifier.getId()) != null)
                .forKeyValue((typeSupport, modifier) -> typeSupport
                        .setAdjustments(modifier, adjustmentsByBidModifierId.get(modifier.getId())));

        return bidModifiers;
    }

    /**
     * Получает список корректировок по идентификаторам.
     */
    public List<BidModifier> getAllByIds(int shard, Multimap<BidModifierType, Long> idsByType,
                                         Set<BidModifierType> allowedTypes) {
        var all = getByIds(shard, idsByType, emptySet(), emptySet(), allowedTypes, ALL_LEVELS);
        fillAdjustments(dslContextProvider.ppc(shard), all, false);
        return all;
    }

    public List<BidModifier> getByCampaignIds(int shard, Collection<Long> campaignIds,
                                              Set<BidModifierType> types, Set<BidModifierLevel> levels) {
        return getByCampaignIds(dslContextProvider.ppc(shard), campaignIds, types, levels);
    }

    public List<BidModifier> getByCampaignIds(DSLContext dslContext, Collection<Long> campaignIds,
                                              Set<BidModifierType> types, Set<BidModifierLevel> levels) {
        List<BidModifier> bidModifiers = getEmptyBidModifiersByCampaignIds(dslContext, campaignIds, types, levels);
        fillAdjustments(dslContext, bidModifiers, false);
        return bidModifiers;
    }

    public List<BidModifier> getByAdGroupIds(int shard, Map<Long, Long> campaignIdsByAdGroupIds,
                                             Set<BidModifierType> types, Set<BidModifierLevel> levels) {
        List<BidModifier> modifiers = getEmptyBidModifiersByAdGroupIds(shard, campaignIdsByAdGroupIds, types, levels);
        fillAdjustments(dslContextProvider.ppc(shard), modifiers, false);
        return modifiers;
    }

    /**
     * Выполняет добавление указанных наборов корректировок к существующим.
     * Побочные эффекты: в modifiers могут быть установлены campaignId там, где их не было.
     * <p>
     * Должен вызываться в контексте транзакции.
     */
    public Map<BidModifierKey, AddedBidModifierInfo> addModifiers(
            DSLContext txContext,
            Collection<BidModifier> modifiers,
            Map<BidModifierKey, BidModifier> existingModifiers,
            ClientId clientId, long operatorUid) {
        // campaignId и type должны быть проставлены вызывающим кодом
        checkState(modifiers.stream().allMatch(it -> it.getCampaignId() != null && it.getType() != null));

        Set<BidModifierKey> allKeys = modifiers.stream().map(BidModifierKey::new).collect(toSet());

        // Снова загружаем все наборы элементов, соответствующих валидным, но теперь уже с блокировкой
        Map<BidModifierKey, BidModifier> lockedExistingModifiers = getBidModifiersByKeys(txContext, allKeys, true);

        // Ищем конфликты изменений
        // Обходим элементы, и если между валидацией и сохранением содержимое корректировок поменялось,
        // добавляем их в список тех, которых мы пропустим при сохранении, вернув на них затем дефект "Конфликт
        // изменений"
        Set<BidModifierKey> conflictedKeys = allKeys.stream()
                .filter(key -> hasConflict(existingModifiers.get(key), lockedExistingModifiers.get(key)))
                .collect(toSet());

        // Отфильтровываем конфликтующие элементы
        List<BidModifier> modifiersToSave =
                modifiers.stream().filter(it -> !conflictedKeys.contains(new BidModifierKey(it))).collect(toList());

        // Добавляем
        Map<BidModifierKey, AddedBidModifierInfo> addedInfoMap = typeSupportDispatcher.addAll(
                txContext, modifiersToSave, lockedExistingModifiers, clientId, operatorUid);

        // Собираем результаты
        HashMap<BidModifierKey, AddedBidModifierInfo> resultMap = new HashMap<>(addedInfoMap);
        resultMap.putAll(conflictedKeys.stream().collect(toMap(identity(), it -> AddedBidModifierInfo.notAdded())));
        return resultMap;
    }

    /**
     * Выполняет установку наборов корректировок на указанные кампании/группы объявлений.
     * Новые корректировки заменят старые. Если новых корректировок для какой-то кампании/группы указано не было,
     * но кампания/группа присутствует в affectedObjects, то с неё все корректировки будут удалены.
     * <p>
     * Возвращает набор объектов, к которым реально были применены изменения.
     */
    public Set<CampaignIdAndAdGroupIdPair> replaceModifiers(DSLContext txContext, List<BidModifier> modifiers,
                                                            Set<CampaignIdAndAdGroupIdPair> affectedObjects,
                                                            ClientId clientId, long operatorUid) {
        // campaignId и type должны быть проставлены вызывающим кодом
        checkState(modifiers.stream().allMatch(it -> it.getCampaignId() != null && it.getType() != null));
        // Собираем все возможные значения BidModifierKey для указанных наборов cid+pid
        // Они понадобятся для выборки из базы всех существующих наборов корректировок под updLock'ом
        // (чтобы работал индекс по cid+syntheticHash)
        Set<BidModifierKey> allPossibleKeys = affectedObjects.stream().flatMap(it ->
                ALL_TYPES_INTERNAL.stream().map(type -> new BidModifierKey(it.getCampaignId(), it.getAdGroupId(), type))
        ).collect(toSet());

        // Загружаем все наборы элементов с блокировкой
        Map<BidModifierKey, BidModifier> lockedExistingModifiers =
                getBidModifiersByKeys(txContext, allPossibleKeys, true);

        // Применяем изменения
        return typeSupportDispatcher.addOrReplace(txContext, modifiers, lockedExistingModifiers, clientId, operatorUid);
    }

    private boolean hasConflict(@Nullable BidModifier oldModifier, @Nullable BidModifier newModifier) {
        if (oldModifier == null) {
            return newModifier != null;
        }
        // В текущем состоянии в базе ничего нет, а раньше было
        // Но если наше состояние было валидно даже вместе с тем, что было раньше, то оно, очевидно, валидно и без него
        if (newModifier == null) {
            return false;
        }
        // Сравниваем содержимое наборов корректировок перед валидацией и перед сохранением
        return !typeSupportDispatcher.areEqual(oldModifier, newModifier);
    }

    /**
     * Генерирует несколько id для корректировок. Эти id используются для всех таблиц
     * с корректировками, т.к. id в них сквозные.
     */
    public static List<Long> generateIds(ShardHelper shardHelper, int count) {
        checkArgument(count >= 0);
        return shardHelper.generateHierarchicalMultiplierIds(count);
    }

    public Map<BidModifierKey, BidModifier> getBidModifiersByKeys(int shard, Collection<BidModifierKey> modifierKeys) {
        return getBidModifiersByKeys(dslContextProvider.ppc(shard), modifierKeys, false);
    }

    /**
     * Удаляет корректировки из наборов и наборы целиком, если в них не осталось больше корректировок (в базе).
     * Должен вызываться в рамках транзакции.
     *
     * @param bidModifiers Наборы корректировок для удаления. Удаляются все adjustment'ы, указанные в них
     */
    public void deleteAdjustments(DSLContext txContext, ClientId clientId, long operatorUid,
                                  Collection<BidModifier> bidModifiers) {
        if (bidModifiers.isEmpty()) {
            return;
        }

        // Выполняем удаление
        EntryStream.of(Multimaps.index(bidModifiers, BidModifier::getType).asMap())
                .mapKeys(typeSupportDispatcher::getTypeSupport)
                .forKeyValue((typeSupport, modifiers) ->
                        typeSupport.delete(txContext, clientId, operatorUid, modifiers));
    }

    /**
     * Возвращает пустые наборы корректировок (без значений adjustments) по идентификаторам.
     * Если adGroupIdsOnly не пустой, то добавляется фильтрация по campaignIdsOnly+adGroupIdsOnly
     * Если adGroupIdsOnly пустой, но campaignIdsOnly не пустой, то добавляется фильтрация только по campaignIdsOnly
     */
    private List<BidModifier> getEmptyBidModifiersByIds(int shard, Collection<Long> ids,
                                                        Set<Long> campaignIdsOnly, Set<Long> adGroupIdsOnly,
                                                        Set<BidModifierLevel> levels) {
        List<BidModifier> emptyBidModifiers = getEmptyBidModifiersByIds(shard, ids, levels);

        // Получаем те кампании, которые пользователь может просматривать
        Set<Long> allCampaignIds;
        Set<Long> bidModifiersCampaignIds = emptyBidModifiers.stream().map(BidModifier::getCampaignId).collect(toSet());
        if (campaignIdsOnly.isEmpty()) {
            allCampaignIds = bidModifiersCampaignIds;
        } else {
            allCampaignIds = Sets.intersection(campaignIdsOnly, bidModifiersCampaignIds);
        }

        // Отфильтровываем наборы корректировок, относящиеся к кампаниям, к которым у пользователя нет доступа
        // и к группам объявлений (если они указаны)
        if (!adGroupIdsOnly.isEmpty()) {
            return emptyBidModifiers.stream()
                    .filter(m -> allCampaignIds.contains(m.getCampaignId())
                            && m.getAdGroupId() != null && adGroupIdsOnly.contains(m.getAdGroupId()))
                    .collect(toList());
        }
        return emptyBidModifiers.stream()
                .filter(m -> allCampaignIds.contains(m.getCampaignId()))
                .collect(toList());
    }

    public List<BidModifier> getEmptyBidModifiersByCampaignIds(DSLContext dslContext, Collection<Long> campaignIds,
                                                               Set<BidModifierType> types,
                                                               Set<BidModifierLevel> levels) {
        Result<Record> records = addLevelCondition(
                dslContext
                        .select(ALL_BIDMODIFIER_FIELDS)
                        .from(HIERARCHICAL_MULTIPLIERS)
                        .where(HIERARCHICAL_MULTIPLIERS.CID.in(campaignIds))
                        .and(HIERARCHICAL_MULTIPLIERS.TYPE.in(
                                types.stream().map(BidModifierType::toSource).collect(toSet()))), levels)
                .fetch();

        Multimap<BidModifierType, Record> recordTypeMultimap =
                typeSupportDispatcher.groupRecordsByTypes(records);

        return EntryStream.of(recordTypeMultimap.asMap())
                .mapKeys(typeSupportDispatcher::getTypeSupport)
                .mapKeyValue(BidModifierTypeSupport::createEmptyBidModifiersFromRecords)
                .flatMap(Collection::stream)
                .toList();
    }

    /**
     * Добавляет через AND к переданному WHERE условие соответствия нужным уровням привязки корректировок.
     */
    private SelectConditionStep<Record> addLevelCondition(SelectConditionStep<Record> selectConditionStep,
                                                          Set<BidModifierLevel> levels) {
        if (levels.contains(BidModifierLevel.CAMPAIGN) && levels.contains(BidModifierLevel.ADGROUP)) {
            return selectConditionStep;
        }
        if (levels.contains(BidModifierLevel.CAMPAIGN)) {
            return selectConditionStep.and(HierarchicalMultipliers.HIERARCHICAL_MULTIPLIERS.PID.isNull());
        }
        checkState(levels.contains(BidModifierLevel.ADGROUP));
        return selectConditionStep.and(HierarchicalMultipliers.HIERARCHICAL_MULTIPLIERS.PID.isNotNull());
    }

    /**
     * Получить записи hierarchical_multipliers без уточнений корректировок для всех уровней
     */
    public List<BidModifier> getEmptyBidModifiersByIds(int shard, Collection<Long> ids) {
        if (ids.isEmpty()) {
            return emptyList();
        }
        return getEmptyBidModifiersByIds(shard, ids, ALL_LEVELS);
    }

    private List<BidModifier> getEmptyBidModifiersByIds(int shard, Collection<Long> ids, Set<BidModifierLevel> levels) {
        Result<Record> records = addLevelCondition(
                dslContextProvider.ppc(shard)
                        .select(ALL_BIDMODIFIER_FIELDS)
                        .from(HIERARCHICAL_MULTIPLIERS)
                        .where(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID.in(ids)), levels)
                .fetch();

        Multimap<BidModifierType, Record> recordTypeMultimap = typeSupportDispatcher.groupRecordsByTypes(records);

        return EntryStream.of(recordTypeMultimap.asMap())
                .mapKeys(typeSupportDispatcher::getTypeSupport)
                .mapKeyValue(BidModifierTypeSupport::createEmptyBidModifiersFromRecords)
                .flatMap(Collection::stream)
                .toList();
    }

    public List<BidModifier> getEmptyBidModifiersByAdGroupIds(int shard, Map<Long, Long> campaignIdsByAdGroupIds,
                                                              Set<BidModifierType> types,
                                                              Set<BidModifierLevel> levels) {
        List<Condition> conditions = campaignIdsByAdGroupIds.entrySet().stream().map(
                e -> {
                    Long adGroupId = e.getKey();
                    Long campaignId = e.getValue();
                    return HIERARCHICAL_MULTIPLIERS.CID.eq(campaignId)
                            .and(HIERARCHICAL_MULTIPLIERS.PID.eq(adGroupId));
                }
        ).collect(toList());

        Result<Record> records = addLevelCondition(
                dslContextProvider.ppc(shard)
                        .select(ALL_BIDMODIFIER_FIELDS)
                        .from(HIERARCHICAL_MULTIPLIERS)
                        .where(DSL.condition(Operator.OR, conditions))
                        .and(HIERARCHICAL_MULTIPLIERS.TYPE.in(
                                types.stream().map(BidModifierType::toSource).collect(toSet()))), levels)
                .fetch();

        Multimap<BidModifierType, Record> recordTypeMultimap =
                typeSupportDispatcher.groupRecordsByTypes(records);

        return EntryStream.of(recordTypeMultimap.asMap())
                .mapKeys(typeSupportDispatcher::getTypeSupport)
                .mapKeyValue(BidModifierTypeSupport::createEmptyBidModifiersFromRecords)
                .flatMap(Collection::stream)
                .toList();
    }

    /**
     * Получает id корректировок, привязанных к указанным группам объявлений
     *
     * @param shard      обрабатываемый шард
     * @param adGroupIds id групп объявлений, в которых ищутся корректировки
     * @return id корректировок, привязанных к группа объявлений
     */
    public Map<Long, List<Long>> getBidModifierIdsByAdGroupIds(int shard, Collection<Long> adGroupIds) {
        var campaignIds = dslContextProvider.ppc(shard)
                .select(PHRASES.CID)
                .from(PHRASES)
                .where(PHRASES.PID.in(adGroupIds))
                .fetchSet(PHRASES.CID);

        return dslContextProvider.ppc(shard)
                .select(HIERARCHICAL_MULTIPLIERS.PID, HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID)
                .from(HIERARCHICAL_MULTIPLIERS)
                .where(HIERARCHICAL_MULTIPLIERS.CID.in(campaignIds))
                .and(HIERARCHICAL_MULTIPLIERS.PID.in(adGroupIds))
                .fetchGroups(HIERARCHICAL_MULTIPLIERS.PID, HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID);
    }

    /**
     * Получает id корректировок, привязанных к указанным кампаниям
     *
     * @param shard       обрабатываемый шард
     * @param campaignIds id кампаний, в которых ищутся корректировки
     * @return id корректировок, привязанных к кампаниям
     */
    public Map<Long, List<Long>> getBidModifierIdsByCampaignIds(int shard, Collection<Long> campaignIds) {
        return dslContextProvider.ppc(shard)
                .select(HIERARCHICAL_MULTIPLIERS.CID, HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID)
                .from(HIERARCHICAL_MULTIPLIERS)
                .where(HIERARCHICAL_MULTIPLIERS.CID.in(campaignIds))
                .and(HIERARCHICAL_MULTIPLIERS.PID.isNull())
                .fetchGroups(HIERARCHICAL_MULTIPLIERS.CID, HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID);
    }

    /**
     * Получает коллекцию ключей корректировок, она в дальнейшем используется при получении полных корректировок
     *
     * @param shard          шард
     * @param bidModifierIds id корректировок (верхнего уровня/из таблицы HIERARCHICAL_MULTIPLIERS
     * @return мапа id корректировки - ключ этой корректировки
     */
    public Map<Long, BidModifierKey> getBidModifierKeysByIds(int shard, Collection<Long> bidModifierIds) {
        if (bidModifierIds.isEmpty()) {
            return emptyMap();
        }
        return dslContextProvider.ppc(shard)
                .select(HIERARCHICAL_MULTIPLIERS.CID, HIERARCHICAL_MULTIPLIERS.PID, HIERARCHICAL_MULTIPLIERS.TYPE,
                        HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID)
                .from(HIERARCHICAL_MULTIPLIERS)
                .where(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID.in(bidModifierIds))
                .fetchMap(r -> r.getValue(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID),
                        r -> new BidModifierKey(r.getValue(HIERARCHICAL_MULTIPLIERS.CID),
                                r.getValue(HIERARCHICAL_MULTIPLIERS.PID),
                                BidModifierType.fromSource(r.getValue(HIERARCHICAL_MULTIPLIERS.TYPE))));
    }

    /**
     * Возвращает список идентификаторов существующих записей конретного типа корректировки
     */
    public Collection<Long> getExistingMultiValuedBidModifiersIds(int shard, BidModifierType type, List<Long> ids) {
        var typeSupport = typeSupportDispatcher.getTypeSupport(type);
        if (!typeSupport.isMultiValues()) {
            throw new RuntimeException("unsupported type: $type");
        } else {
            var multiValuesTypeSupport = (BidModifierMultipleValuesTypeSupport<BidModifier, BidModifierAdjustment>)
                    typeSupport;
            Map<Long, List<BidModifierAdjustment>> adjustmentIdsByParentId =
                    multiValuesTypeSupport.getAdjustmentsByIds(dslContextProvider.ppc(shard), ids);
            return adjustmentIdsByParentId.values().stream()
                    .flatMap(a -> a.stream().map(aa -> aa.getId())).collect(Collectors.toList());
        }
    }


    private Map<BidModifierKey, BidModifier> getBidModifiersByKeys(DSLContext dslContext,
                                                                   Collection<BidModifierKey> modifierKeys,
                                                                   boolean updLock) {
        List<BidModifier> bidModifiers = getBidModifiersListByKeys(dslContext, modifierKeys, updLock);
        return Maps.uniqueIndex(bidModifiers, BidModifierKey::new);
    }

    /**
     * Выбирает все hierarchical_multipliers по указанным наборам (campaignId, adGroupId, type).
     */
    private List<BidModifier> getBidModifiersListByKeys(DSLContext dslContext,
                                                        Collection<BidModifierKey> itemsToLoad, boolean updLock) {
        List<BidModifier> result = new ArrayList<>();
        for (var chunk : Iterables.partition(itemsToLoad, BID_MODIFIERS_SELECT_CHUNK)) {
            result.addAll(getBidModifiersListByKeyChunked(dslContext, chunk, updLock));
        }
        return result;
    }

    private List<BidModifier> getBidModifiersListByKeyChunked(DSLContext dslContext,
                                                              Collection<BidModifierKey> itemsToLoad, boolean updLock) {
        // Используем здесь synteticKeyHash для того, чтобы при взятии updlock-блокировки
        // использовался индекс cid+synteticKeyHash, и не лочились лишние интервалы записей.
        List<Condition> conditions = itemsToLoad.stream().map(
                key ->
                        // В этом месте key.AdGroupId() может быть null
                        HIERARCHICAL_MULTIPLIERS.CID.eq(key.getCampaignId())
                                .and(HIERARCHICAL_MULTIPLIERS.SYNTETIC_KEY_HASH.eq(
                                        ULong.valueOf(computeSyntheticKeyHash(
                                                key.getType(), key.getCampaignId(), key.getAdGroupId()))))

        ).collect(toList());

        SelectFinalStep<Record> selectStep = dslContext.select(ALL_BIDMODIFIER_FIELDS)
                .from(HIERARCHICAL_MULTIPLIERS.forceIndex(Indexes.HIERARCHICAL_MULTIPLIERS_CID_HASH.getName()))
                .where(DSL.condition(Operator.OR, conditions));
        if (updLock) {
            //noinspection unchecked
            selectStep = ((SelectForUpdateStep) selectStep).forUpdate();
        }
        Result<Record> records = selectStep.fetch();

        Multimap<BidModifierType, Record> recordTypeMultimap =
                typeSupportDispatcher.groupRecordsByTypes(records);

        List<BidModifier> bidModifiers = EntryStream.of(recordTypeMultimap.asMap())
                .mapKeys(typeSupportDispatcher::getTypeSupport)
                .mapKeyValue(BidModifierTypeSupport::createEmptyBidModifiersFromRecords)
                .flatMap(Collection::stream)
                .toList();

        fillAdjustments(dslContext, bidModifiers, updLock);

        return bidModifiers;
    }

    private void fillAdjustments(DSLContext dslContext, List<BidModifier> bidModifiers, boolean updLock) {
        Multimap<BidModifierType, BidModifier> bidModifiersByType =
                Multimaps.index(bidModifiers, BidModifier::getType);
        //noinspection unchecked
        EntryStream.of(bidModifiersByType.asMap())
                .mapKeys(typeSupportDispatcher::getTypeSupport)
                .filterKeys(BidModifierTypeSupport::isMultiValues)
                .mapKeys(typeSupport -> (BidModifierMultipleValuesTypeSupport) typeSupport)
                .forKeyValue((typeSupport, items) -> typeSupport.fillAdjustments(dslContext, items, updLock));
    }

    /**
     * Обновляет значения коэффициентов на корректировках.
     *
     * @param txContext                Контекст транзакции
     * @param clientId                 id клиента. Нужен для определения страны, в которой работает клиент
     *                                 (для транслокальности в геокорректировках)
     * @param applicableAppliedChanges Применяемые изменения
     * @param bidModifiers             Наборы корректировок, которые затрагиваются этими изменениями. Нужны для того,
     *                                 чтобы для геокорректировок не ходить лишний раз в базу перед тем, как забрать.
     *                                 Сюда не обязательно передавать наборы, соответствующие только валидным элементам,
     *                                 они не будут изменяться, а только служить информацией для typeSupport'ов
     */
    public void updatePercents(DSLContext txContext, ClientId clientId, long operatorUid,
                               List<AppliedChanges<BidModifierAdjustment>> applicableAppliedChanges,
                               List<BidModifier> bidModifiers) {
        // Обновляем значения процентов
        EntryStream.of(applicableAppliedChanges.stream().collect(groupingBy(ac -> ac.getModel().getClass())))
                .mapKeys(clazz -> typeSupportDispatcher
                        .getTypeSupport(typeSupportDispatcher.getTypeByAdjustmentClass(clazz)))
                .forKeyValue((typeSupport, appliedChanges) -> {
                    List<BidModifier> modifiersOfThisType = bidModifiers.stream()
                            .filter(it -> it.getType() == typeSupport.getType())
                            .collect(toList());
                    typeSupport.updatePercents(clientId, operatorUid, appliedChanges, modifiersOfThisType, txContext);
                });
    }

    public void updateHierarchicalMultiplierEnabled(DSLContext dslContext,
                                                    List<AppliedChanges<BidModifier>> appliedChanges) {
        if (appliedChanges.isEmpty()) {
            return;
        }

        JooqUpdateBuilder<HierarchicalMultipliersRecord, BidModifier> ub =
                new JooqUpdateBuilder<>(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID, appliedChanges);

        ub.processProperty(BidModifier.ENABLED, HIERARCHICAL_MULTIPLIERS.IS_ENABLED, RepositoryUtils::booleanToLong);

        dslContext.update(HIERARCHICAL_MULTIPLIERS)
                .set(ub.getValues())
                .set(HIERARCHICAL_MULTIPLIERS.LAST_CHANGE, LocalDateTime.now())
                .where(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID.in(ub.getChangedIds()))
                .execute();
    }
}
