package ru.yandex.direct.core.entity.bidmodifiers.repository.typesupport;

import java.util.Collection;
import java.util.List;

import org.jooq.DSLContext;
import org.jooq.Record;

import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.AppliedChanges;

import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public interface BidModifierTypeSupport<TModifier extends BidModifier,
        TAdjustment extends BidModifierAdjustment> {
    /**
     * Возвращает тип корректровок, с которой работает данная реализация.
     */
    BidModifierType getType();

    /**
     * Возвращает класс набора корректировок, с которым работает данная реализация.
     */
    Class<TModifier> getBidModifierClass();

    /**
     * Создаёт пустой набор корректировок (без дочерних Adjustments) из записи БД.
     * Но если этот тип корректировки не использует дочерних таблиц, то Adjustment у результата будет
     * инициализирован сразу.
     */
    TModifier createEmptyBidModifierFromRecord(Record record);

    /**
     * Создаёт стрим пустых наборов корректировок из набора записей БД.
     */
    default List<TModifier> createEmptyBidModifiersFromRecords(Collection<Record> records) {
        return mapList(records, this::createEmptyBidModifierFromRecord);
    }

    /**
     * Удаляет указанные корректировки из наборов, а затем и сами наборы, если они опустели
     * Должен быть вызван в контексте транзакции (для multi-value реализаций это необходимо,
     * чтобы можно было консистентно удалить запись из таблицы hierarchical_multipliers после того, как
     * значения из дочерних таблиц будут удалены)
     */
    void delete(DSLContext txContext, ClientId clientId, long operatorUid, Collection<TModifier> modifiers);

    Class<TAdjustment> getAdjustmentClass();

    /**
     * Проверяет, совпадают ли наборы корректировок по содержимому.
     */
    boolean areEqual(TModifier a, TModifier b);

    /**
     * Возвращает true, если корректировки в наборах этого типа хранятся в дочерних таблицах, иначе false.
     */
    boolean isMultiValues();

    /**
     * Вытаскивает список adjustments из набора корректировок.
     */
    List<TAdjustment> getAdjustments(TModifier modifier);

    /**
     * Обновляет значения процентов на корректировках
     *
     * @param clientId       id клиента. Нужен для определения страны, в которой работает клиент
     *                       (для транслокальности в геокорректировках)
     * @param operatorUid
     * @param appliedChanges Применяемые изменения
     * @param bidModifiers   Наборы корректировок, которые затрагиваются этими изменениями. Нужны для того,
     */
    void updatePercents(ClientId clientId, long operatorUid, List<AppliedChanges<TAdjustment>> appliedChanges,
                        List<TModifier> bidModifiers, DSLContext dslContext);

    /**
     * Перед добавлением или обновлением подготовить системные поля
     */
    void prepareSystemFields(List<TModifier> bidModifier);
}
