package ru.yandex.direct.core.entity.bidmodifiers.repository.typesupport;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Maps;
import com.google.common.collect.Multimap;
import com.google.common.collect.Multimaps;
import one.util.streamex.EntryStream;
import org.jooq.DSLContext;
import org.jooq.Record;
import org.jooq.Result;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierType;
import ru.yandex.direct.core.entity.bidmodifiers.container.AddedBidModifierInfo;
import ru.yandex.direct.core.entity.bidmodifiers.container.BidModifierKey;
import ru.yandex.direct.core.entity.container.CampaignIdAndAdGroupIdPair;
import ru.yandex.direct.dbutil.model.ClientId;

import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.dbschema.ppc.Tables.HIERARCHICAL_MULTIPLIERS;

@Component
@ParametersAreNonnullByDefault
public class BidModifierTypeSupportDispatcher {
    private final Map<Class<? extends BidModifier>, BidModifierTypeSupport> typeSupportMap;
    private final Map<BidModifierType, BidModifierTypeSupport> typeSupportByTypeMap;
    private final Map<Class<? extends BidModifierAdjustment>, BidModifierTypeSupport> typeSupportByAdjustmentMap;

    @Autowired
    public BidModifierTypeSupportDispatcher(List<BidModifierTypeSupport> typeSupportList) {
        this.typeSupportMap = Maps.uniqueIndex(typeSupportList, BidModifierTypeSupport::getBidModifierClass);
        this.typeSupportByTypeMap = Maps.uniqueIndex(typeSupportList, BidModifierTypeSupport::getType);
        this.typeSupportByAdjustmentMap = Maps.uniqueIndex(typeSupportList, BidModifierTypeSupport::getAdjustmentClass);
    }

    @SuppressWarnings("unchecked")
    public boolean areEqual(BidModifier a, BidModifier b) {
        BidModifierTypeSupport typeSupport = typeSupportMap.get(a.getClass());
        return typeSupport.areEqual(a, b);
    }

    public BidModifierType getTypeOf(BidModifier modifier) {
        BidModifierTypeSupport typeSupport = typeSupportMap.get(modifier.getClass());
        return typeSupport.getType();
    }

    public BidModifierType getTypeByAdjustmentClass(
            Class<? extends BidModifierAdjustment> adjustmentClazz) {
        BidModifierTypeSupport typeSupport = typeSupportByAdjustmentMap.get(adjustmentClazz);
        return typeSupport.getType();
    }

    @SuppressWarnings("unchecked")
    public Set<CampaignIdAndAdGroupIdPair> addOrReplace(DSLContext txContext, List<BidModifier> modifiers,
                                                        Map<BidModifierKey, BidModifier> lockedExistingModifiers, ClientId clientId, long operatorUid) {
        Map<BidModifierType, List<BidModifier>> modifiersByType =
                modifiers.stream().collect(groupingBy(this::getTypeOf));

        Set<BidModifierType> allTypes = Stream.concat(
                modifiersByType.keySet().stream(),
                lockedExistingModifiers.keySet().stream().map(BidModifierKey::getType)
        ).collect(Collectors.toSet());

        Set<CampaignIdAndAdGroupIdPair> changedCampaignsAndAdGroups = new HashSet<>();
        for (BidModifierType type : allTypes) {
            List<BidModifier> newModifiers = Optional.ofNullable(modifiersByType.get(type)).orElse(emptyList());
            Map<BidModifierKey, BidModifier> lockedExistingModifiersOfType = EntryStream.of(lockedExistingModifiers)
                    .filterValues(m -> this.getTypeOf(m).equals(type))
                    .toMap();
            BidModifierTypeSupport typeSupport = typeSupportByTypeMap.get(type);
            if (typeSupport.isMultiValues()) {
                BidModifierMultipleValuesTypeSupport multiValuesTypeSupport =
                        (BidModifierMultipleValuesTypeSupport) typeSupport;
                changedCampaignsAndAdGroups.addAll(multiValuesTypeSupport.addOrReplace(
                        txContext, newModifiers, lockedExistingModifiersOfType, clientId, operatorUid));
            } else {
                BidModifierSingleValueTypeSupport singleValueTypeSupport =
                        (BidModifierSingleValueTypeSupport) typeSupport;
                changedCampaignsAndAdGroups.addAll(singleValueTypeSupport
                        .addOrReplace(txContext, newModifiers, lockedExistingModifiersOfType, operatorUid));
            }
        }
        return changedCampaignsAndAdGroups;
    }

    @SuppressWarnings("unchecked")
    public Map<BidModifierKey, AddedBidModifierInfo> addAll(
            DSLContext txContext, List<BidModifier> modifiers,
            Map<BidModifierKey, BidModifier> lockedExistingModifiers, ClientId clientId, long operatorUid) {
        Map<BidModifierType, List<BidModifier>> modifiersByType =
                modifiers.stream().collect(groupingBy(this::getTypeOf));

        Map<BidModifierKey, AddedBidModifierInfo> resultMap = new HashMap<>();
        for (Map.Entry<BidModifierType, List<BidModifier>> entry : modifiersByType.entrySet()) {
            BidModifierType type = entry.getKey();
            List<BidModifier> modifiersToAdd = entry.getValue();

            Map<BidModifierKey, AddedBidModifierInfo> addedMap;

            BidModifierTypeSupport typeSupport = typeSupportByTypeMap.get(type);
            if (typeSupport.isMultiValues()) {
                BidModifierMultipleValuesTypeSupport multiValuesTypeSupport =
                        (BidModifierMultipleValuesTypeSupport) typeSupport;
                addedMap = multiValuesTypeSupport.add(
                        txContext, modifiersToAdd, lockedExistingModifiers, clientId, operatorUid);
            } else {
                BidModifierSingleValueTypeSupport singleValueTypeSupport =
                        (BidModifierSingleValueTypeSupport) typeSupport;
                addedMap = singleValueTypeSupport.add(txContext, modifiersToAdd, operatorUid);
            }

            resultMap.putAll(addedMap);
        }
        return resultMap;
    }

    public void prepareSystemFields(Collection<BidModifier> bidModifiers) {
        bidModifiers.stream()
                .collect(groupingBy(BidModifier::getType, toList()))
                .forEach((modifierType, typedBidModifiers) -> {
                    //noinspection unchecked
                    typeSupportByTypeMap.get(modifierType).prepareSystemFields(typedBidModifiers);
                });
    }

    public BidModifierKey makeKey(BidModifier modifier, Map<Long, Long> campaignIdsByAdGroupIds) {
        return new BidModifierKey(
                modifier.getCampaignId() == null
                        ? campaignIdsByAdGroupIds.get(modifier.getAdGroupId())
                        : modifier.getCampaignId(),
                modifier.getAdGroupId(),
                getTypeOf(modifier));
    }

    public Multimap<BidModifierType, Record> groupRecordsByTypes(Result<Record> records) {
        return Multimaps.index(records, this::getBidModifierType);
    }

    public BidModifierType getBidModifierType(Record record) {
        return BidModifierType
                .fromSource(record.get(HIERARCHICAL_MULTIPLIERS.TYPE,
                        ru.yandex.direct.dbschema.ppc.enums.HierarchicalMultipliersType.class));
    }

    public <B extends BidModifier, A extends BidModifierAdjustment> BidModifierTypeSupport<B, A> getTypeSupport(
            BidModifierType type) {
        return typeSupportByTypeMap.get(type);
    }
}
