package ru.yandex.direct.core.entity.bidmodifiers.repository.typesupport.multivalue;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import javax.annotation.Nullable;

import com.google.common.collect.Maps;
import com.google.common.collect.Multimap;
import com.google.common.collect.Multimaps;
import one.util.streamex.EntryStream;
import org.jooq.DSLContext;
import org.jooq.Record;
import org.jooq.Result;
import org.jooq.SelectFinalStep;
import org.jooq.SelectWhereStep;
import org.jooq.Table;
import org.jooq.TableField;
import org.jooq.UpdatableRecord;

import ru.yandex.direct.common.log.container.bidmodifiers.LogExpressionMultiplierInfo;
import ru.yandex.direct.common.log.container.bidmodifiers.LogMultiplierInfo;
import ru.yandex.direct.common.log.service.LogBidModifiersService;
import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierExpression;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierExpressionAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierType;
import ru.yandex.direct.core.entity.bidmodifier.model.BidModifierExpressionLiteral;
import ru.yandex.direct.dbschema.ppc.tables.records.ExpressionMultiplierValuesRecord;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.utils.JsonUtils;

import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.core.entity.bidmodifier.BidModifierAdjustment.PERCENT;
import static ru.yandex.direct.core.entity.bidmodifier.BidModifierAdjustmentMultiple.LAST_CHANGE;
import static ru.yandex.direct.core.entity.bidmodifiers.repository.mapper.Common.BASE_MAPPER;
import static ru.yandex.direct.core.entity.bidmodifiers.repository.mapper.Common.EXPRESSION_ADJUSTMENT_FIELDS;
import static ru.yandex.direct.core.entity.bidmodifiers.repository.mapper.Common.EXPRESSION_ADJUSTMENT_MAPPER;
import static ru.yandex.direct.dbschema.ppc.tables.ExpressionMultiplierValues.EXPRESSION_MULTIPLIER_VALUES;
import static ru.yandex.direct.dbschema.ppc.tables.HierarchicalMultipliers.HIERARCHICAL_MULTIPLIERS;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public abstract class AbstractBidModifierExpressionTypeSupport
        <TModifier extends BidModifierExpression, TAdjustment extends BidModifierExpressionAdjustment>
        extends AbstractBidModifierMultipleValuesTypeSupport
        <TModifier, TAdjustment, AbstractBidModifierExpressionTypeSupport.ExpressionAdjustmentKey> {

    protected AbstractBidModifierExpressionTypeSupport(ShardHelper shardHelper,
                                                       LogBidModifiersService logBidModifiersService) {
        super(shardHelper, logBidModifiersService);
    }

    @SuppressWarnings("unchecked")
    @Override
    public TModifier createEmptyBidModifierFromRecord(Record record) {
        TModifier model;
        try {
            model = getBidModifierClass().getDeclaredConstructor().newInstance();
        } catch (Exception ex) {
            throw new RuntimeException(ex);
        }
        return (TModifier) BASE_MAPPER.fromDb(record, model);
    }

    @Override
    public boolean areEqual(TModifier a, TModifier b) {
        return Objects.equals(a.getId(), b.getId()) &&
                Objects.equals(a.getCampaignId(), b.getCampaignId()) &&
                Objects.equals(a.getAdGroupId(), b.getAdGroupId()) &&
                Objects.equals(a.getEnabled(), b.getEnabled()) &&
                // Порядок в conditions внутри adjustments пока не учитываем,
                // т.к. они все логически соединяются по '&&'
                Objects.equals(new HashSet<>(a.getExpressionAdjustments()),
                        new HashSet<>(b.getExpressionAdjustments()));
    }

    @Override
    protected ExpressionAdjustmentKey getKey(TAdjustment bidModifierExpressionAdjustment) {
        return new ExpressionAdjustmentKey(bidModifierExpressionAdjustment.getCondition());
    }

    /**
     * Здесь само выражение является ключом
     * (так как нет возможности идентифицировать корректировки чем-нибудь другим)
     */
    static class ExpressionAdjustmentKey {
        private final Set<Set<BidModifierExpressionLiteral>> condition;

        ExpressionAdjustmentKey(List<List<BidModifierExpressionLiteral>> condition) {
            this.condition = listToSet(condition, HashSet::new);
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            ExpressionAdjustmentKey that = (ExpressionAdjustmentKey) o;
            return Objects.equals(condition, that.condition);
        }

        @Override
        public int hashCode() {
            return Objects.hash(condition);
        }
    }

    @Override
    protected List<Long> getAddedIds(List<TAdjustment> added, List<TAdjustment> inserted) {
        Map<ExpressionAdjustmentKey, Long> idsByKey =
                inserted.stream().collect(toMap(this::getKey, BidModifierExpressionAdjustment::getId));
        return mapList(added, adjustment ->
                idsByKey.get(new ExpressionAdjustmentKey(adjustment.getCondition())));
    }

    @Override
    protected void insertAdjustments(Multimap<Long, TAdjustment> adjustments, DSLContext txContext) {
        InsertHelper<ExpressionMultiplierValuesRecord>
                insertHelper = new InsertHelper<>(txContext, EXPRESSION_MULTIPLIER_VALUES);
        adjustments.forEach((modifierId, adjustment) -> {
            insertHelper.add(EXPRESSION_ADJUSTMENT_MAPPER, adjustment);
            insertHelper.set(EXPRESSION_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID, modifierId);
            insertHelper.newRecord();
        });
        insertHelper.executeIfRecordsAdded();
    }

    @Override
    protected void updateAdjustments(Collection<AppliedChanges<TAdjustment>> changes, DSLContext txContext) {
        JooqUpdateBuilder<ExpressionMultiplierValuesRecord, TAdjustment> updateBuilder =
                new JooqUpdateBuilder<>(EXPRESSION_MULTIPLIER_VALUES.EXPRESSION_MULTIPLIER_VALUE_ID, changes);

        updateBuilder.processProperty(PERCENT, EXPRESSION_MULTIPLIER_VALUES.MULTIPLIER_PCT, Integer::longValue);
        updateBuilder.processProperty(LAST_CHANGE, EXPRESSION_MULTIPLIER_VALUES.LAST_CHANGE, it -> LocalDateTime.now());

        txContext.update(EXPRESSION_MULTIPLIER_VALUES)
                .set(updateBuilder.getValues())
                .where(EXPRESSION_MULTIPLIER_VALUES.EXPRESSION_MULTIPLIER_VALUE_ID.in(
                        listToSet(changes, c -> c.getModel().getId())))
                .execute();
    }

    @Override
    protected void deleteAdjustments(Collection<Long> multiplierIds, DSLContext txContext) {
        txContext.deleteFrom(EXPRESSION_MULTIPLIER_VALUES)
                .where(EXPRESSION_MULTIPLIER_VALUES.EXPRESSION_MULTIPLIER_VALUE_ID.in(multiplierIds))
                .execute();
    }

    @Override
    protected Set<Long> getEmptyHierarchicalMultipliersForUpdate(
            Collection<TModifier> bidModifiers,
            DSLContext dslContext
    ) {
        Table table = EXPRESSION_MULTIPLIER_VALUES;
        TableField<? extends UpdatableRecord, Long> field = EXPRESSION_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID;
        return dslContext
                .select(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID)
                .from(HIERARCHICAL_MULTIPLIERS)
                .leftJoin(table)
                .on(field.eq(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID))
                .where(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID.in(
                        listToSet(bidModifiers, BidModifier::getId)))
                .and(HIERARCHICAL_MULTIPLIERS.TYPE.eq(BidModifierType.toSource(getType())))
                .and(field.isNull())
                .forUpdate()
                .fetchSet(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID);
    }

    @Override
    protected LogMultiplierInfo createLogItem(TModifier modifier,
                                              TAdjustment adjustment,
                                              @Nullable Integer oldPercent) {
        return new LogExpressionMultiplierInfo(adjustment.getId(), modifier.getId(),
                BidModifierType.toSource(getType()),
                oldPercent, adjustment.getPercent(), JsonUtils.toJson(adjustment.getCondition()));
    }

    @SuppressWarnings("unchecked")
    @Override
    public void setAdjustments(TModifier modifier, List<TAdjustment> adjustments) {
        modifier.setExpressionAdjustments((List<BidModifierExpressionAdjustment>) adjustments);
    }

    @SuppressWarnings("unchecked")
    @Override
    public List<TAdjustment> getAdjustments(TModifier modifier) {
        return (List<TAdjustment>) modifier.getExpressionAdjustments();
    }

    @Override
    public void updatePercents(ClientId clientId, long operatorUid,
                               List<AppliedChanges<TAdjustment>> changes,
                               List<TModifier> bidModifiers, DSLContext dslContext) {
        JooqUpdateBuilder<ExpressionMultiplierValuesRecord, TAdjustment> updateBuilder =
                new JooqUpdateBuilder<>(EXPRESSION_MULTIPLIER_VALUES.EXPRESSION_MULTIPLIER_VALUE_ID, changes);

        updateBuilder.processProperty(PERCENT, EXPRESSION_MULTIPLIER_VALUES.MULTIPLIER_PCT, Integer::longValue);

        dslContext.update(EXPRESSION_MULTIPLIER_VALUES)
                .set(updateBuilder.getValues())
                .set(EXPRESSION_MULTIPLIER_VALUES.LAST_CHANGE, LocalDateTime.now())
                .where(EXPRESSION_MULTIPLIER_VALUES.EXPRESSION_MULTIPLIER_VALUE_ID.in(
                        listToSet(changes, it -> it.getModel().getId())
                ))
                .execute();

        // Логируем изменения
        logUpdateChanges(operatorUid, changes, bidModifiers);
    }

    @Override
    public void fillAdjustments(DSLContext dslContext, Collection<TModifier> bidModifiers,
                                boolean updLock) {
        SelectFinalStep<Record> selectStep = dslContext
                .select(EXPRESSION_ADJUSTMENT_FIELDS)
                .from(EXPRESSION_MULTIPLIER_VALUES)
                .where(EXPRESSION_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID.in(
                        bidModifiers.stream().map(BidModifier::getId).collect(toSet())
                ));
        if (updLock) {
            selectStep = ((SelectWhereStep<Record>) selectStep).forUpdate();
        }
        Result<Record> records = selectStep.fetch();
        Multimap<Long, Record> recordsByParentId = Multimaps.index(records,
                record -> record.get(EXPRESSION_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID, Long.class));
        Map<Long, TModifier> bidModifiersById = Maps.uniqueIndex(bidModifiers, BidModifier::getId);
        EntryStream.of(recordsByParentId.asMap())
                .forKeyValue((bidModifierId, recordsList) -> {
                    List<BidModifierExpressionAdjustment> adjustments =
                            mapList(recordsList, this::adjustmentFromRecord);
                    bidModifiersById.get(bidModifierId).withExpressionAdjustments(adjustments);
                });
    }

    @Override
    public Map<Long, List<TAdjustment>> getAdjustmentsByIds(DSLContext dslContext,
                                                            Collection<Long> ids) {
        Result<Record> records = dslContext.select(EXPRESSION_ADJUSTMENT_FIELDS)
                .from(EXPRESSION_MULTIPLIER_VALUES)
                .where(EXPRESSION_MULTIPLIER_VALUES.EXPRESSION_MULTIPLIER_VALUE_ID.in(ids)).fetch();
        Multimap<Long, Record> recordsByParentId = Multimaps.index(records,
                record -> record.get(EXPRESSION_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID, Long.class));
        return EntryStream.of(recordsByParentId.asMap())
                .mapValues(list -> mapList(list, this::adjustmentFromRecord))
                .toMap();
    }


    @Override
    public void prepareSystemFields(List<TModifier> bidModifiers) {
        super.prepareSystemFields(bidModifiers);
        LocalDateTime now = LocalDateTime.now();
        bidModifiers.forEach(bidModifier -> bidModifier.getExpressionAdjustments()
                .forEach(expressionAdjustment -> expressionAdjustment.withLastChange(now)));
    }

    @SuppressWarnings("unchecked")
    private TAdjustment adjustmentFromRecord(Record record) {
        TAdjustment adjustment;
        try {
            adjustment = this.getAdjustmentClass().getDeclaredConstructor().newInstance();
        } catch (Exception ex) {
            throw new RuntimeException(ex);
        }
        return (TAdjustment) EXPRESSION_ADJUSTMENT_MAPPER.fromDb(record, adjustment);
    }
}
