package ru.yandex.direct.core.entity.bidmodifiers.repository.typesupport.multivalue;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Stream;

import javax.annotation.Nullable;

import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Multimap;
import com.google.common.collect.Sets;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.ObjectUtils;
import org.jooq.DSLContext;
import org.jooq.impl.DSL;

import ru.yandex.direct.common.log.container.bidmodifiers.LogBidModifierData;
import ru.yandex.direct.common.log.container.bidmodifiers.LogMultiplierInfo;
import ru.yandex.direct.common.log.service.LogBidModifiersService;
import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierAdjustmentMultiple;
import ru.yandex.direct.core.entity.bidmodifiers.container.AddedBidModifierInfo;
import ru.yandex.direct.core.entity.bidmodifiers.container.BidModifierKey;
import ru.yandex.direct.core.entity.bidmodifiers.repository.BidModifierRepository;
import ru.yandex.direct.core.entity.bidmodifiers.repository.typesupport.BidModifierMultipleValuesTypeSupport;
import ru.yandex.direct.core.entity.container.CampaignIdAndAdGroupIdPair;
import ru.yandex.direct.dbschema.ppc.tables.HierarchicalMultipliers;
import ru.yandex.direct.dbschema.ppc.tables.records.HierarchicalMultipliersRecord;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptyList;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.core.entity.bidmodifiers.repository.mapper.Common.BASE_MAPPER;
import static ru.yandex.direct.core.entity.bidmodifiers.repository.typesupport.multivalue.AbstractBidModifierMultipleValuesTypeSupport.HierarchicalMultiplierAction.INSERT;
import static ru.yandex.direct.core.entity.bidmodifiers.repository.typesupport.multivalue.AbstractBidModifierMultipleValuesTypeSupport.HierarchicalMultiplierAction.NOTHING;
import static ru.yandex.direct.dbschema.ppc.tables.HierarchicalMultipliers.HIERARCHICAL_MULTIPLIERS;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

public abstract class AbstractBidModifierMultipleValuesTypeSupport<
        TModifier extends BidModifier,
        TAdjustment extends BidModifierAdjustmentMultiple,
        TAdjustmentKey>
        implements BidModifierMultipleValuesTypeSupport<TModifier, TAdjustment> {
    private final ShardHelper shardHelper;
    private final LogBidModifiersService logBidModifiersService;

    protected AbstractBidModifierMultipleValuesTypeSupport(ShardHelper shardHelper,
                                                           LogBidModifiersService logBidModifiersService) {
        this.shardHelper = shardHelper;
        this.logBidModifiersService = logBidModifiersService;
    }

    public abstract Class<TModifier> getBidModifierClass();

    /**
     * Реализация должна вернуть ключ, по которому можно однозначно идентифицировать запись в
     * дочерней таблице `*_multiplier_values`. Например, для ретаргетинга это будет просто Long RetCondId,
     * а для гео -- комбинация RegionId + Hidden. Класс ключа должен иметь корректные реализации
     * equals и hashCode.
     */
    protected abstract TAdjustmentKey getKey(TAdjustment adjustment);

    /**
     * Возвращает список id добавленных корректировок в том же порядке, в каком они были перечислены в исходном
     * modifier.
     * Необходимо для того, чтобы вернуть пользователю результаты в правильном порядке.
     */
    protected abstract List<Long> getAddedIds(List<TAdjustment> added, List<TAdjustment> inserted);

    /**
     * Добавляет записи в таблицу `*_multiplier_values` в контексте транзакции.
     */
    protected abstract void insertAdjustments(Multimap<Long, TAdjustment> adjustments, DSLContext txContext);

    /**
     * Обновляет указанные записи в таблице `*_multiplier_values` в контексте транзакции
     * (необходимо, чтобы была поддержка обновления MULTIPLIER_PCT и LAST_CHANGED)
     */
    protected abstract void updateAdjustments(Collection<AppliedChanges<TAdjustment>> changes, DSLContext txContext);

    /**
     * Удаляет указанные записи из таблицы `*_multiplier_values` в контексте транзакции
     */
    protected abstract void deleteAdjustments(Collection<Long> multiplierIds, DSLContext txContext);

    /**
     * Получает идентификаторы пустых наборов корректировок (у которых в базе не осталось ни одной корректировки)
     */
    protected abstract Set<Long> getEmptyHierarchicalMultipliersForUpdate(Collection<TModifier> bidModifiers,
                                                                          DSLContext dslContext);

    /**
     * Вычисляет полный список adjustments для добавления: старые+новые. Точка расширения для гео.
     */
    protected List<TAdjustment> computeAllAdjustments(TModifier modifier, @Nullable TModifier existingModifier,
                                                      ClientId clientId) {
        checkNotNull(modifier);
        List<TAdjustment> allAdjustments = new ArrayList<>(getAdjustments(modifier));
        if (existingModifier != null) {
            allAdjustments.addAll(getAdjustments(existingModifier));
        }
        return allAdjustments;
    }

    /**
     * Вычисляет полный список adjustments для сохранения: учитываются только новые корректировки, имеющиеся
     * игнорируются. Точка расширения для гео.
     */
    protected List<TAdjustment> computeOnlyNewAdjustments(@Nullable TModifier modifier,
                                                          @Nullable TModifier existingModifier, ClientId clientId) {
        return modifier != null ? getAdjustments(modifier) : emptyList();
    }

    @FunctionalInterface
    public interface AdjustmentsMergeFunction<TModifier, TAdjustment> {
        List<TAdjustment> computeAllAdjustments(@Nullable TModifier modifier, @Nullable TModifier existingModifier,
                                                ClientId clientId);
    }

    /**
     * Добавляет указанные корректировки в БД в рамках контекста транзакции txContext.
     * Для тех наборов, которые уже присутствуют в БД, записи будут смёрджены.
     *
     * @param txContext               транзакция
     * @param modifiers               наборы корректировок, которые нужно добавить
     * @param lockedExistingModifiers уже имеющиеся в БД наборы корректировок, прочитанные с UpdLock-блокировкой
     * @param clientId                ID клиента (для обеспечения транслокальности в гео-корректировках)
     * @param operatorUid             uid оператора
     */
    public Map<BidModifierKey, AddedBidModifierInfo> add(DSLContext txContext, List<TModifier> modifiers,
                                                         Map<BidModifierKey, TModifier> lockedExistingModifiers,
                                                         ClientId clientId, long operatorUid) {
        Set<CampaignIdAndAdGroupIdPair> changedCampaignsAndAdGroups = new HashSet<>();
        return save(txContext, modifiers, lockedExistingModifiers, clientId, operatorUid,
                this::computeAllAdjustments, changedCampaignsAndAdGroups);
    }

    /**
     * Сохраняет указанные корректировки в БД в рамках контекста транзакции txContext.
     * Корректировки, которые уже присутствуют в БД, будут удалены, если в новых наборах их не окажется
     * (в отличие от поведения метода {@link #add})
     */
    public Set<CampaignIdAndAdGroupIdPair> addOrReplace(DSLContext txContext, List<TModifier> modifiers,
                                                        Map<BidModifierKey, TModifier> lockedExistingModifiers,
                                                        ClientId clientId, long operatorUid) {
        Set<CampaignIdAndAdGroupIdPair> changedCampaignsAndAdGroups = new HashSet<>();
        save(txContext, modifiers, lockedExistingModifiers, clientId, operatorUid,
                this::computeOnlyNewAdjustments, changedCampaignsAndAdGroups);
        return changedCampaignsAndAdGroups;
    }

    public Map<BidModifierKey, AddedBidModifierInfo> save(DSLContext txContext, List<TModifier> modifiers,
                                                          Map<BidModifierKey, TModifier> lockedExistingModifiers,
                                                          ClientId clientId, long operatorUid,
                                                          AdjustmentsMergeFunction<TModifier, TAdjustment> mergeFunction,
                                                          Set<CampaignIdAndAdGroupIdPair> changedCampaignsAndAdGroups) {
        Map<BidModifierKey, TModifier> modifierByKeys = StreamEx.of(modifiers)
                .distinct(BidModifierKey::new)
                .collect(toMap(BidModifierKey::new, identity()));

        Map<BidModifierKey, DiffInfo<TAdjustment>> diffMap = new HashMap<>();

        Set<BidModifierKey> allKeys = Stream.concat(
                modifierByKeys.keySet().stream(),
                lockedExistingModifiers.keySet().stream().filter(key -> key.getType().equals(getType()))
        ).collect(toSet());

        // Чтобы у всех обновлённых last_change было одинаковое время
        LocalDateTime now = LocalDateTime.now();
        for (BidModifierKey key : allKeys) {
            @Nullable TModifier modifier = modifierByKeys.get(key);

            @Nullable TModifier existingModifier = lockedExistingModifiers.get(key);

            // Вычисляем изменения, которые нужно применить к БД для добавления указанных корректировок
            List<TAdjustment> existingAdjustments;
            if (existingModifier != null) {
                existingAdjustments = getAdjustments(existingModifier);
            } else {
                existingAdjustments = emptyList();
            }
            List<TAdjustment> allAdjustments =
                    mergeFunction.computeAllAdjustments(modifier, existingModifier, clientId);
            DiffInfo<TAdjustment> diff = computeDiff(allAdjustments, existingAdjustments, existingModifier,
                    getActualEnabled(modifier), now);
            diffMap.put(key, diff);

            // Если действительно есть что изменять -- добавляем этот объект во множество изменённых
            if (!diff.isEmpty()) {
                changedCampaignsAndAdGroups.add(new CampaignIdAndAdGroupIdPair()
                        .withCampaignId(key.getCampaignId())
                        .withAdGroupId(key.getAdGroupId()));
            }
        }

        // Определяем, сколько необходимо создать новых записей в таблицах
        Integer totalCount = diffMap.values().stream()
                .map(it -> it.toInsert.size() + (it.hierarchicalMultiplierAction == INSERT ? 1 : 0))
                .reduce(Integer::sum).orElse(0);

        // Генерируем айдишники для новых записей
        List<Long> generatedIds = BidModifierRepository.generateIds(shardHelper, totalCount);

        // Подготавливаем запросы
        List<TModifier> hierarchicalMultipliersToInsert = new ArrayList<>();
        List<AppliedChanges<TModifier>> hierarchicalMultipliersChangesToUpdate = new ArrayList<>();
        List<TModifier> hierarchicalMultipliersToDelete = new ArrayList<>();
        Multimap<Long, TAdjustment> toInsert = ArrayListMultimap.create();
        Multimap<Long, AppliedChanges<TAdjustment>> toUpdate = ArrayListMultimap.create();
        Multimap<Long, TAdjustment> toDelete = ArrayListMultimap.create();

        Map<BidModifierKey, AddedBidModifierInfo> resultMap = new HashMap<>();

        Iterator<Long> idsIterator = generatedIds.iterator();
        for (Map.Entry<BidModifierKey, DiffInfo<TAdjustment>> diffEntry : diffMap.entrySet()) {
            BidModifierKey key = diffEntry.getKey();
            DiffInfo<TAdjustment> diff = diffEntry.getValue();

            @Nullable TModifier modifier = modifierByKeys.get(key);
            @Nullable TModifier existingModifier = lockedExistingModifiers.get(key);

            long hierarchicalMultiplierId;
            switch (diff.hierarchicalMultiplierAction) {
                case INSERT: {
                    checkNotNull(modifier);
                    hierarchicalMultiplierId = idsIterator.next();
                    modifier.withId(hierarchicalMultiplierId);
                    hierarchicalMultipliersToInsert.add(modifier);
                    break;
                }
                case UPDATE: {
                    checkNotNull(modifier);
                    checkState(existingModifier != null);
                    //
                    hierarchicalMultiplierId = existingModifier.getId();
                    ModelChanges<TModifier> modelChanges =
                            new ModelChanges<>(hierarchicalMultiplierId, getBidModifierClass());
                    modelChanges.process(modifier.getEnabled(), BidModifier.ENABLED);
                    modelChanges.process(now, BidModifier.LAST_CHANGE);
                    hierarchicalMultipliersChangesToUpdate.add(modelChanges.applyTo(existingModifier));
                    //
                    break;
                }
                case DELETE: {
                    checkState(existingModifier != null);
                    //
                    hierarchicalMultiplierId = existingModifier.getId();
                    hierarchicalMultipliersToDelete.add(existingModifier);
                    break;
                }
                case NOTHING: {
                    // Do nothing
                    checkState(existingModifier != null);
                    hierarchicalMultiplierId = existingModifier.getId();
                    break;
                }
                default: {
                    throw new IllegalStateException("Unexpected action");
                }
            }

            // Вставляем сгенерированные идентификаторы в создаваемые записи
            for (TAdjustment multiplier : diff.toInsert) {
                multiplier.withId(idsIterator.next());
            }
            toInsert.putAll(hierarchicalMultiplierId, diff.toInsert);
            toUpdate.putAll(hierarchicalMultiplierId, diff.toUpdate);
            toDelete.putAll(hierarchicalMultiplierId, diff.toDelete);

            AddedBidModifierInfo added;
            if (modifier != null) {
                added = AddedBidModifierInfo.added(getAddedIds(getAdjustments(modifier), diff.toInsert));
            } else {
                added = AddedBidModifierInfo.notAdded();
            }
            resultMap.put(key, added);
        }

        // Применяем все изменения массово
        applyChanges(hierarchicalMultipliersToDelete, hierarchicalMultipliersToInsert,
                hierarchicalMultipliersChangesToUpdate, toDelete, toInsert, toUpdate, txContext);

        // Логируем сделанные изменения
        logChanges(modifierByKeys.keySet(), hierarchicalMultipliersToDelete, hierarchicalMultipliersToInsert,
                hierarchicalMultipliersChangesToUpdate.stream().map(AppliedChanges::getModel).collect(toList()),
                toDelete, toInsert, toUpdate, operatorUid);

        return resultMap;
    }

    private boolean getActualEnabled(@Nullable TModifier modifier) {
        if (modifier == null) {
            return false;
        }
        return ObjectUtils.firstNonNull(modifier.getEnabled(), true);
    }

    private void applyChanges(List<TModifier> hierarchicalMultipliersToDelete,
                              List<TModifier> hierarchicalMultipliersToInsert,
                              List<AppliedChanges<TModifier>> hierarchicalMultipliersChangesToUpdate,
                              Multimap<Long, TAdjustment> toDelete,
                              Multimap<Long, TAdjustment> toInsert,
                              Multimap<Long, AppliedChanges<TAdjustment>> toUpdate,
                              DSLContext txContext) {
        // Сначала делаем удаление -- если этого не сделать в первую очередь, они могут помешать последующему добавлению
        if (!toDelete.isEmpty()) {
            deleteAdjustments(toDelete.values().stream().map(TAdjustment::getId).collect(toSet()), txContext);
        }
        if (!hierarchicalMultipliersToDelete.isEmpty()) {
            deleteHierarchicalMultipliers(hierarchicalMultipliersToDelete.stream()
                    .map(BidModifier::getId).collect(toSet()), txContext);
        }
        // Теперь добавляем и апдейтим записи, начиная с родительских таблиц
        if (!hierarchicalMultipliersToInsert.isEmpty()) {
            insertBidModifiers(hierarchicalMultipliersToInsert, txContext);
        }
        if (!hierarchicalMultipliersChangesToUpdate.isEmpty()) {
            updateBidModifiers(hierarchicalMultipliersChangesToUpdate, txContext);
        }
        if (!toInsert.isEmpty()) {
            insertAdjustments(toInsert, txContext);
        }
        if (!toUpdate.isEmpty()) {
            updateAdjustments(toUpdate.values(), txContext);
        }
    }

    private void insertBidModifiers(Collection<TModifier> bidModifiers, DSLContext txContext) {
        new InsertHelper<>(txContext, HIERARCHICAL_MULTIPLIERS).addAll(BASE_MAPPER, bidModifiers).execute();
    }

    private void updateBidModifiers(List<AppliedChanges<TModifier>> changes, DSLContext txContext) {
        JooqUpdateBuilder<HierarchicalMultipliersRecord, TModifier> updateBuilder =
                new JooqUpdateBuilder<>(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID, changes);

        updateBuilder.processProperty(TModifier.ENABLED, HIERARCHICAL_MULTIPLIERS.IS_ENABLED,
                RepositoryUtils::booleanToLong);
        updateBuilder.processProperty(TModifier.LAST_CHANGE, HIERARCHICAL_MULTIPLIERS.LAST_CHANGE);

        txContext.update(HIERARCHICAL_MULTIPLIERS)
                .set(updateBuilder.getValues())
                .where(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID.in(
                        listToSet(changes, it -> it.getModel().getId())
                ))
                .execute();
    }

    private void logChanges(Set<BidModifierKey> allKeys,
                            List<TModifier> modifiersToDelete,
                            List<TModifier> modifiersToInsert,
                            List<TModifier> modifiersToUpdate,
                            Multimap<Long, TAdjustment> toDelete,
                            Multimap<Long, TAdjustment> toInsert,
                            Multimap<Long, AppliedChanges<TAdjustment>> toUpdate,
                            long operatorUid) {
        Map<BidModifierKey, TModifier> modifiersByKey =
                Stream.of(modifiersToDelete, modifiersToInsert, modifiersToUpdate)
                        .flatMap(Collection::stream).collect(toMap(BidModifierKey::new, identity()));
        Map<Long, TModifier> modifiersById = Stream.of(modifiersToDelete, modifiersToInsert, modifiersToUpdate)
                .flatMap(Collection::stream).collect(toMap(BidModifier::getId, identity()));
        Map<BidModifierKey, Collection<TAdjustment>> toDeleteByKey =
                EntryStream.of(toDelete.asMap()).mapKeys(modifiersById::get).mapKeys(BidModifierKey::new).toMap();
        Map<BidModifierKey, Collection<TAdjustment>> toInsertByKey =
                EntryStream.of(toInsert.asMap()).mapKeys(modifiersById::get).mapKeys(BidModifierKey::new).toMap();
        Map<BidModifierKey, Collection<AppliedChanges<TAdjustment>>> toUpdateByKey =
                EntryStream.of(toUpdate.asMap()).mapKeys(modifiersById::get).mapKeys(BidModifierKey::new).toMap();

        Set<BidModifierKey> modifierKeysToDelete = modifiersToDelete.stream().map(BidModifierKey::new).collect(toSet());
        List<LogBidModifierData> logDataRecords = allKeys.stream().map(key -> {
            LogBidModifierData logData = new LogBidModifierData(key.getCampaignId(), key.getAdGroupId());
            TModifier modifier = modifiersByKey.get(key);
            if (modifierKeysToDelete.contains(key)) {
                logData.withDeletedSet(modifier.getId());
            }
            if (toInsertByKey.containsKey(key)) {
                logData.withInserted(toInsertByKey.get(key).stream().map(
                        adjustment -> createSimpleLogItem(modifier, adjustment)).collect(toList()));
            }
            if (toDeleteByKey.containsKey(key)) {
                logData.withDeleted(toDeleteByKey.get(key).stream().map(
                        adjustment -> createSimpleLogItem(modifier, adjustment)).collect(toList()));
            }
            if (toUpdateByKey.containsKey(key)) {
                logData.withUpdated(toUpdateByKey.get(key).stream()
                        .map(it -> createLogItem(modifier, it.getModel(), it.getOldValue(TAdjustment.PERCENT)))
                        .collect(toList()));
            }
            return logData;
        }).collect(toList());
        logBidModifiersService.logBidModifiers(logDataRecords, operatorUid);
    }

    /**
     * Конвертирует multiplier в запись LogMultiplierInfo, указывая дополнительные поля для наследников.
     */
    protected abstract LogMultiplierInfo createLogItem(TModifier modifier, TAdjustment multiplier,
                                                       @Nullable Integer oldPercent);

    private LogMultiplierInfo createSimpleLogItem(TModifier modifier, TAdjustment multiplier) {
        return createLogItem(modifier, multiplier, null);
    }

    enum HierarchicalMultiplierAction {
        INSERT,
        UPDATE,
        DELETE,
        NOTHING
    }

    /**
     * Что нужно сделать с набором корректировок.
     */
    protected static class DiffInfo<TAdjustment extends BidModifierAdjustmentMultiple> {
        final List<TAdjustment> toDelete;
        final List<TAdjustment> toInsert;
        final List<AppliedChanges<TAdjustment>> toUpdate;
        final HierarchicalMultiplierAction hierarchicalMultiplierAction;

        DiffInfo(List<TAdjustment> toDelete, List<TAdjustment> toInsert,
                 List<AppliedChanges<TAdjustment>> toUpdate,
                 HierarchicalMultiplierAction hierarchicalMultiplierAction) {
            this.toDelete = toDelete;
            this.toInsert = toInsert;
            this.toUpdate = toUpdate;
            this.hierarchicalMultiplierAction = hierarchicalMultiplierAction;
        }

        boolean isEmpty() {
            return toDelete.isEmpty() && toInsert.isEmpty() && toUpdate.isEmpty()
                    && hierarchicalMultiplierAction == NOTHING;
        }
    }

    /**
     * Вычисляет набор изменений, который нужно применить к БД для того, чтобы от текущего состояния,
     * представленного набором existingValues, перейти к целевому состоянию allValues (таким образом, если
     * передать пустой список в allValues, то все имеющиеся корректировки будут помечены для удаления).
     *
     * @param allValues           Целевое состояние
     * @param existingValues      Уже имеющиеся values (или emptyList, если записей нет)
     * @param existingBidModifier Уже имеющийся в базе hierarchicalMultiplier либо null, если его нет
     * @param newEnabled          Должен ли быть включён набор корректировок или нет
     */
    DiffInfo<TAdjustment> computeDiff(List<TAdjustment> allValues, List<TAdjustment> existingValues,
                                      @Nullable BidModifier existingBidModifier, boolean newEnabled,
                                      LocalDateTime now) {
        List<TAdjustmentKey> allValuesKeys = allValues.stream().map(this::getKey).collect(toList());

        // Определяем, что делать с каждым объектом: добавить, удалить или изменить
        Map<TAdjustmentKey, TAdjustment> existingValuesMap =
                existingValues.stream().collect(toMap(this::getKey, identity()));

        List<TAdjustment> toInsert = new ArrayList<>();
        List<AppliedChanges<TAdjustment>> toUpdate = new ArrayList<>();
        for (TAdjustment value : allValues) {
            TAdjustmentKey key = getKey(value);
            if (existingValuesMap.containsKey(key)) {
                TAdjustment oldValue = existingValuesMap.get(key);
                if (!oldValue.getPercent().equals(value.getPercent())) {
                    ModelChanges<TAdjustment> modelChanges = new ModelChanges<>(oldValue.getId(), getAdjustmentClass());
                    modelChanges.process(value.getPercent(), BidModifierAdjustmentMultiple.PERCENT);
                    modelChanges.process(now, BidModifierAdjustmentMultiple.LAST_CHANGE);
                    toUpdate.add(modelChanges.applyTo(oldValue));
                }
            } else {
                toInsert.add(value);
            }
        }

        List<TAdjustment> toDelete = Sets.difference(existingValuesMap.keySet(), new HashSet<>(allValuesKeys))
                .stream().map(existingValuesMap::get).collect(toList());

        // Определяем, что делать с hierarchical_multiplier
        HierarchicalMultiplierAction hierarchicalMultiplierAction = HierarchicalMultiplierAction.NOTHING;
        if (existingBidModifier != null) {
            if (!allValues.isEmpty()) {
                // Если в результате должны быть установлены какие-то значения, обновляем is_enabled и last_change
                boolean isEnabledHasChanged = existingBidModifier.getEnabled() != newEnabled;
                boolean needUpdateLastChange = !toInsert.isEmpty() || !toDelete.isEmpty() || !toUpdate.isEmpty();
                if (isEnabledHasChanged || needUpdateLastChange) {
                    hierarchicalMultiplierAction = HierarchicalMultiplierAction.UPDATE;
                }
            } else {
                // А если теперь корректировок не осталось, то можно удалить и запись в hierarchical_multipliers
                hierarchicalMultiplierAction = HierarchicalMultiplierAction.DELETE;
            }
        } else {
            if (!allValues.isEmpty()) {
                hierarchicalMultiplierAction = HierarchicalMultiplierAction.INSERT;
            }
        }

        return new DiffInfo<>(toDelete, toInsert, toUpdate, hierarchicalMultiplierAction);
    }

    @Override
    public void delete(DSLContext txContext, ClientId clientId, long operatorUid, Collection<TModifier> bidModifiers) {
        Set<Long> idsToDelete =
                bidModifiers.stream().map(this::getAdjustments)
                        .flatMap(Collection::stream)
                        .map(TAdjustment::getId)
                        .collect(toSet());

        deleteAndLogChanges(bidModifiers, idsToDelete, txContext, operatorUid);
    }

    protected void deleteAndLogChanges(Collection<TModifier> bidModifiers, Set<Long> idsToDelete,
                                       DSLContext txContext, long operatorUid) {
        deleteAdjustments(idsToDelete, txContext);

        // Получаем записи таблицы HIERARCHICAL_MULTIPLIERS без подчиненных записей и затем их удаляем
        // Делаем в транзакции, чтобы быть уверенным, что после получения не добавятся другие записи
        Set<Long> deletedParentIds = txContext.transactionResult(configuration -> {
            DSLContext txContextForUpdate = DSL.using(configuration);
            Set<Long> emptyIds = getEmptyHierarchicalMultipliersForUpdate(bidModifiers, txContextForUpdate);
            deleteHierarchicalMultipliers(emptyIds, txContextForUpdate);
            return emptyIds;
        });

        // Записываем сделанные изменения в логи
        List<LogBidModifierData> logItems = bidModifiers.stream()
                .map(modifier -> convertModifierToLogItem(modifier, deletedParentIds))
                .collect(toList());

        logBidModifiersService.logBidModifiers(logItems, operatorUid);
    }

    private LogBidModifierData convertModifierToLogItem(TModifier modifier, Set<Long> deletedParentIds) {
        LogBidModifierData logItem =
                new LogBidModifierData(modifier.getCampaignId(), modifier.getAdGroupId())
                        .withDeleted(
                                getAdjustments(modifier).stream().map(
                                        adjustment -> createSimpleLogItem(modifier, adjustment)
                                ).collect(toList())
                        );
        if (deletedParentIds.contains(modifier.getId())) {
            logItem.withDeletedSet(modifier.getId());
        }
        return logItem;
    }

    private static void deleteHierarchicalMultipliers(Collection<Long> ids, DSLContext txContext) {
        txContext.deleteFrom(HierarchicalMultipliers.HIERARCHICAL_MULTIPLIERS)
                .where(HierarchicalMultipliers.HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID.in(ids))
                .execute();
    }

    /**
     * Логирует изменения корректировок changes, выполненные в рамках наборов корректировок bidModifiers.
     */
    protected void logUpdateChanges(long operatorUid, List<AppliedChanges<TAdjustment>> changes,
                                    List<TModifier> bidModifiers) {
        Map<Long, Integer> oldPercentById = changes.stream()
                .collect(toMap(ch -> ch.getModel().getId(), ch -> ch.getOldValue(BidModifierAdjustment.PERCENT)));

        List<LogBidModifierData> logDataItems = bidModifiers.stream().map(modifier -> {
            LogBidModifierData logData = new LogBidModifierData(modifier.getCampaignId(), modifier.getAdGroupId());

            List<LogMultiplierInfo> updated = getAdjustments(modifier).stream()
                    .map(it -> {
                        Long id = it.getId();
                        Integer oldPercent = oldPercentById.get(id);
                        return createLogItem(modifier, it, oldPercent);
                    }).collect(toList());

            logData.withUpdated(updated);

            return logData;
        }).collect(toList());

        logBidModifiersService.logBidModifiers(logDataItems, operatorUid);
    }

    @Override
    public final boolean isMultiValues() {
        return true;
    }

    @Override
    public void prepareSystemFields(List<TModifier> bidModifiers) {
        LocalDateTime now = LocalDateTime.now();
        bidModifiers.forEach(bidModifier -> {
            bidModifier
                    .withEnabled(nvl(bidModifier.getEnabled(), true))
                    .withLastChange(now);
        });
    }
}
