package ru.yandex.direct.core.entity.bidmodifiers.repository.typesupport.multivalue;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import javax.annotation.Nullable;

import com.google.common.collect.Maps;
import com.google.common.collect.Multimap;
import com.google.common.collect.Multimaps;
import one.util.streamex.EntryStream;
import org.jooq.DSLContext;
import org.jooq.Record;
import org.jooq.Result;
import org.jooq.SelectFinalStep;
import org.jooq.SelectForUpdateStep;
import org.jooq.Table;
import org.jooq.TableField;
import org.jooq.UpdatableRecord;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.common.log.container.bidmodifiers.LogMultiplierInfo;
import ru.yandex.direct.common.log.container.bidmodifiers.LogRetargetingMultiplierInfo;
import ru.yandex.direct.common.log.service.LogBidModifiersService;
import ru.yandex.direct.core.entity.bidmodifier.AbstractBidModifierRetargeting;
import ru.yandex.direct.core.entity.bidmodifier.AbstractBidModifierRetargetingAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierRetargetingAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierType;
import ru.yandex.direct.dbschema.ppc.tables.RetargetingMultiplierValues;
import ru.yandex.direct.dbschema.ppc.tables.records.RetargetingMultiplierValuesRecord;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder;
import ru.yandex.direct.model.AppliedChanges;

import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.core.entity.bidmodifier.BidModifierAdjustment.PERCENT;
import static ru.yandex.direct.core.entity.bidmodifier.BidModifierRetargetingAdjustment.LAST_CHANGE;
import static ru.yandex.direct.core.entity.bidmodifiers.repository.mapper.Common.RETARGETING_ADJUSTMENT_FIELDS;
import static ru.yandex.direct.core.entity.bidmodifiers.repository.mapper.Common.RETARGETING_IS_ACCESSIBLE_FIELD_MAPPER;
import static ru.yandex.direct.dbschema.ppc.Tables.RETARGETING_MULTIPLIER_VALUES;
import static ru.yandex.direct.dbschema.ppc.tables.HierarchicalMultipliers.HIERARCHICAL_MULTIPLIERS;
import static ru.yandex.direct.dbschema.ppc.tables.RetargetingGoals.RETARGETING_GOALS;

public abstract class AbstractBidModifierRetargetingTypeSupport<
        TModifier extends AbstractBidModifierRetargeting,
        TAdjustment extends AbstractBidModifierRetargetingAdjustment>
        extends AbstractBidModifierMultipleValuesTypeSupport<TModifier, TAdjustment, Long> {

    @Autowired
    protected AbstractBidModifierRetargetingTypeSupport(ShardHelper shardHelper,
                                                        LogBidModifiersService logBidModifiersService) {
        super(shardHelper, logBidModifiersService);
    }

    @Override
    public boolean areEqual(TModifier a, TModifier b) {
        return Objects.equals(a.getId(), b.getId()) &&
                Objects.equals(a.getCampaignId(), b.getCampaignId()) &&
                Objects.equals(a.getAdGroupId(), b.getAdGroupId()) &&
                Objects.equals(a.getEnabled(), b.getEnabled()) &&
                Objects.equals(new HashSet<>(a.getRetargetingAdjustments()),
                        new HashSet<>(b.getRetargetingAdjustments()));
    }

    @Override
    public List<TModifier> createEmptyBidModifiersFromRecords(Collection<Record> records) {
        return records.stream().map(this::createEmptyBidModifierFromRecord).collect(toList());
    }

    @SuppressWarnings("unchecked")
    @Override
    public void setAdjustments(TModifier modifier,
                               List<TAdjustment> retargetingAdjustments) {
        modifier.setRetargetingAdjustments((List<AbstractBidModifierRetargetingAdjustment>) retargetingAdjustments);
    }

    @Override
    public Map<Long, List<TAdjustment>> getAdjustmentsByIds(DSLContext dslContext,
                                                            Collection<Long> ids) {
        Result<Record> records = dslContext.select(RETARGETING_ADJUSTMENT_FIELDS)
                .select(
                        DSL.min(
                                DSL.ifnull(RETARGETING_GOALS.IS_ACCESSIBLE, 0L)
                        ).as(RETARGETING_GOALS.IS_ACCESSIBLE)
                )
                .from(RETARGETING_MULTIPLIER_VALUES)
                .leftJoin(RETARGETING_GOALS)
                .on(RETARGETING_GOALS.RET_COND_ID
                        .eq(RetargetingMultiplierValues.RETARGETING_MULTIPLIER_VALUES.RET_COND_ID))
                .where(RETARGETING_MULTIPLIER_VALUES.RETARGETING_MULTIPLIER_VALUE_ID.in(ids))
                .groupBy(RETARGETING_ADJUSTMENT_FIELDS)
                .fetch();
        Multimap<Long, Record> recordsByParentId = Multimaps.index(records,
                record -> record.get(RETARGETING_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID, Long.class));
        return EntryStream.of(recordsByParentId.asMap())
                .mapValues(list -> list.stream().map(record -> {
                    TAdjustment adjustment = getAdjustmentMapper().fromDb(record);
                    RETARGETING_IS_ACCESSIBLE_FIELD_MAPPER.fromDb(adjustment, record);
                    return adjustment;
                }).collect(toList()))
                .toMap();
    }

    @Override
    public void fillAdjustments(DSLContext dslContext, Collection<TModifier> bidModifiers,
                                boolean updLock) {
        SelectFinalStep<Record> selectStep = dslContext
                .select(RETARGETING_ADJUSTMENT_FIELDS)
                .select(
                        DSL.min(
                                DSL.ifnull(RETARGETING_GOALS.IS_ACCESSIBLE, 0L)
                        ).as(RETARGETING_GOALS.IS_ACCESSIBLE)
                )
                .from(RETARGETING_MULTIPLIER_VALUES)
                .leftJoin(RETARGETING_GOALS)
                .on(RETARGETING_GOALS.RET_COND_ID
                        .eq(RetargetingMultiplierValues.RETARGETING_MULTIPLIER_VALUES.RET_COND_ID))
                .where(RETARGETING_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID.in(
                        bidModifiers.stream().map(BidModifier::getId).collect(toSet())
                ))
                .groupBy(RETARGETING_ADJUSTMENT_FIELDS);
        if (updLock) {
            selectStep = ((SelectForUpdateStep<Record>) selectStep).forUpdate();
        }
        Result<Record> records = selectStep.fetch();
        Multimap<Long, Record> recordsByParentId = Multimaps.index(records,
                record -> record.get(RETARGETING_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID, Long.class));
        Map<Long, TModifier> bidModifiersById = Maps.uniqueIndex(bidModifiers, BidModifier::getId);
        EntryStream.of(recordsByParentId.asMap())
                .forKeyValue((bidModifierId, recordsList) -> {
                    List<AbstractBidModifierRetargetingAdjustment> adjustments =
                            recordsList.stream().map(record -> {
                                TAdjustment adjustment =
                                        getAdjustmentMapper().fromDb(record);
                                RETARGETING_IS_ACCESSIBLE_FIELD_MAPPER.fromDb(adjustment, record);
                                return adjustment;
                            }).collect(toList());
                    bidModifiersById.get(bidModifierId).withRetargetingAdjustments(adjustments);
                });
    }

    abstract protected JooqMapperWithSupplier<TAdjustment> getAdjustmentMapper();


    @SuppressWarnings("unchecked")
    @Override
    public List<TAdjustment> getAdjustments(TModifier modifier) {
        return (List<TAdjustment>) modifier.getRetargetingAdjustments();
    }

    @Override
    protected List<Long> getAddedIds(List<TAdjustment> added,
                                     List<TAdjustment> inserted) {
        Map<Long, Long> idsByRetCondId = inserted.stream()
                .collect(toMap(AbstractBidModifierRetargetingAdjustment::getRetargetingConditionId,
                        AbstractBidModifierRetargetingAdjustment::getId));
        return added.stream()
                .map(adjustment -> idsByRetCondId.get(adjustment.getRetargetingConditionId()))
                .collect(toList());
    }

    @Override
    protected void deleteAdjustments(Collection<Long> multiplierIds, DSLContext txContext) {
        txContext.deleteFrom(RetargetingMultiplierValues.RETARGETING_MULTIPLIER_VALUES)
                .where(RetargetingMultiplierValues.RETARGETING_MULTIPLIER_VALUES.RETARGETING_MULTIPLIER_VALUE_ID
                        .in(multiplierIds))
                .execute();
    }

    @Override
    protected Long getKey(TAdjustment adjustment) {
        return adjustment.getRetargetingConditionId();
    }

    @Override
    protected void insertAdjustments(Multimap<Long, TAdjustment> adjustments,
                                     DSLContext txContext) {
        InsertHelper<RetargetingMultiplierValuesRecord>
                insertHelper = new InsertHelper<>(txContext, RETARGETING_MULTIPLIER_VALUES);
        adjustments.forEach((modifierId, adjustment) -> {
            insertHelper.add(getAdjustmentMapper(), adjustment);
            insertHelper.set(RETARGETING_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID, modifierId);
            insertHelper.newRecord();
        });
        insertHelper.executeIfRecordsAdded();
    }

    @Override
    protected void updateAdjustments(Collection<AppliedChanges<TAdjustment>> changes,
                                     DSLContext txContext) {
        JooqUpdateBuilder<RetargetingMultiplierValuesRecord, TAdjustment> updateBuilder =
                new JooqUpdateBuilder<>(RETARGETING_MULTIPLIER_VALUES.RETARGETING_MULTIPLIER_VALUE_ID, changes);

        updateBuilder.processProperty(PERCENT, RETARGETING_MULTIPLIER_VALUES.MULTIPLIER_PCT, Integer::longValue);
        updateBuilder
                .processProperty(LAST_CHANGE, RETARGETING_MULTIPLIER_VALUES.LAST_CHANGE, it -> LocalDateTime.now());

        txContext.update(RETARGETING_MULTIPLIER_VALUES)
                .set(updateBuilder.getValues())
                .where(RETARGETING_MULTIPLIER_VALUES.RETARGETING_MULTIPLIER_VALUE_ID.in(
                        changes.stream().map(it -> it.getModel().getId()).collect(toSet())))
                .execute();
    }

    @Override
    protected LogMultiplierInfo createLogItem(TModifier modifier,
                                              TAdjustment adjustment, @Nullable Integer oldPercent) {
        return new LogRetargetingMultiplierInfo(adjustment.getId(), modifier.getId(),
                BidModifierType.toSource(modifier.getType()),
                oldPercent, adjustment.getPercent(), adjustment.getRetargetingConditionId());
    }

    @Override
    protected Set<Long> getEmptyHierarchicalMultipliersForUpdate(Collection<TModifier> bidModifiers,
                                                                 DSLContext dslContext) {
        Table table = RETARGETING_MULTIPLIER_VALUES;
        TableField<? extends UpdatableRecord, Long> field = RETARGETING_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID;
        return dslContext
                .select(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID)
                .from(HIERARCHICAL_MULTIPLIERS)
                .leftJoin(table)
                .on(field.eq(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID))
                .where(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID.in(
                        bidModifiers.stream().map(BidModifier::getId).collect(toSet())))
                .and(HIERARCHICAL_MULTIPLIERS.TYPE.eq(BidModifierType.toSource(getType())))
                .and(field.isNull())
                .forUpdate()
                .fetchSet(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID);
    }

    @Override
    public void updatePercents(ClientId clientId, long operatorUid,
                               List<AppliedChanges<TAdjustment>> changes,
                               List<TModifier> bidModifiers, DSLContext dslContext) {
        JooqUpdateBuilder<RetargetingMultiplierValuesRecord, TAdjustment> updateBuilder =
                new JooqUpdateBuilder<>(RETARGETING_MULTIPLIER_VALUES.RETARGETING_MULTIPLIER_VALUE_ID, changes);

        updateBuilder.processProperty(BidModifierRetargetingAdjustment.PERCENT,
                RETARGETING_MULTIPLIER_VALUES.MULTIPLIER_PCT, Integer::longValue);

        dslContext.update(RETARGETING_MULTIPLIER_VALUES)
                .set(updateBuilder.getValues())
                .set(RETARGETING_MULTIPLIER_VALUES.LAST_CHANGE, LocalDateTime.now())
                .where(RETARGETING_MULTIPLIER_VALUES.RETARGETING_MULTIPLIER_VALUE_ID.in(
                        changes.stream()
                                .map(it -> it.getModel().getId())
                                .collect(toSet())
                ))
                .execute();

        // Логируем изменения
        logUpdateChanges(operatorUid, changes, bidModifiers);
    }

    @Override
    public void prepareSystemFields(List<TModifier> bidModifiers) {
        super.prepareSystemFields(bidModifiers);
        LocalDateTime now = LocalDateTime.now();
        bidModifiers.forEach(bidModifier -> bidModifier.getRetargetingAdjustments()
                .forEach(retargetingAdjustment -> retargetingAdjustment.withLastChange(now)));
    }
}
