package ru.yandex.direct.core.entity.bidmodifiers.repository.typesupport.multivalue;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Maps;
import com.google.common.collect.Multimap;
import com.google.common.collect.Multimaps;
import one.util.streamex.EntryStream;
import org.jooq.DSLContext;
import org.jooq.Record;
import org.jooq.Result;
import org.jooq.SelectFinalStep;
import org.jooq.SelectForUpdateStep;
import org.jooq.TableField;
import org.jooq.UpdatableRecord;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.common.log.container.bidmodifiers.LogABSegmentMultiplierInfo;
import ru.yandex.direct.common.log.container.bidmodifiers.LogMultiplierInfo;
import ru.yandex.direct.common.log.service.LogBidModifiersService;
import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierABSegment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierABSegmentAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierType;
import ru.yandex.direct.core.entity.bidmodifiers.container.AddedBidModifierInfo;
import ru.yandex.direct.core.entity.bidmodifiers.container.BidModifierKey;
import ru.yandex.direct.core.entity.container.CampaignIdAndAdGroupIdPair;
import ru.yandex.direct.core.entity.retargeting.model.ConditionType;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.GoalType;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition;
import ru.yandex.direct.core.entity.retargeting.model.Rule;
import ru.yandex.direct.core.entity.retargeting.model.RuleType;
import ru.yandex.direct.core.entity.retargeting.service.RetargetingConditionService;
import ru.yandex.direct.dbschema.ppc.enums.HierarchicalMultipliersType;
import ru.yandex.direct.dbschema.ppc.enums.RetargetingConditionsRetargetingConditionsType;
import ru.yandex.direct.dbschema.ppc.tables.records.AbSegmentMultiplierValuesRecord;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.result.MassResult;

import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.core.entity.bidmodifier.BidModifierAdjustment.PERCENT;
import static ru.yandex.direct.core.entity.bidmodifier.BidModifierAdjustmentMultiple.LAST_CHANGE;
import static ru.yandex.direct.core.entity.bidmodifiers.repository.mapper.Common.AB_SEGMENT_ADJUSTMENT_FIELDS;
import static ru.yandex.direct.core.entity.bidmodifiers.repository.mapper.Common.AB_SEGMENT_ADJUSTMENT_MAPPER;
import static ru.yandex.direct.core.entity.bidmodifiers.repository.mapper.Common.AB_SEGMENT_IS_ACCESSIBLE_FIELD_MAPPER;
import static ru.yandex.direct.core.entity.bidmodifiers.repository.mapper.Common.AB_SEGMENT_MAPPER;
import static ru.yandex.direct.dbschema.ppc.tables.AbSegmentMultiplierValues.AB_SEGMENT_MULTIPLIER_VALUES;
import static ru.yandex.direct.dbschema.ppc.tables.HierarchicalMultipliers.HIERARCHICAL_MULTIPLIERS;
import static ru.yandex.direct.dbschema.ppc.tables.RetargetingConditions.RETARGETING_CONDITIONS;
import static ru.yandex.direct.dbschema.ppc.tables.RetargetingGoals.RETARGETING_GOALS;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.JsonUtils.fromJson;

@Component
@ParametersAreNonnullByDefault
public class BidModifierABSegmentTypeSupport extends AbstractBidModifierMultipleValuesTypeSupport<
        BidModifierABSegment, BidModifierABSegmentAdjustment, Long> {
    private final RetargetingConditionService retargetingConditionService;

    @Autowired
    protected BidModifierABSegmentTypeSupport(ShardHelper shardHelper,
                                              LogBidModifiersService logBidModifiersService,
                                              RetargetingConditionService retargetingConditionService) {
        super(shardHelper, logBidModifiersService);
        this.retargetingConditionService = retargetingConditionService;
    }

    @Override
    public BidModifierType getType() {
        return BidModifierType.AB_SEGMENT_MULTIPLIER;
    }

    @Override
    public Class<BidModifierABSegment> getBidModifierClass() {
        return BidModifierABSegment.class;
    }

    @Override
    public boolean areEqual(BidModifierABSegment a, BidModifierABSegment b) {
        return Objects.equals(a.getId(), b.getId()) &&
                Objects.equals(a.getCampaignId(), b.getCampaignId()) &&
                Objects.equals(a.getAdGroupId(), b.getAdGroupId()) &&
                Objects.equals(a.getEnabled(), b.getEnabled()) &&
                Objects.equals(new HashSet<>(a.getAbSegmentAdjustments()),
                        new HashSet<>(b.getAbSegmentAdjustments()));
    }

    @Override
    public BidModifierABSegment createEmptyBidModifierFromRecord(Record record) {
        return AB_SEGMENT_MAPPER.fromDb(record);
    }

    @Override
    public List<BidModifierABSegment> createEmptyBidModifiersFromRecords(Collection<Record> records) {
        return records.stream().map(this::createEmptyBidModifierFromRecord).collect(toList());
    }

    @Override
    public void setAdjustments(BidModifierABSegment modifier,
                               List<BidModifierABSegmentAdjustment> abSegmentsAdjustment) {
        modifier.setAbSegmentAdjustments(abSegmentsAdjustment);
    }

    @Override
    public Map<Long, List<BidModifierABSegmentAdjustment>> getAdjustmentsByIds(DSLContext dslContext,
                                                                               Collection<Long> ids) {
        Result<Record> records = dslContext.select(AB_SEGMENT_ADJUSTMENT_FIELDS)
                .select(
                        DSL.min(
                                DSL.ifnull(RETARGETING_GOALS.IS_ACCESSIBLE, 0L)
                        ).as(RETARGETING_GOALS.IS_ACCESSIBLE)
                )
                .from(AB_SEGMENT_MULTIPLIER_VALUES)
                .leftJoin(RETARGETING_GOALS)
                .on(RETARGETING_GOALS.RET_COND_ID
                        .eq(AB_SEGMENT_MULTIPLIER_VALUES.AB_SEGMENT_RET_COND_ID))
                .where(AB_SEGMENT_MULTIPLIER_VALUES.AB_SEGMENT_MULTIPLIER_VALUE_ID.in(ids))
                .groupBy(AB_SEGMENT_ADJUSTMENT_FIELDS)
                .fetch();
        Multimap<Long, Record> recordsByParentId = Multimaps.index(records,
                record -> record.get(AB_SEGMENT_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID, Long.class));

        return EntryStream.of(recordsByParentId.asMap())
                .mapValues(list -> list.stream().map(record -> {
                    BidModifierABSegmentAdjustment adjustment = AB_SEGMENT_ADJUSTMENT_MAPPER.fromDb(record);
                    AB_SEGMENT_IS_ACCESSIBLE_FIELD_MAPPER.fromDb(adjustment, record);

                    return adjustment;
                }).collect(toList()))
                .toMap();
    }

    @Override
    public void fillAdjustments(DSLContext dslContext, Collection<BidModifierABSegment> bidModifiers,
                                boolean updLock) {
        SelectFinalStep<Record> selectStep = dslContext
                .select(AB_SEGMENT_ADJUSTMENT_FIELDS)
                .select(
                        DSL.min(
                                DSL.ifnull(RETARGETING_GOALS.IS_ACCESSIBLE, 0L)
                        ).as(RETARGETING_GOALS.IS_ACCESSIBLE)
                )
                .select(RETARGETING_CONDITIONS.CONDITION_JSON)
                .from(AB_SEGMENT_MULTIPLIER_VALUES)
                .leftJoin(RETARGETING_GOALS)
                .on(RETARGETING_GOALS.RET_COND_ID
                        .eq(AB_SEGMENT_MULTIPLIER_VALUES.AB_SEGMENT_RET_COND_ID))
                .leftJoin(RETARGETING_CONDITIONS)
                .on(RETARGETING_CONDITIONS.RET_COND_ID
                        .eq(AB_SEGMENT_MULTIPLIER_VALUES.AB_SEGMENT_RET_COND_ID))
                .where(AB_SEGMENT_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID.in(
                        bidModifiers.stream().map(BidModifier::getId).collect(toSet())),
                        RETARGETING_CONDITIONS.RETARGETING_CONDITIONS_TYPE
                                .eq(RetargetingConditionsRetargetingConditionsType.ab_segments))
                .groupBy(AB_SEGMENT_ADJUSTMENT_FIELDS);
        if (updLock) {
            selectStep = ((SelectForUpdateStep) selectStep).forUpdate();
        }
        Result<Record> records = selectStep.fetch();
        Multimap<Long, Record> recordsByParentId = Multimaps.index(records,
                record -> record.get(AB_SEGMENT_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID, Long.class));
        Map<Long, BidModifierABSegment> bidModifiersById = Maps.uniqueIndex(bidModifiers, BidModifier::getId);
        EntryStream.of(recordsByParentId.asMap())
                .forKeyValue((bidModifierId, recordsList) -> {
                    List<BidModifierABSegmentAdjustment> adjustments =
                            recordsList.stream().map(record -> {
                                BidModifierABSegmentAdjustment adjustment =
                                        AB_SEGMENT_ADJUSTMENT_MAPPER.fromDb(record);
                                AB_SEGMENT_IS_ACCESSIBLE_FIELD_MAPPER.fromDb(adjustment, record);
                                Rule[] conditions =
                                        fromJson(record.get(RETARGETING_CONDITIONS.CONDITION_JSON), Rule[].class);
                                if (conditions.length > 0) {
                                    adjustment.setSectionId(conditions[0].getSectionId());
                                    // Целей в условии теоретически может быть несколько, но валидное состояние для AB-сегментов - одна
                                    adjustment.setSegmentId(conditions[0].getGoals().get(0).getId());

                                }
                                return adjustment;
                            }).collect(toList());
                    bidModifiersById.get(bidModifierId).withAbSegmentAdjustments(adjustments);
                });

    }

    @Override
    public Class<BidModifierABSegmentAdjustment> getAdjustmentClass() {
        return BidModifierABSegmentAdjustment.class;
    }

    @Override
    public List<BidModifierABSegmentAdjustment> getAdjustments(BidModifierABSegment modifier) {
        return modifier.getAbSegmentAdjustments();
    }

    @Override
    protected List<Long> getAddedIds(List<BidModifierABSegmentAdjustment> added,
                                     List<BidModifierABSegmentAdjustment> inserted) {
        Map<Long, Long> idsByRetCondId = inserted.stream()
                .collect(toMap(BidModifierABSegmentAdjustment::getAbSegmentRetargetingConditionId,
                        BidModifierABSegmentAdjustment::getId));
        return added.stream()
                .map(adjustment -> idsByRetCondId.get(adjustment.getAbSegmentRetargetingConditionId()))
                .collect(toList());
    }

    @Override
    protected void deleteAdjustments(Collection<Long> multiplierIds, DSLContext txContext) {
        txContext.deleteFrom(AB_SEGMENT_MULTIPLIER_VALUES)
                .where(AB_SEGMENT_MULTIPLIER_VALUES.AB_SEGMENT_MULTIPLIER_VALUE_ID
                        .in(multiplierIds))
                .execute();
    }

    @Override
    protected Long getKey(BidModifierABSegmentAdjustment adjustment) {
        return adjustment.getAbSegmentRetargetingConditionId();
    }

    @Override
    protected void insertAdjustments(Multimap<Long, BidModifierABSegmentAdjustment> adjustments,
                                     DSLContext txContext) {
        InsertHelper<AbSegmentMultiplierValuesRecord>
                insertHelper = new InsertHelper<>(txContext, AB_SEGMENT_MULTIPLIER_VALUES);
        adjustments.forEach((modifierId, adjustment) -> {
            insertHelper.add(AB_SEGMENT_ADJUSTMENT_MAPPER, adjustment);
            insertHelper.set(AB_SEGMENT_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID, modifierId);
            insertHelper.newRecord();
        });
        insertHelper.executeIfRecordsAdded();
    }

    @Override
    protected void updateAdjustments(Collection<AppliedChanges<BidModifierABSegmentAdjustment>> changes,
                                     DSLContext txContext) {
        JooqUpdateBuilder<AbSegmentMultiplierValuesRecord, BidModifierABSegmentAdjustment> updateBuilder =
                new JooqUpdateBuilder<>(AB_SEGMENT_MULTIPLIER_VALUES.AB_SEGMENT_MULTIPLIER_VALUE_ID, changes);

        updateBuilder.processProperty(PERCENT, AB_SEGMENT_MULTIPLIER_VALUES.MULTIPLIER_PCT, Integer::longValue);
        updateBuilder
                .processProperty(LAST_CHANGE, AB_SEGMENT_MULTIPLIER_VALUES.LAST_CHANGE, it -> LocalDateTime.now());

        txContext.update(AB_SEGMENT_MULTIPLIER_VALUES)
                .set(updateBuilder.getValues())
                .where(AB_SEGMENT_MULTIPLIER_VALUES.AB_SEGMENT_MULTIPLIER_VALUE_ID.in(
                        changes.stream().map(it -> it.getModel().getId()).collect(toSet())))
                .execute();
    }

    @Override
    protected LogMultiplierInfo createLogItem(BidModifierABSegment modifier,
                                              BidModifierABSegmentAdjustment adjustment, @Nullable Integer oldPercent) {
        return new LogABSegmentMultiplierInfo(adjustment.getId(), modifier.getId(),
                HierarchicalMultipliersType.ab_segment_multiplier,
                oldPercent, adjustment.getPercent(), adjustment.getAbSegmentRetargetingConditionId());
    }

    @Override
    protected Set<Long> getEmptyHierarchicalMultipliersForUpdate(Collection<BidModifierABSegment> bidModifiers,
                                                                 DSLContext dslContext) {
        TableField<? extends UpdatableRecord, Long> field = AB_SEGMENT_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID;

        return dslContext
                .select(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID)
                .from(HIERARCHICAL_MULTIPLIERS)
                .leftJoin(AB_SEGMENT_MULTIPLIER_VALUES)
                .on(field.eq(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID))
                .where(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID.in(
                        bidModifiers.stream().map(BidModifier::getId).collect(toSet())))
                .and(HIERARCHICAL_MULTIPLIERS.TYPE.eq(BidModifierType.toSource(getType())))
                .and(field.isNull())
                .forUpdate()
                .fetchSet(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID);
    }

    @Override
    public void updatePercents(ClientId clientId, long operatorUid,
                               List<AppliedChanges<BidModifierABSegmentAdjustment>> changes,
                               List<BidModifierABSegment> bidModifiers, DSLContext dslContext) {
        JooqUpdateBuilder<AbSegmentMultiplierValuesRecord, BidModifierABSegmentAdjustment> updateBuilder =
                new JooqUpdateBuilder<>(AB_SEGMENT_MULTIPLIER_VALUES.AB_SEGMENT_MULTIPLIER_VALUE_ID, changes);

        updateBuilder.processProperty(BidModifierABSegmentAdjustment.PERCENT,
                AB_SEGMENT_MULTIPLIER_VALUES.MULTIPLIER_PCT, Integer::longValue);

        dslContext.update(AB_SEGMENT_MULTIPLIER_VALUES)
                .set(updateBuilder.getValues())
                .set(AB_SEGMENT_MULTIPLIER_VALUES.LAST_CHANGE, LocalDateTime.now())
                .where(AB_SEGMENT_MULTIPLIER_VALUES.AB_SEGMENT_MULTIPLIER_VALUE_ID.in(
                        changes.stream()
                                .map(it -> it.getModel().getId())
                                .collect(toSet())
                ))
                .execute();

        // Логируем изменения
        logUpdateChanges(operatorUid, changes, bidModifiers);
    }

    @Override
    public Map<BidModifierKey, AddedBidModifierInfo> save(DSLContext txContext, List<BidModifierABSegment> modifiers,
                                                          Map<BidModifierKey, BidModifierABSegment> lockedExistingModifiers, ClientId clientId, long operatorUid,
                                                          AbstractBidModifierMultipleValuesTypeSupport.AdjustmentsMergeFunction<BidModifierABSegment, BidModifierABSegmentAdjustment> mergeFunction,
                                                          Set<CampaignIdAndAdGroupIdPair> changedCampaignsAndAdGroups) {

        BidModifierABSegment bidModifierABSegment = null;
        List<BidModifierABSegmentAdjustment> newABSegmentAdjustments = new ArrayList<>();
        // вообще коорректировки на АБ-сегменты.
        for (BidModifierABSegment modifier : modifiers) {

            bidModifierABSegment = lockedExistingModifiers
                    .get(new BidModifierKey(modifier.getCampaignId(), null, BidModifierType.AB_SEGMENT_MULTIPLIER));

            // если у таблице hierarchical_multipliers есть записи для кампании с type == ab_segment_multiplier
            if (bidModifierABSegment != null) {
                Map<Long, BidModifierABSegmentAdjustment> segmentIdToValueHash;
                segmentIdToValueHash = listToMap(bidModifierABSegment.getAbSegmentAdjustments(),
                        BidModifierABSegmentAdjustment::getSegmentId);
                for (BidModifierABSegmentAdjustment adjustment : modifier.getAbSegmentAdjustments()) {
                    if (adjustment.getAbSegmentRetargetingConditionId() != null) {
                        // Если abSegmentRetargetingConditionId заполнен, нет смысла искать чем бы его заполнить.
                        continue;
                    }
                    BidModifierABSegmentAdjustment existedSegmentAdjustment =
                            segmentIdToValueHash.get(adjustment.getSegmentId());
                    if (existedSegmentAdjustment != null && existedSegmentAdjustment.getSectionId()
                            .equals(adjustment.getSectionId())) {
                        adjustment.withAbSegmentRetargetingConditionId(
                                existedSegmentAdjustment.getAbSegmentRetargetingConditionId())
                                .withAccessible(existedSegmentAdjustment.getAccessible())
                                .withId(existedSegmentAdjustment.getId());
                    } else {
                        newABSegmentAdjustments.add(adjustment);
                    }
                }
            } else {
                // Корректировок АБ сегментов вообще нет
                newABSegmentAdjustments.addAll(modifier.getAbSegmentAdjustments());
            }
        }
        if (!newABSegmentAdjustments.isEmpty()) {
            //Не было корректировок для целей в эксперименте
            fillABSegmentRetargetingCondition(clientId, newABSegmentAdjustments);
        }

        return super.save(txContext, modifiers, lockedExistingModifiers, clientId, operatorUid,
                mergeFunction, changedCampaignsAndAdGroups);
    }

    /*
        Создает ретаргетинги для АБ-сегментов и проставляет их Id в корректировки
        (BidModifierABSegmentAdjustment.abSegmentRetargetingConditionId)
     */
    private void fillABSegmentRetargetingCondition(ClientId clientId,
                                                   List<BidModifierABSegmentAdjustment> bidModifierABSegmentAdjustments) {
        Map<Long, Goal> metrikaGoalsHash =
                listToMap(retargetingConditionService.getAvailableMetrikaGoalsForRetargeting(clientId,
                        GoalType.AB_SEGMENT),
                        Goal::getId);

        List<RetargetingCondition> retargetingConditions = mapList(bidModifierABSegmentAdjustments,
                v -> (RetargetingCondition) new RetargetingCondition()
                        .withType(ConditionType.ab_segments)
                        .withName(RetargetingCondition.DEFAULT_NAME_FOR_TYPES_WITHOUT_NAME)
                        .withClientId(clientId.asLong())
                        .withRules(singletonList(new Rule()
                                .withSectionId(v.getSectionId())
                                .withType(RuleType.OR)
                                .withGoals(singletonList(metrikaGoalsHash.get(v.getSegmentId()))))));

        MassResult<Long> retCondAddedResult =
                retargetingConditionService.addRetargetingConditions(retargetingConditions, clientId);
        for (int i = 0; i < bidModifierABSegmentAdjustments.size(); i++) {
            if (retCondAddedResult.get(i).isSuccessful()) {
                bidModifierABSegmentAdjustments.get(i)
                        .setAbSegmentRetargetingConditionId(retCondAddedResult.get(i).getResult());
            }
        }
    }

    @Override
    public void prepareSystemFields(List<BidModifierABSegment> bidModifiers) {
        super.prepareSystemFields(bidModifiers);
        LocalDateTime now = LocalDateTime.now();
        bidModifiers.forEach(bidModifier -> bidModifier.getAbSegmentAdjustments()
                .forEach(abSegmentAdjustment -> abSegmentAdjustment.withLastChange(now)));
    }
}
