package ru.yandex.direct.core.entity.bidmodifiers.repository.typesupport.multivalue;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Maps;
import com.google.common.collect.Multimap;
import org.jooq.DSLContext;
import org.jooq.Record;
import org.jooq.SelectConditionStep;
import org.jooq.SelectOptionStep;
import org.jooq.TableField;
import org.jooq.UpdatableRecord;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.common.log.container.bidmodifiers.LogBannerTypeMultiplierInfo;
import ru.yandex.direct.common.log.container.bidmodifiers.LogMultiplierInfo;
import ru.yandex.direct.common.log.service.LogBidModifiersService;
import ru.yandex.direct.core.entity.bidmodifier.BannerType;
import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierBannerType;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierBannerTypeAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierType;
import ru.yandex.direct.dbschema.ppc.enums.HierarchicalMultipliersType;
import ru.yandex.direct.dbschema.ppc.tables.BannerTypeMultiplierValues;
import ru.yandex.direct.dbschema.ppc.tables.records.BannerTypeMultiplierValuesRecord;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder;
import ru.yandex.direct.model.AppliedChanges;

import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.core.entity.bidmodifier.BidModifierAdjustment.PERCENT;
import static ru.yandex.direct.core.entity.bidmodifier.BidModifierBannerTypeAdjustment.LAST_CHANGE;
import static ru.yandex.direct.core.entity.bidmodifiers.repository.mapper.Common.BANNER_TYPE_ADJUSTMENT_FIELDS;
import static ru.yandex.direct.core.entity.bidmodifiers.repository.mapper.Common.BANNER_TYPE_ADJUSTMENT_MAPPER;
import static ru.yandex.direct.core.entity.bidmodifiers.repository.mapper.Common.BANNER_TYPE_MAPPER;
import static ru.yandex.direct.dbschema.ppc.tables.HierarchicalMultipliers.HIERARCHICAL_MULTIPLIERS;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
@ParametersAreNonnullByDefault
public class BidModifierBannerTypeTypeSupport extends AbstractBidModifierMultipleValuesTypeSupport<
        BidModifierBannerType, BidModifierBannerTypeAdjustment, BannerType> {

    private static final BannerTypeMultiplierValues TABLE = BannerTypeMultiplierValues.BANNER_TYPE_MULTIPLIER_VALUES;

    @Autowired
    public BidModifierBannerTypeTypeSupport(ShardHelper shardHelper, LogBidModifiersService logBidModifiersService) {
        super(shardHelper, logBidModifiersService);
    }

    @Override
    public BidModifierType getType() {
        return BidModifierType.BANNER_TYPE_MULTIPLIER;
    }

    @Override
    public Class<BidModifierBannerType> getBidModifierClass() {
        return BidModifierBannerType.class;
    }

    @Override
    public boolean areEqual(BidModifierBannerType a, BidModifierBannerType b) {
        return Objects.equals(a.getId(), b.getId()) &&
                Objects.equals(a.getCampaignId(), b.getCampaignId()) &&
                Objects.equals(a.getAdGroupId(), b.getAdGroupId()) &&
                Objects.equals(a.getEnabled(), b.getEnabled()) &&
                Objects.equals(new HashSet<>(a.getBannerTypeAdjustments()),
                        new HashSet<>(b.getBannerTypeAdjustments()));
    }

    @Override
    public BidModifierBannerType createEmptyBidModifierFromRecord(Record record) {
        return BANNER_TYPE_MAPPER.fromDb(record);
    }

    @Override
    public List<BidModifierBannerType> createEmptyBidModifiersFromRecords(Collection<Record> records) {
        return mapList(records, this::createEmptyBidModifierFromRecord);
    }

    @Override
    public Map<Long, List<BidModifierBannerTypeAdjustment>> getAdjustmentsByIds(DSLContext dslContext,
                                                                                Collection<Long> ids) {
        return dslContext.select(BANNER_TYPE_ADJUSTMENT_FIELDS)
                .from(TABLE)
                .where(TABLE.BANNER_TYPE_MULTIPLIER_VALUE_ID.in(ids))
                .fetchGroups(r -> r.getValue(TABLE.HIERARCHICAL_MULTIPLIER_ID), BANNER_TYPE_ADJUSTMENT_MAPPER::fromDb);
    }

    @Override
    public void setAdjustments(BidModifierBannerType modifier, List<BidModifierBannerTypeAdjustment> adjustments) {
        modifier.setBannerTypeAdjustments(adjustments);
    }

    @Override
    public void fillAdjustments(DSLContext dslContext, Collection<BidModifierBannerType> bidModifiers,
                                boolean updLock) {
        SelectConditionStep<Record> selectStep = dslContext
                .select(BANNER_TYPE_ADJUSTMENT_FIELDS)
                .from(TABLE)
                .where(TABLE.HIERARCHICAL_MULTIPLIER_ID.in(listToSet(bidModifiers, BidModifier::getId)));
        SelectOptionStep<Record> optionStep = selectStep;
        if (updLock) {
            optionStep = selectStep.forUpdate();
        }

        Map<Long, List<BidModifierBannerTypeAdjustment>> adjustemntBidModifiers = optionStep.fetchGroups(
                r -> r.getValue(TABLE.HIERARCHICAL_MULTIPLIER_ID),
                BANNER_TYPE_ADJUSTMENT_MAPPER::fromDb
        );

        Map<Long, BidModifierBannerType> bidModifiersById = Maps.uniqueIndex(bidModifiers, BidModifier::getId);

        adjustemntBidModifiers.forEach(
                (id, adjustments) -> bidModifiersById.get(id).withBannerTypeAdjustments(adjustments)
        );
    }

    @Override
    protected BannerType getKey(BidModifierBannerTypeAdjustment adjustment) {
        return adjustment.getBannerType();
    }

    @Override
    protected void insertAdjustments(Multimap<Long, BidModifierBannerTypeAdjustment> adjustments, DSLContext txContext) {
        InsertHelper<BannerTypeMultiplierValuesRecord> insertHelper = new InsertHelper<>(txContext, TABLE);
        adjustments.forEach((modifierId, adjustment) -> {
            insertHelper.add(BANNER_TYPE_ADJUSTMENT_MAPPER, adjustment);
            insertHelper.set(TABLE.HIERARCHICAL_MULTIPLIER_ID, modifierId);
            insertHelper.newRecord();
        });
        insertHelper.executeIfRecordsAdded();
    }

    @Override
    protected void updateAdjustments(Collection<AppliedChanges<BidModifierBannerTypeAdjustment>> changes,
                                     DSLContext txContext) {
        JooqUpdateBuilder<BannerTypeMultiplierValuesRecord, BidModifierBannerTypeAdjustment> updateBuilder =
                new JooqUpdateBuilder<>(TABLE.BANNER_TYPE_MULTIPLIER_VALUE_ID, changes);

        updateBuilder.processProperty(PERCENT, TABLE.MULTIPLIER_PCT, Integer::longValue);
        updateBuilder
                .processProperty(LAST_CHANGE, TABLE.LAST_CHANGE, it -> LocalDateTime.now());

        txContext.update(TABLE)
                .set(updateBuilder.getValues())
                .where(TABLE.BANNER_TYPE_MULTIPLIER_VALUE_ID.in(
                        changes.stream().map(it -> it.getModel().getId()).collect(toSet())))
                .execute();
    }

    @Override
    public Class<BidModifierBannerTypeAdjustment> getAdjustmentClass() {
        return BidModifierBannerTypeAdjustment.class;
    }

    @Override
    public List<BidModifierBannerTypeAdjustment> getAdjustments(BidModifierBannerType modifier) {
        return modifier.getBannerTypeAdjustments();
    }

    @Override
    protected void deleteAdjustments(Collection<Long> multiplierIds, DSLContext txContext) {
        txContext.deleteFrom(TABLE)
                .where(TABLE.BANNER_TYPE_MULTIPLIER_VALUE_ID.in(multiplierIds))
                .execute();
    }

    @Override
    protected List<Long> getAddedIds(List<BidModifierBannerTypeAdjustment> addedAdjustments,
                                     List<BidModifierBannerTypeAdjustment> insertedMultipliers) {
        Map<BannerType, Long> idsByKey =
                listToMap(insertedMultipliers, this::getKey, BidModifierBannerTypeAdjustment::getId);
        return addedAdjustments.stream().map(adjustment ->
                idsByKey.get(adjustment.getBannerType()))
                .collect(toList());
    }

    @Override
    protected LogMultiplierInfo createLogItem(BidModifierBannerType modifier,
                                              BidModifierBannerTypeAdjustment adjustment, @Nullable Integer oldPercent) {
        return new LogBannerTypeMultiplierInfo(adjustment.getId(), modifier.getId(),
                HierarchicalMultipliersType.banner_type_multiplier,
                oldPercent, adjustment.getPercent(), BannerType.toSource(adjustment.getBannerType()));
    }

    protected Set<Long> getEmptyHierarchicalMultipliersForUpdate(Collection<BidModifierBannerType> bidModifiers,
                                                                 DSLContext dslContext) {
        TableField<? extends UpdatableRecord, Long> field = TABLE.HIERARCHICAL_MULTIPLIER_ID;
        return dslContext
                .select(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID)
                .from(HIERARCHICAL_MULTIPLIERS)
                .leftJoin(TABLE)
                .on(field.eq(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID))
                .where(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID.in(
                        bidModifiers.stream().map(BidModifier::getId).collect(toSet())))
                .and(HIERARCHICAL_MULTIPLIERS.TYPE.eq(BidModifierType.toSource(getType())))
                .and(field.isNull())
                .forUpdate()
                .fetchSet(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID);
    }

    @Override
    public void updatePercents(ClientId clientId, long operatorUid,
                               List<AppliedChanges<BidModifierBannerTypeAdjustment>> changes,
                               List<BidModifierBannerType> bidModifiers, DSLContext dslContext) {
        JooqUpdateBuilder<BannerTypeMultiplierValuesRecord, BidModifierBannerTypeAdjustment> updateBuilder =
                new JooqUpdateBuilder<>(TABLE.BANNER_TYPE_MULTIPLIER_VALUE_ID, changes);

        updateBuilder
                .processProperty(BidModifierBannerTypeAdjustment.PERCENT, TABLE.MULTIPLIER_PCT, Integer::longValue);

        dslContext.update(TABLE)
                .set(updateBuilder.getValues())
                .set(TABLE.LAST_CHANGE, LocalDateTime.now())
                .where(TABLE.BANNER_TYPE_MULTIPLIER_VALUE_ID.in(
                        changes.stream()
                                .map(it -> it.getModel().getId())
                                .collect(toSet())
                ))
                .execute();

        // Логируем изменения
        logUpdateChanges(operatorUid, changes, bidModifiers);
    }

    @Override
    public void prepareSystemFields(List<BidModifierBannerType> bidModifiers) {
        super.prepareSystemFields(bidModifiers);
        LocalDateTime now = LocalDateTime.now();
        bidModifiers.forEach(bidModifier -> bidModifier.getBannerTypeAdjustments()
                .forEach(bannerTypeAdjustment -> bannerTypeAdjustment.withLastChange(now)));
    }
}
