package ru.yandex.direct.core.entity.bidmodifiers.repository.typesupport.multivalue;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Maps;
import com.google.common.collect.Multimap;
import com.google.common.collect.Multimaps;
import one.util.streamex.EntryStream;
import org.jooq.DSLContext;
import org.jooq.Record;
import org.jooq.Result;
import org.jooq.SelectFinalStep;
import org.jooq.SelectForUpdateStep;
import org.jooq.Table;
import org.jooq.TableField;
import org.jooq.UpdatableRecord;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.common.log.container.bidmodifiers.LogDemographyMultiplierInfo;
import ru.yandex.direct.common.log.container.bidmodifiers.LogMultiplierInfo;
import ru.yandex.direct.common.log.service.LogBidModifiersService;
import ru.yandex.direct.core.entity.bidmodifier.AgeType;
import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDemographics;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDemographicsAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierType;
import ru.yandex.direct.core.entity.bidmodifier.GenderType;
import ru.yandex.direct.dbschema.ppc.enums.HierarchicalMultipliersType;
import ru.yandex.direct.dbschema.ppc.tables.records.DemographyMultiplierValuesRecord;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder;
import ru.yandex.direct.model.AppliedChanges;

import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.core.entity.bidmodifier.BidModifierAdjustment.PERCENT;
import static ru.yandex.direct.core.entity.bidmodifier.BidModifierDemographicsAdjustment.LAST_CHANGE;
import static ru.yandex.direct.core.entity.bidmodifiers.repository.mapper.Common.DEMOGRAPHICS_MAPPER;
import static ru.yandex.direct.core.entity.bidmodifiers.repository.mapper.Common.DEMOGRAPHY_ADJUSTMENT_FIELDS;
import static ru.yandex.direct.core.entity.bidmodifiers.repository.mapper.Common.DEMOGRAPHY_ADJUSTMENT_MAPPER;
import static ru.yandex.direct.dbschema.ppc.tables.DemographyMultiplierValues.DEMOGRAPHY_MULTIPLIER_VALUES;
import static ru.yandex.direct.dbschema.ppc.tables.HierarchicalMultipliers.HIERARCHICAL_MULTIPLIERS;

@Component
@ParametersAreNonnullByDefault
public class BidModifierDemographicsTypeSupport extends AbstractBidModifierMultipleValuesTypeSupport<
        BidModifierDemographics, BidModifierDemographicsAdjustment, BidModifierDemographicsTypeSupport.DemographyKey> {
    @Autowired
    public BidModifierDemographicsTypeSupport(ShardHelper shardHelper,
                                              LogBidModifiersService logBidModifiersService) {
        super(shardHelper, logBidModifiersService);
    }

    @Override
    public BidModifierType getType() {
        return BidModifierType.DEMOGRAPHY_MULTIPLIER;
    }

    @Override
    public Class<BidModifierDemographics> getBidModifierClass() {
        return BidModifierDemographics.class;
    }

    @Override
    public boolean areEqual(BidModifierDemographics a, BidModifierDemographics b) {
        return Objects.equals(a.getId(), b.getId()) &&
                Objects.equals(a.getCampaignId(), b.getCampaignId()) &&
                Objects.equals(a.getAdGroupId(), b.getAdGroupId()) &&
                Objects.equals(a.getEnabled(), b.getEnabled()) &&
                Objects.equals(new HashSet<>(a.getDemographicsAdjustments()),
                        new HashSet<>(b.getDemographicsAdjustments()));
    }

    @Override
    public BidModifierDemographics createEmptyBidModifierFromRecord(Record record) {
        return DEMOGRAPHICS_MAPPER.fromDb(record);
    }

    @Override
    public List<BidModifierDemographics> createEmptyBidModifiersFromRecords(Collection<Record> records) {
        return records.stream().map(this::createEmptyBidModifierFromRecord).collect(toList());
    }

    @Override
    public Map<Long, List<BidModifierDemographicsAdjustment>> getAdjustmentsByIds(DSLContext dslContext,
                                                                                  Collection<Long> ids) {
        Result<Record> records = dslContext.select(DEMOGRAPHY_ADJUSTMENT_FIELDS)
                .from(DEMOGRAPHY_MULTIPLIER_VALUES)
                .where(DEMOGRAPHY_MULTIPLIER_VALUES.DEMOGRAPHY_MULTIPLIER_VALUE_ID.in(ids)).fetch();
        Multimap<Long, Record> recordsByParentId = Multimaps.index(records,
                record -> record.get(DEMOGRAPHY_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID, Long.class));
        return EntryStream.of(recordsByParentId.asMap())
                .mapValues(list -> list.stream().map(DEMOGRAPHY_ADJUSTMENT_MAPPER::fromDb).collect(toList()))
                .toMap();
    }

    @Override
    public void setAdjustments(BidModifierDemographics modifier, List<BidModifierDemographicsAdjustment> adjustments) {
        modifier.setDemographicsAdjustments(adjustments);
    }

    @Override
    public void fillAdjustments(DSLContext dslContext, Collection<BidModifierDemographics> bidModifiers,
                                boolean updLock) {
        SelectFinalStep<Record> selectStep = dslContext
                .select(DEMOGRAPHY_ADJUSTMENT_FIELDS)
                .from(DEMOGRAPHY_MULTIPLIER_VALUES)
                .where(DEMOGRAPHY_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID.in(
                        bidModifiers.stream().map(BidModifier::getId).collect(toSet())
                ));
        if (updLock) {
            selectStep = ((SelectForUpdateStep) selectStep).forUpdate();
        }
        Result<Record> records = selectStep.fetch();
        Multimap<Long, Record> recordsByParentId = Multimaps.index(records,
                record -> record.get(DEMOGRAPHY_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID, Long.class));
        Map<Long, BidModifierDemographics> bidModifiersById = Maps.uniqueIndex(bidModifiers, BidModifier::getId);
        EntryStream.of(recordsByParentId.asMap())
                .forKeyValue((bidModifierId, recordsList) -> {
                    List<BidModifierDemographicsAdjustment> adjustments =
                            recordsList.stream().map(DEMOGRAPHY_ADJUSTMENT_MAPPER::fromDb).collect(toList());
                    bidModifiersById.get(bidModifierId).withDemographicsAdjustments(adjustments);
                });
    }

    @Override
    protected DemographyKey getKey(BidModifierDemographicsAdjustment adjustment) {
        return new DemographyKey(adjustment.getGender(), adjustment.getAge());
    }

    @Override
    protected void insertAdjustments(Multimap<Long, BidModifierDemographicsAdjustment> adjustments,
                                     DSLContext txContext) {
        InsertHelper<DemographyMultiplierValuesRecord>
                insertHelper = new InsertHelper<>(txContext, DEMOGRAPHY_MULTIPLIER_VALUES);
        adjustments.forEach((modifierId, adjustment) -> {
            insertHelper.add(DEMOGRAPHY_ADJUSTMENT_MAPPER, adjustment);
            insertHelper.set(DEMOGRAPHY_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID, modifierId);
            insertHelper.newRecord();
        });
        insertHelper.executeIfRecordsAdded();
    }

    @Override
    protected void updateAdjustments(Collection<AppliedChanges<BidModifierDemographicsAdjustment>> changes,
                                     DSLContext txContext) {
        JooqUpdateBuilder<DemographyMultiplierValuesRecord, BidModifierDemographicsAdjustment> updateBuilder =
                new JooqUpdateBuilder<>(DEMOGRAPHY_MULTIPLIER_VALUES.DEMOGRAPHY_MULTIPLIER_VALUE_ID, changes);

        updateBuilder.processProperty(PERCENT, DEMOGRAPHY_MULTIPLIER_VALUES.MULTIPLIER_PCT, Integer::longValue);
        updateBuilder.processProperty(LAST_CHANGE, DEMOGRAPHY_MULTIPLIER_VALUES.LAST_CHANGE, it -> LocalDateTime.now());

        txContext.update(DEMOGRAPHY_MULTIPLIER_VALUES)
                .set(updateBuilder.getValues())
                .where(DEMOGRAPHY_MULTIPLIER_VALUES.DEMOGRAPHY_MULTIPLIER_VALUE_ID.in(
                        changes.stream().map(it -> it.getModel().getId()).collect(toSet())))
                .execute();
    }

    @Override
    public Class<BidModifierDemographicsAdjustment> getAdjustmentClass() {
        return BidModifierDemographicsAdjustment.class;
    }

    @Override
    public List<BidModifierDemographicsAdjustment> getAdjustments(BidModifierDemographics modifier) {
        return modifier.getDemographicsAdjustments();
    }

    @Override
    protected void deleteAdjustments(Collection<Long> multiplierIds, DSLContext txContext) {
        txContext.deleteFrom(DEMOGRAPHY_MULTIPLIER_VALUES)
                .where(DEMOGRAPHY_MULTIPLIER_VALUES.DEMOGRAPHY_MULTIPLIER_VALUE_ID.in(multiplierIds))
                .execute();
    }

    @Override
    protected List<Long> getAddedIds(List<BidModifierDemographicsAdjustment> addedAdjustments,
                                     List<BidModifierDemographicsAdjustment> insertedMultipliers) {
        Map<DemographyKey, Long> idsByKey =
                insertedMultipliers.stream().collect(toMap(this::getKey, BidModifierDemographicsAdjustment::getId));
        return addedAdjustments.stream().map(adjustment ->
                idsByKey.get(new DemographyKey(adjustment.getGender(), adjustment.getAge())))
                .collect(toList());
    }

    @Override
    protected LogMultiplierInfo createLogItem(BidModifierDemographics modifier,
                                              BidModifierDemographicsAdjustment adjustment, @Nullable Integer oldPercent) {
        return new LogDemographyMultiplierInfo(adjustment.getId(), modifier.getId(),
                HierarchicalMultipliersType.demography_multiplier,
                oldPercent, adjustment.getPercent(), AgeType.toSource(adjustment.getAge()),
                GenderType.toSource(adjustment.getGender()));
    }

    static class DemographyKey {
        private final GenderType gender;
        private final AgeType age;

        public DemographyKey(GenderType gender, AgeType age) {
            this.gender = gender;
            this.age = age;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            DemographyKey that = (DemographyKey) o;
            return gender == that.gender &&
                    age == that.age;
        }

        @Override
        public int hashCode() {
            return Objects.hash(gender, age);
        }
    }

    protected Set<Long> getEmptyHierarchicalMultipliersForUpdate(Collection<BidModifierDemographics> bidModifiers,
                                                                 DSLContext dslContext) {
        Table table = DEMOGRAPHY_MULTIPLIER_VALUES;
        TableField<? extends UpdatableRecord, Long> field = DEMOGRAPHY_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID;
        return dslContext
                .select(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID)
                .from(HIERARCHICAL_MULTIPLIERS)
                .leftJoin(table)
                .on(field.eq(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID))
                .where(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID.in(
                        bidModifiers.stream().map(BidModifier::getId).collect(toSet())))
                .and(HIERARCHICAL_MULTIPLIERS.TYPE.eq(BidModifierType.toSource(getType())))
                .and(field.isNull())
                .forUpdate()
                .fetchSet(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID);
    }

    @Override
    public void updatePercents(ClientId clientId, long operatorUid,
                               List<AppliedChanges<BidModifierDemographicsAdjustment>> changes,
                               List<BidModifierDemographics> bidModifiers, DSLContext dslContext) {
        JooqUpdateBuilder<DemographyMultiplierValuesRecord, BidModifierDemographicsAdjustment> updateBuilder =
                new JooqUpdateBuilder<>(DEMOGRAPHY_MULTIPLIER_VALUES.DEMOGRAPHY_MULTIPLIER_VALUE_ID, changes);

        updateBuilder.processProperty(BidModifierDemographicsAdjustment.PERCENT,
                DEMOGRAPHY_MULTIPLIER_VALUES.MULTIPLIER_PCT, Integer::longValue);

        dslContext.update(DEMOGRAPHY_MULTIPLIER_VALUES)
                .set(updateBuilder.getValues())
                .set(DEMOGRAPHY_MULTIPLIER_VALUES.LAST_CHANGE, LocalDateTime.now())
                .where(DEMOGRAPHY_MULTIPLIER_VALUES.DEMOGRAPHY_MULTIPLIER_VALUE_ID.in(
                        changes.stream()
                                .map(it -> it.getModel().getId())
                                .collect(toSet())
                ))
                .execute();

        // Логируем изменения
        logUpdateChanges(operatorUid, changes, bidModifiers);
    }

    @Override
    public void prepareSystemFields(List<BidModifierDemographics> bidModifiers) {
        super.prepareSystemFields(bidModifiers);
        LocalDateTime now = LocalDateTime.now();
        bidModifiers.forEach(bidModifier -> bidModifier.getDemographicsAdjustments()
                .forEach(demographicsAdjustment -> demographicsAdjustment.withLastChange(now)));
    }
}
